// -*- C++ -*-
/*!
 * @file Manager.h
 * @brief RTComponent manager class
 * @date $Date: 2007-12-31 03:08:04 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef RTC_MANAGER_H
#define RTC_MANAGER_H

#include <rtm/RTC.h>

#include <iostream>
#include <string>
#include <vector>

#include <coil/Mutex.h>
#include <coil/Guard.h>
#include <coil/Task.h>

#include <rtm/Factory.h>
#include <rtm/ECFactory.h>
#include <rtm/ObjectManager.h>
#include <rtm/SystemLogger.h>

namespace RTM
{
  class ManagerServant;
}

namespace coil
{
  class Timer;
};

namespace RTC
{
//  class Properties;
  class CorbaNaming;
  class ModuleManager;
  class NamingManager;
  class Manager;
  class RTObject_impl;
  typedef RTObject_impl RtcBase;

  typedef void (*ModuleInitProc)(Manager* manager);  
  
  /*!
   * @if jp
   * @class Manager
   * @brief Manager 饹
   *
   * ݡͥȤʤɳƼξԤޥ͡㥯饹
   *
   * @since 0.2.0
   *
   * @else
   * @class Manager
   * @brief Manager class
   *
   * This is a manager class that manages various information
   * such as components.
   *
   * @since 0.2.0
   *
   * @endif
   */
  class Manager
  {
    typedef coil::Mutex Mutex;
    typedef coil::Guard<Mutex> Guard;
  protected:
    /*!
     * @if jp
     * @brief Protected 󥹥ȥ饯
     *
     * Protected 󥹥ȥ饯
     *
     * @else
     * @brief Protected Constructor
     *
     * Protected Constructor
     *
     * @endif
     */
    Manager();
    
    /*!
     * @if jp
     * @brief Protected ԡ󥹥ȥ饯
     *
     * Protected ԡ󥹥ȥ饯
     *
     * @param manager ԡޥ͡㥪֥
     *
     * @else
     * @brief Protected Copy Constructor
     *
     * Protected Copy Constructor
     *
     * @param manager Manager object of copy source
     *
     * @endif
     */
    Manager(const Manager& manager);
    //      Manager& operator=(const Manager& manager){return manager;};
    
  public:
    /*!
     * @if jp
     * @brief ޥ͡ν
     *
     * ޥ͡ static дؿ
     * ޥ͡򥳥ޥɥ饤Ϳƽ롣
     * ޥ͡Ѥϡɬνдؿ init() 
     * ƤФʤФʤʤ
     * ޥ͡Υ󥹥󥹤ˡȤơinit(), instance() 
     * 2Ĥ static дؿѰդƤ뤬init()Ǥ
     * ԤʤᡢManager ¸֤ΰֺǽˤinit()Ƥɬפ롣
     *
     * ޥ͡ν
     * - initManager: configեɤ߹ߡ֥ƥ
     * - initLogger: Logger
     * - initORB: ORB 
     * - initNaming: NamingService 
     * - initExecutionContext: ExecutionContext factory 
     * - initTimer: Timer 
     *
     * @param argc ޥɥ饤ο
     * @param argv ޥɥ饤
     * 
     * @return Manager ͣΥ󥹥󥹤λ
     *
     * @else
     * @brief Initialize manager
     *
     * This is the static member function to initialize the Manager.
     * The Manager is initialized by given commandline arguments.
     * To use the manager, this initialization member function init() must be
     * called. The manager has two static functions to get the instance such as
     * init() and instance(). Since initializing process is only performed by
     * the init() function, the init() has to be called at the beginning of
     * the lifecycle of the Manager.
     *
     * *Initialization of manager
     * - initManager: Argument processing, reading config file,
     *                initialization of subsystem
     * - initLogger: Initialization of Logger
     * - initORB: Initialization of ORB
     * - initNaming: Initialization of NamingService
     * - initExecutionContext: Initialization of ExecutionContext factory
     * - initTimer: Initialization of Timer
     *
     * @param argc The number of command line arguments. 
     * @param argv The array of the command line arguments.
     *
     * @return Reference of the unique instance of Manager
     *
     * @endif
     */
    static Manager* init(int argc, char** argv);
    
    /*!
     * @if jp
     * @brief ޥ͡Υ󥹥󥹤μ
     *
     * ޥ͡Υ󥹥󥹤 static дؿ
     * δؿƤˡɬνдؿ init() ƤФƤ
     * ɬפ롣
     *
     * @return Manager ͣΥ󥹥󥹤λ
     * 
     * @else
     *
     * @brief Get instance of the manager
     *
     * This is the static member function to get the instance of the Manager.
     * Before calling this function, ensure that the initialization function
     * "init()" is called.
     *
     * @return The only instance reference of the manager
     *
     * @endif
     */ 
    static Manager& instance();
    
    //============================================================
    // Manager
    //============================================================
    
    /*!
     * @if jp
     * @brief ޥ͡㽪λ
     *
     * ޥ͡νλ¹Ԥ롣
     *
     * @else
     * @brief Terminate manager
     *
     * Terminate manager's processing
     *
     * @endif
     */
    void terminate();
    
    /*!
     * @if jp
     * @brief ޥ͡㡦åȥ
     *
     * ޥ͡νλ¹Ԥ롣
     * ORBλ塢Ʊäƽλ롣
     *
     * @else
     * @brief Shutdown Manager
     *
     * Terminate manager's processing.
     * After terminating ORB, shutdown manager in sync.
     *
     * @endif
     */
    void shutdown();
    
    /*!
     * @if jp
     * @brief ޥ͡㽪λԤ碌
     *
     * Ʊ뤿ᡢޥ͡㽪λԤ碌Ԥ
     *
     * @else
     * @brief Wait for Manager's termination
     *
     * Wait for Manager's termination to synchronize.
     *
     * @endif
     */
    void join();
    
    /*!
     * @if jp
     * @brief Хåեμ
     *
     * ޥ͡ꤷХåե롣
     *
     * @return ޥ͡ꤷХåե
     *
     * @else
     * @brief Get the log buffer
     *
     * Get the log buffer that has been set to manager.
     *
     * @return Log buffer to set to manager
     *
     * @endif
     */
    LogStreamBuf& getLogStreamBuf() {return m_logStreamBuf;}

    /*!
     * @if jp
     * @brief ե졼Υ٥μ
     *
     * ե졼Υ٥롣
     *
     * @return ե졼Υ٥
     *
     * @else
     * @brief Get the log level of the configuration.
     *
     * Get the log level of the configuration.
     *
     * @return Log level of Manager's configuration
     *
     * @endif
     */
    std::string& getLogLevel() {return m_config["logger.log_level"];}
    
    /*!
     * @if jp
     * @brief ޥ͡㥳ե졼μ
     *
     * ޥ͡ꤷե졼롣
     *
     * @return ޥ͡Υե졼
     *
     * @else
     * @brief Get the manager configuration
     *
     * Get the manager configuration that has been set to manager.
     *
     * @return Manager's configuration
     *
     * @endif
     */
    coil::Properties& getConfig() { return m_config;}
    
    /*!
     * @if jp
     *
     * @brief ץΥå
     *
     * Υڥ졼ϥ桼Ԥ⥸塼νץ
     * ꤹ롣ꤵ줿ץϡޥ͡㤬졢
     * ƥֲ줿塢Ŭڤʥߥ󥰤Ǽ¹Ԥ롣
     *
     * @param proc ץδؿݥ
     *
     * @else
     *
     * @brief Set initial procedure
     *
     * This operation sets the initial procedure call to process module
     * initialization, other user defined initialization and so on.
     * The given procedure will be called at the proper timing after the 
     * manager initialization, activation and run.
     *
     * @param proc A function pointer to the initial procedure call
     *
     * @endif
     */
    void setModuleInitProc(ModuleInitProc proc);
    
    /*!
     * @if jp
     *
     * @brief ManagerΥƥֲ
     *
     * Υڥ졼ϰʲνԤ
     * - CORBA POAManager Υƥֲ
     * - ޥ͡CORBA֥ȤΥƥֲ
     * - Manager Υ֥ȻȤϿ
     *
     * Υڥ졼ϡޥ͡ν塢runManager()
     * ˸Ƥɬפ롣
     *
     *
     * @return (ƥֲ:true:false)
     *
     * @else
     *
     * @brief Activate the Manager
     *
     * This operation do the following:
     * - Activate CORBA POAManager
     * - Activate Manager CORBA object
     * - Bind object reference of the Manager to the nameserver
     *
     * This operation should be invoked after Manager:init(),
     * and before runManager().
     *
     * @return Activation result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool activateManager();
    
    /*!
     * @if jp
     *
     * @brief Managerμ¹
     *
     * Υڥ졼ϥޥ͡Υᥤ롼פ¹Ԥ롣
     * Υᥤ롼ǤϡCORBA ORBΥ٥ȥ롼
     * 롣ǥեȤǤϡΥڥ졼ϥ֥å
     * Manager::destroy() ƤФޤǽᤵʤ
     *  no_block  true ꤵƤϡǥ٥ȥ롼
     * 륹åɤư֥å˽᤹
     *
     * @param no_block false: ֥å󥰥⡼, true: Υ֥å󥰥⡼
     *
     * @else
     *
     * @brief Run the Manager
     *
     * This operation processes the main event loop of the Manager.
     * In this main loop, CORBA's ORB event loop or other processes
     * are performed. As the default behavior, this operation is going to
     * blocking mode and never returns until Manager::destroy() is called.
     * When the given argument "no_block" is set to "true", this operation
     * creates a thread to process the event loop internally, and it doesn't
     * block and returns.
     *
     * @param no_block false: Blocking mode, true: non-blocking mode.
     *
     * @endif
     */
    void runManager(bool no_block = false);
    
    //============================================================
    // Module management
    //============================================================
    /*!
     * @if jp
     * @brief [CORBA interface] ⥸塼Υ
     *
     * ꤷݡͥȤΥ⥸塼ɤȤȤˡ
     * ꤷؿ¹Ԥ롣
     *
     * @param fname   ⥸塼ե̾
     * @param initfunc ؿ̾
     * 
     * @else
     *
     * @brief [CORBA interface] Load module
     *
     * Load specified module (shared library, DLL etc..),
     * and invoke initialize function.
     *
     * @param fname    The module file name
     * @param initfunc The initialize function name
     *
     * @endif
     */  
    void load(const char* fname, const char* initfunc);
    
    /*!
     * @if jp
     *
     * @brief ⥸塼Υ
     *
     * ⥸塼򥢥ɤ
     *
     * @param fname ⥸塼Υե̾
     * 
     * @else
     *
     * @brief Unload module
     *
     * Unload module.
     *
     * @param fname The module file name
     *
     * @endif
     */ 
    void unload(const char* fname);
    
    /*!
     * @if jp
     *
     * @brief ⥸塼Υ
     *
     * ⥸塼򤹤٤ƥɤ
     *
     * @else
     *
     * @brief Unload all modules
     *
     * Unload all modules.
     *
     * @endif
     */ 
    void unloadAll();
    
    /*!
     * @if jp
     * @brief ɺѤߤΥ⥸塼ꥹȤ
     *
     * ߥޥ͡˥ɤƤ⥸塼ΥꥹȤ롣
     *
     * @return ɺѤߥ⥸塼ꥹ
     *
     * @else
     * @brief Get a list of loaded modules
     *
     * Get module list that is currently loaded into manager.
     *
     * @return Module list that has been loaded.
     *
     * @endif
     */
    std::vector<coil::Properties> getLoadedModules();
    
    /*!
     * @if jp
     * @brief ɲǽʥ⥸塼ꥹȤ
     *
     * ɲǽ⥸塼ΥꥹȤ롣
     * (ߤModuleManager¦̤)
     *
     * @return ɲǽ⥸塼롡ꥹ
     *
     * @else
     * @brief Get a list of loadable modules
     *
     * Get loadable module list.
     * (Currently, unimplemented on ModuleManager side)
     *
     * @return Loadable module list
     *
     * @endif
     */
  std::vector<coil::Properties> getLoadableModules();
    
    //============================================================
    // Component Factory Management
    //============================================================
    /*!
     * @if jp
     * @brief RTݡͥѥեȥϿ
     *
     * RTݡͥȤΥ󥹥󥹤뤿
     * FactoryϿ롣
     *
     * @param profile RTݡͥ ץե
     * @param new_func RTݡͥѴؿ
     * @param delete_func RTݡͥ˴Ѵؿ
     *
     * @return Ͽ(Ͽ:true:false)
     *
     * @else
     * @brief Register RT-Component Factory
     *
     * Register Factory to create RT-Component's instances.
     *
     * @param profile RT-Component profile
     * @param new_func RT-Component creation function
     * @param delete_func RT-Component destruction function
     *
     * @return Registration result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool registerFactory(coil::Properties& profile,
			 RtcNewFunc new_func,
			 RtcDeleteFunc delete_func);

    /*!
     * @if jp
     * @brief եȥΥץե
     *
     * եȥΥץե롣
     *
     * @return եȥΥץե
     *
     * @else
     * @brief Get profiles of factories. 
     *
     * Get profiles of factories. 
     *
     * @return profiles of factories
     *
     * @endif
     */
    std::vector<coil::Properties> getFactoryProfiles();

    /*!
     * @if jp
     * @brief ExecutionContextѥեȥϿ
     *
     * ExecutionContextΥ󥹥󥹤뤿
     * FactoryϿ롣
     *
     * @param name оExecutionContext̾
     * @param new_func ExecutionContextѴؿ
     * @param delete_func ExecutionContext˴Ѵؿ
     *
     * @return Ͽ(Ͽ:true:false)
     *
     * @else
     * @brief Register ExecutionContext Factory
     *
     * Register Factory to create ExecutionContext's instances.
     *
     * @param name ExecutionContext name for the creation 
     * @param new_func ExecutionContext creation function
     * @param delete_func ExecutionContext destruction function
     *
     * @return Registration result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool registerECFactory(const char* name,
			   ECNewFunc new_func,
			   ECDeleteFunc delete_func);
    
    /*!
     * @if jp
     * @brief եȥꥹȤ
     *
     * ϿƤեȥꥹȤ롣
     *
     * @return Ͽեȥ ꥹ
     *
     * @else
     * @brief Get the list of all Factories
     *
     * Get the list of all factories that have been registered.
     *
     * @return Registered factory list
     *
     * @endif
     */
    std::vector<std::string> getModulesFactories();
    
    //============================================================
    // Component management
    //============================================================
    /*!
     * @if jp
     * @brief RTݡͥȤ
     *
     * ꤷRTݡͥȤΥ󥹥󥹤Ͽ줿Factoryͳ
     * 롣
     *
     * 륳ݡͥȤγƼץեϰʲ̤ͥ
     * ꤵ롣
     *
     * -# createComponent() ΰͿ줿ץե
     * -# rtc.confǻꤵ줿եͿ줿ץե
     * --# category.instance_name.config_file
     * --# category.component_type.config_file
     * -# ɤޤ줿ץե 
     *
     * 󥹥硢ʻưʲν¹Ԥ롣
     *  - եꤷե졼ɤ߹ߡ
     *  - ExecutionContextΥХɡư
     *  - ͡ߥ󥰥ӥؤϿ
     *
     * @param comp_args оRTݡͥIDӥե졼
     * եޥåȤ礭ʬ "id"  "configuration" 
     * ʬ¸ߤ롣
     *
     * comp_args:     [id]?[configuration]
     *                id ɬܡconfigurationϥץ
     * id:            RTC:[vendor]:[category]:[implementation_id]:[version]
     *                RTC ϸ꤫ɬ
     *                vendor, category, version ϥץ
     *                implementation_id ɬ
     *                ץάǤ ":" ϾάԲ
     * configuration: [key0]=[value0]&[key1]=[value1]&[key2]=[value2].....
     *                RTCPropertiesͤ򤹤٤ƾ񤭤뤳ȤǤ롣
     *                key=value ηǵҤ"&" Ƕڤ
     *
     * 㤨С
     * RTC:jp.go.aist:example:ConfigSample:1.0?conf.default.str_param0=munya
     * RTC::example:ConfigSample:?conf.default.int_param0=100
     *
     * @return RTݡͥȤΥ󥹥
     *
     * @else
     * @brief Create RT-Components
     *
     * Create specified RT-Component's instances via registered Factory.
     * When its instances have been created successfully, the following
     * processings are also executed.
     *  - Read and set configuration information that was set by external file.
     *  - Bind ExecutionContext and start operation.
     *  - Register to naming service.
     *
     * @param module_name Target RT-Component names for the creation
     *
     * @return Created RT-Component's instances
     *
     * @endif
     */
    RTObject_impl* createComponent(const char* comp_args);
    /*!
     * @if jp
     * @brief Context
     *
     * @return ConetextΥ󥹥
     *
     * @else
     * @brief Create Context
     *
     * @return Created Context's instances
     *
     * @endif
     */
    ExecutionContextBase* createContext(const char* ec_args);
    
    /*!
     * @if jp
     * @brief RTݡͥȤϿ
     *
     * ꤷRTݡͥȤΥ󥹥󥹤͡ߥ󥰥ӥ
     * Ͽ롣
     *
     * @param comp ϿоRTݡͥ
     *
     * @else
     * @brief Unregister RT-Components
     *
     * Unregister specified RT-Component's instances from naming service.
     *
     * @param comp Target RT-Components for the unregistration
     *
     * @endif
     */
    void cleanupComponent(RTObject_impl* comp);

    /*!
     * @if jp
     * @brief RTݡͥȤκ
     *
     * notifyFinalized()ˤäϿ줿RTݡͥȤ롣
     *
     * @else
     * @brief This method deletes RT-Components. 
     *
     * This method deletes RT-Components registered by notifyFinalized(). 
     *
     * @endif
     */
    void cleanupComponents();

    /*!
     * @if jp
     * @brief RTݡͥȤκ
     *
     * RTݡͥȤϿ롣
     * Ͽ줿RTݡͥȤ cleanupComponents() Ǻ롣
     *
     * @param RTݡͥ
     *
     * @else
     * @brief This method deletes RT-Components. 
     *
     * The deleted RT-Component is registered. The registered RT-Components 
     * are deleted by cleanupComponents(). 
     *
     * @param Deleted RT component
     * @endif
     */
    void notifyFinalized(RTObject_impl* comp);

    /*!
     * @if jp
     * @brief RTݡͥȤľ Manager Ͽ
     *
     * ꤷRTݡͥȤΥ󥹥󥹤
     * եȥͳǤϤʤľܥޥ͡Ͽ롣
     *
     * @param comp ϿоRTݡͥȤΥ󥹥
     *
     * @return Ͽ(Ͽ:true:false)
     *
     * @else
     * @brief Register RT-Component directly without Factory
     *
     * Register specified RT-Component's instances not via Factory
     * to Manager directly.
     *
     * @param comp Target RT-Component's instances for the registration
     *
     * @return Registration result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool registerComponent(RTObject_impl* comp);
    
    /*!
     * @if jp
     * @brief RTݡͥȤϿ
     *
     * ꤷRTݡͥȤϿ롣
     *
     * @param comp ϿоRTݡͥȤΥ󥹥
     *
     * @return Ͽ(:true:false)
     *
     * @else
     * @brief Unregister RT-Components
     *
     * Unregister specified RT-Components
     *
     * @param comp Target RT-Component's instances for the unregistration
     *
     * @return Unregistration result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool unregisterComponent(RTObject_impl* comp);
    
    
    /*!
     * @if jp
     * @brief Manager ϿƤRTݡͥȤ
     *
     * ޥ͡ϿƤRTݡͥȤ롣
     * ꤵ줿RTݡͥȤ͡ߥ󥰥ӥ
     * RTݡͥȼΤλȤȤˡ󥹥󥹤롣
     *
     * @param comp оRTݡͥȤΥ󥹥
     *
     * @else
     * @brief Unregister RT-Components that have been registered to Manager
     *
     * Unregister RT-Components that have been registered to manager
     * Remove specified RT-Component from naming service, terminate itself
     * and release its instances.
     *
     * @param comp Target RT-Component's instances for the unregistration
     *
     * @endif
     */
    void deleteComponent(RTObject_impl* comp);
    
    /*!
     * @if jp
     * @brief Manager ϿƤRTݡͥȤ
     *
     * ޥ͡ϿƤRTݡͥȤ롣
     * ꤵ줿RTݡͥȤ͡ߥ󥰥ӥ
     * RTݡͥȼΤλȤȤˡ󥹥󥹤롣
     *
     * @param instance_name оRTݡͥȤΥ󥹥̾
     *
     * @else
     * @brief Unregister RT-Components that have been registered to Manager
     *
     * Unregister RT-Components that have been registered to manager
     * Remove specified RT-Component from naming service, terminate itself
     * and release its instances.
     *
     * @param instance_name Target RT-Component's instances for the 
     *                      unregistration
     *
     * @endif
     */
    void deleteComponent(const char* instance_name);

    
    /*!
     * @if jp
     * @brief Manager ϿƤRTݡͥȤ򸡺
     *
     * Manager ϿƤRTݡͥȤꤷ̾ΤǸ
     * פ륳ݡͥȤ롣
     *
     * @param instance_name оRTݡͥȤ̾
     *
     * @return ̾ΤפRTݡͥȤΥ󥹥
     *
     * @else
     * @brief Get RT-Component's pointer
     *
     * Search RT-Component that has been registered to Manager by its specified
     * name, and get it that matches.
     *
     * @param instance_name Target RT-Component's name for searching
     *
     * @return Target RT-Component's instances that matches
     *
     * @endif
     */
    RTObject_impl* getComponent(const char* instance_name);
    
    /*!
     * @if jp
     * @brief Manager ϿƤRTݡͥȤ
     *
     * Manager ϿƤRTݡͥȤ󥹥󥹤롣
     *
     * @return RTݡͥȤΥ󥹥󥹥ꥹ
     *
     * @else
     * @brief Get all RT-Components registered in the Manager
     *
     * Get all RT-Component's instances that have been registered to Manager.
     *
     * @return List of all RT-Component's instances
     *
     * @endif
     */
    std::vector<RTObject_impl*> getComponents();
    
    //============================================================
    // CORBA Ϣ
    //============================================================
    /*!
     * @if jp
     * @brief ORB Υݥ󥿤
     *
     * Manager ꤵ줿 ORB Υݥ󥿤롣
     *
     * @return ORB ֥
     *
     * @else
     * @brief Get the pointer to ORB
     *
     * Get the pointer to ORB that has been set to Manager.
     *
     * @return ORB object
     *
     * @endif
     */
    CORBA::ORB_ptr getORB();
    
    /*!
     * @if jp
     * @brief Manager  RootPOA Υݥ󥿤
     *
     * Manager ꤵ줿 RootPOA ؤΥݥ󥿤롣
     *
     * @return RootPOA֥
     *
     * @else
     * @brief Get a pointer to RootPOA held by Manager
     *
     * Get the pointer to RootPOA that has been set to Manager.
     *
     * @return RootPOA object
     *
     * @endif
     */
    PortableServer::POA_ptr getPOA();
    
    /*!
     * @if jp
     * @brief Manager  POAManager 
     *
     * Manager ꤵ줿 POAMAnager 롣
     *
     * @return POAޥ͡
     *
     * @else
     * @brief Get POAManager that Manager has
     *
     * Get POAMAnager that has been set to Manager.
     *
     * @return POA manager
     *
     * @endif
     */
    PortableServer::POAManager_ptr getPOAManager();
    
    //============================================================
    // Protected functions
    //============================================================
  protected:
    
    //============================================================
    // Manager initialize and finalization
    //============================================================
    /*!
     * @if jp
     * @brief Manager 
     * 
     * Manager ¹Ԥ롣
     *  - Manager ե졼
     *  - ϥե
     *  - λѥåɤ
     *  - ѥåɤ(޻ѻ)
     *
     * @param argc ޥɥ饤ο
     * @param argv ޥɥ饤
     * 
     * @else
     * @brief Manager internal initialization
     * 
     * Execute Manager's internal initialization processing.
     *  - Set Manager configuration
     *  - Set log output file
     *  - Create termination processing thread
     *  - Create timer thread (when using timer)
     *
     * @param argc Number of commandline arguments
     * @param argv Commandline arguments
     * 
     * @endif
     */
    void initManager(int argc, char** argv);
    
    /*!
     * @if jp
     * @brief Manager νλ
     *
     * Manager λ
     * (ߤ̤)
     *
     * @else
     * @brief Shutdown Manager
     *
     * Shutdown Manager
     * (However, not implemented now)
     *
     * @endif
     */
    void shutdownManager();

    /*!
     * @if jp
     * @brief Manager νλ
     *
     * configuration  "manager.shutdown_on_nortcs" YES ǡ
     * ݡͥȤϿƤʤ Manager λ롣
     *
     * @else
     * @brief Shutdown Manager
     *
     * This method shutdowns Manager as follows.
     * - "Manager.shutdown_on_nortcs" of configuration is YES. 
     * - The component is not registered. 
     *
     * @endif
     */
    void shutdownOnNoRtcs();

    //============================================================
    // Logger initialize and terminator
    //============================================================
    /*!
     * @if jp
     * @brief System logger ν
     *
     * System logger ν¹Ԥ롣
     * ե졼եꤵ줿˴Ť
     * ν¹Ԥ롣
     *
     * @return ¹Է(:true:false)
     *
     * @else
     * @brief System logger initialization
     *
     * Initialize System logger.
     * Initialize logger and set it according to the set information in
     * configuration file,
     *
     * @return Initialization result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initLogger();
    
    /*!
     * @if jp
     * @brief System Logger νλ
     *
     * System Loggerνλ¹Ԥ롣
     * ХåեݻƤ¸ߤˤϡ
     * Ū˥ե˽ϤեĤ롣
     *
     * @else
     * @brief System Logger finalization
     *
     * Finalize System Logger.
     * If log information stored in the buffer exists, output information
     * to the log file forcibly and close it.
     *
     * @endif
     */
    void shutdownLogger();
    
    //============================================================
    // ORB initialization and finalization
    //============================================================
    /*!
     * @if jp
     * @brief CORBA ORB ν
     *
     * ˤͿ줿򸵤ORB롣
     *
     * @return ORB (:true:false)
     *
     * @else
     * @brief CORBA ORB initialization
     *
     * Initialize ORB based on the configuration given by arguments.
     *
     * @return ORB initialization result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initORB();
    
    /*!
     * @if jp
     * @brief ORB Υޥɥ饤󥪥ץ
     *
     * ե졼ꤵ줿Ƥ
     * ORB εưץ롣
     *
     * @return ORB ưץ
     *
     * @else
     * @brief Create ORB command options
     *
     * Create ORB launch options from configuration information
     * that has been set.
     *
     * @return ORB launch options
     *
     * @endif
     */
    std::string createORBOptions();

    /*!
     * @if jp
     * @brief ɥݥȤ
     *
     * ե졼󤫤饨ɥݥȤ롣
     *
     * @param endpoints ɥݥȥꥹ
     *
     * @else
     * @brief Create Endpoints
     *
     * Create Endpoints from the configuration.
     * 
     * @param endpoints Endpoints list
     *
     * @endif
     */
    void createORBEndpoints(coil::vstring& endpoints);
    
    /*!
     * @if jp
     * @brief ORB  Endpoint Υޥɥ饤󥪥ץ
     * @param opt ޥɥ饤󥪥ץ
     * @param endpoint ɥݥȥꥹ
     *
     * @else
     * @brief Create a command optional line of Endpoint of ORB.
     * @param opt ORB options
     * @param endpoint Endpoints list
     *
     * @endif
     */
    void createORBEndpointOption(std::string& opt, coil::vstring& endpoint);

    /*!
     * @if jp
     * @brief ORB νλ
     *
     * ORB νλ¹Ԥ롣
     * ¹Ԥν¸ߤˤϡνλޤԤġ
     * ºݤνλǤϡPOA Manager ORB Υåȥ¹
     * 롣
     *
     * @else
     * @brief ORB finalization
     *
     * Finalize ORB .
     * When the waiting process exists, wait until it completes.
     * In actual finalization, deactivate POA Manager and then shutdown of ORB.
     *
     * @endif
     */
    void shutdownORB();
    
    //============================================================
    // NamingService initialization and finalization
    //============================================================
    /*!
     * @if jp
     * @brief NamingManager ν
     *
     * NamingManager ν¹Ԥ롣
     * NamingManager Ѥʤ褦˥ץѥƥꤵƤ
     * ˤϲ⤷ʤ
     * NamingManager Ѥ硢ץѥƥꤵƤ
     * ǥե NamingServer Ͽ롣
     * ޤŪ˾򹹿褦ꤵƤˤϡꤵ줿
     * ǼưԤΥޤưȤȤˡѥ᥽åɤ򥿥ޤ
     * Ͽ롣
     *
     * @return (:true:false)
     *
     * @else
     * @brief NamingManager initialization
     *
     * Initialize NamingManager .
     * However, operate nothing, if it is set to property that NamingManager
     * is not used.
     * Register default NamingServer that is set to property information,
     * when NamingManager is used.
     * Also, launch a timer that updates information automatically at specified
     * cycle and register the method for the update to the timer, when it is set
     * to update it reguraly.
     *
     * @return Initialization result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initNaming();
    
    /*!
     * @if jp
     * @brief NamingManager νλ
     *
     * NamingManager λ롣
     * ϿƤǤ򥢥Хɤλ롣
     *
     * @else
     * @brief NamingManager finalization
     *
     * Finalize NamingManager.
     * Unbind all registered elements and shutdown them.
     *
     * @endif
     */
    void shutdownNaming();
    
    //============================================================
    // Component management
    //============================================================
    /*!
     * @if jp
     * @brief NamingManager ϿƤ RTݡͥȤνλ
     *
     * NamingManager ϿƤRTݡͥȤΥꥹȤ
     * ݡͥȤλ롣
     *
     * @else
     * @brief NamingManager finalization
     *
     * Get a list of RT-Components that have been registered to NamingManager,
     * and shutdown all components.
     *
     * @endif
     */
    void shutdownComponents();


    /*!
     * @if jp
     * @brief ʸ󤫤饳ݡͥȷ̾ץѥƥФ
     *
     * ʸ󤫤饳ݡͥȷȥݡͥȤΥץѥƥФ롣
     * ͿʸΥեޥåȤ RTC  ID ȥե졼
     * 󤫤ʤ
     *
     * [RTC type]?[key(0)]=[val(0)]&[key(1)]=[val(1)]&...&[key(n)]=[val(n)]
     * 
     * Ǥ롣ʤRTC type  implementation_id Τߡ⤷ϡ
     *  RTC ID 
     *
     * RTC:[vendor]:[category]:[impl_id]:[version]
     *
     * դ롣ͤǤ롢comp_id ϡ
     * "vendor", "category", "implementation_id", "version" Υ
     * Properties Υ֥ȤȤ֤롣
     * comp_conf ˤ "?" ʲ˵Ҥ륳ݡͥȤͿץѥƥ
     *  Properties Υ֥ȤȤ֤롣
     * 
     * @return comp_arg ˥ݡͥȷޤޤƤʤfalse
     * @param comp_arg  ٤ʸ
     * @param comp_id Ф줿ݡͥȤη̾
     * @param comp_conf Ф줿ݡͥȤΥץѥƥ
     *
     * @else
     * @brief Extracting component type/properties from the given string
     *
     * This operation extracts component type name and its properties
     * from the figen character string.
     * The given string formats is the following.
     *
     * [RTC type]?[key(0)]=[val(0)]&[key(1)]=[val(1)]...[key(n)]=[val(n)]
     *
     * Returned value "comp_id" has keys of "vendor", "category",
     * "implementation_id", "version", and returned as Properties type
     * object. "comp_conf" is returned as Properties type object
     * includeing component properties to be given to component.
     * 
     * @return comp_arg false will returned if no component type in arg
     * @param comp_arg  character string to be processed
     * @param comp_type extracted component type name
     * @param comp_prop extracted component's properties
     *
     * @endif
     */
    bool procComponentArgs(const char* comp_arg,
                           coil::Properties& comp_id,
                           coil::Properties& comp_conf);
    /*!
     * @if jp
     * @brief ʸ󤫤ExecutionContext̾ץѥƥФ
     *
     * ʸ󤫤ExecutionContext̾ȥץѥƥФ롣
     * ͿʸΥեޥåȤ RTC  ID ȥե졼
     * 󤫤ʤ
     *
     * [ExecutionContext̾]?[key(0)]=[val(0)]&[key(1)]=[val(1)]&...&[key(n)]=[val(n)]
     * 
     * Ǥ롣
     *
     * ec_conf ˤ "?" ʲ˵Ҥ륳ݡͥȤͿץѥƥ
     *  Properties Υ֥ȤȤ֤롣
     * 
     * @return ec_args ExecutionContext̾ޤޤƤʤfalse
     * @param ec_args  ٤ʸ
     * @param ec_id Ф줿ExecutionContext̾
     * @param ec_conf Ф줿ExecutionContextΥץѥƥ
     *
     * @else
     * @brief Extracting ExecutionContext's name/properties from the given 
     *        string
     *
     * This operation extracts ExecutionContext's name and its properties
     * from the figen character string.
     * The given string formats is the following.
     *
     * [ExecutionContext's name]?[key(0)]=[val(0)]&[key(1)]=[val(1)]...[key(n)]=[val(n)]
     *
     * "ec_conf" is returned as Properties type object
     * includeing component properties to be given to component.
     * 
     * @return ec_arg false will returned if no ExecutionContext's name in arg
     * @param ec_arg  character string to be processed
     * @param ec_type extracted ExecutionContext's name
     * @param ec_prop extracted ExecutionContext's properties
     *
     * @endif
     */
    bool procContextArgs(const char* ec_args,
                         std::string& ec_id,
                         coil::Properties& ec_conf);

    /*!
     * @if jp
     * @brief RTݡͥȤΥե졼
     *
     * RTݡͥȤηӥ󥹥˵ܤ줿ץѥƥե
     * ɤ߹ߡݡͥȤꤹ롣
     * ޤƥݡͥȤ NamingService Ͽ̾Τꤹ롣
     *
     * @param comp ե졼оRTݡͥ
     *
     * @else
     * @brief Configure RT-Component
     *
     * Read property files described each RT-Component's type and instance,
     * and configure it to the component.
     * Also, get each component's registered name when registering to
     * NamingService and configure it.
     *
     * @param comp Target RT-Component for the configuration
     *
     * @endif
     */
    void configureComponent(RTObject_impl* comp, const coil::Properties& prop);
    
    /*!
     * @if jp
     * @brief ExecutionContextManager ν
     *
     * Ѥ ExecutionContext ν¹Ԥ ExecutionContext 
     *  Factory  ExecutionContextManager Ͽ롣
     *
     * @return ExecutionContextManager ¹Է
     *         (:true:false)
     *
     * @else
     * @brief ExecutionContextManager initialization
     *
     * Initialize each ExecutionContext that is used, and register each 
     * ExecutionContext creation Factory to ExecutionContextManager.
     *
     * @return ExecutionContextManager initialization result
     *          (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initExecContext();

    /*!
     * @if jp
     * @brief PeriodicECSharedComposite ν
     *
     * @return PeriodicECSharedComposite ¹Է
     *         (:true:false)
     *
     * @else
     * @brief PeriodicECSharedComposite initialization
     *
     * @return PeriodicECSharedComposite initialization result
     *          (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initComposite();

    /*!
     * @if jp
     * @brief եȥν
     *
     * Хåեåɡѥ֥å㡢ץХ󥷥塼ޤ
     * եȥ롣
     *
     * @return եȥ¹Է
     *         (:true:false)
     *
     * @else
     * @brief Factories initialization
     *
     * Initialize buffer factories, thread factories, publisher factories, 
     * provider factories, and consumer factories. 
     *
     * @return PeriodicECSharedComposite initialization result
     *          (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initFactories();

    /*!
     * @if jp
     * @brief Timer ν
     *
     * Ѥ Timer ν¹Ԥ롣
     * (μǤϲ⤷ʤ)
     *
     * @return Timer ¹Է(:true:false)
     *
     * @else
     * @brief Timer initialization
     *
     * Initialize each Timer that is used.
     * (In current implementation, nothing is done.)
     *
     * @return Timer Initialization result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initTimer();

    /*!
     * @if jp
     * @brief ManagerServant ν
     *
     * @return Timer ¹Է(:true:false)
     *
     * @else
     * @brief ManagerServant initialization
     *
     * @return Timer Initialization result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool initManagerServant();

    /*!
     * @if jp
     * @brief ManagerServant ؤΥݥ
     * @else
     * @brief The pointer to the ManagerServant
     * @endif
     */
    RTM::ManagerServant* m_mgrservant;

    /*!
     * @if jp
     * @brief ץѥƥΥޡ
     *
     * ꤵ줿եꤵƤץѥƥɤ
     * ¸Ѥߥץѥƥȥޡ롣
     *
     * @param prop ޡоݥץѥƥ
     * @param file_name ץѥƥ󤬵ҤƤե̾
     *
     * @return ޡ¹Է(ޡ:trueޡ:false)
     *
     * @else
     * @brief Merge property information
     *
     * Load property information that is configured in the specified file,
     * and merge existing properties that has been configured.
     *
     * @param prop Target properties for the merge
     * @param file_name File name that property information is described
     *
     * @return Merge result (Successful:true, Failed:false)
     *
     * @endif
     */
    bool mergeProperty(coil::Properties& prop, const char* file_name);
    
    /*!
     * @if jp
     * @brief NamingServer ϿݤϿȤΩƤ
     *
     * ꤵ줿񼰤ȥץѥƥ NameServer Ͽݤξ
     * ȤΩƤ롣
     * ƽ񼰻ʸΰ̣ϰʲΤȤ
     * - % : ƥȤζڤ
     * - n : 󥹥̾
     * - t : ̾
     * - m : ̾
     * - v : С
     * - V : ٥
     * - c : ƥ
     * - h : ۥ̾
     * - M : ޥ̾͡
     * - p : ץID
     *
     * @param naming_format NamingService Ͽ񼰻
     * @param prop Ѥץѥƥ
     *
     * @return Ѵ
     *
     * @else
     * @brief Construct registration information when registering to 
     *        Naming server
     *
     * Construct information when registering to NameServer based on specified
     * format and property information.
     * Each format specification character means as follows:
     * - % : Break of Context
     * - n : Instance's name
     * - t : Type name
     * - m : Type name
     * - v : Version
     * - V : Vender
     * - c : Category
     * - h : Host name
     * - M : Manager name
     * - p : Process ID
     *
     * @param naming_format Format specification for NamingService registration
     * @param prop Property information that is used
     *
     * @return Specification format conversion result
     *
     * @endif
     */
    std::string formatString(const char* naming_format,
			     coil::Properties& prop);


    
    //============================================================
    // protected ѿ
    //============================================================
    
    //------------------------------------------------------------
    // static var
    //------------------------------------------------------------
    /*!
     * @if jp
     * @brief ͣ Manager ؤΥݥ
     * @else
     * @brief The pointer to the Manager
     * @endif
     */
    static Manager* manager;
    
    /*!
     * @if jp
     * @brief ͣ Manager ؤΥݥ󥿤Ф mutex
     * @else
     * @brief The mutex of the pointer to the Manager 
     * @endif
     */
    static Mutex mutex;
    
    //------------------------------------------------------------
    // CORBA var
    //------------------------------------------------------------
    /*!
     * @if jp
     * @brief ORB ؤΥݥ
     * @else
     * @brief The pointer to the ORB
     * @endif
     */
    CORBA::ORB_var m_pORB;
    
    /*!
     * @if jp
     * @brief POA ؤΥݥ
     * @else
     * @brief The pointer to the POA
     * @endif
     */
    PortableServer::POA_var m_pPOA;
    
    /*!
     * @if jp
     * @brief POAManager ؤΥݥ
     * @else
     * @brief The pointer to the POAManager
     * @endif
     */
    PortableServer::POAManager_var m_pPOAManager;
    
    //------------------------------------------------------------
    // Manager's variable
    //------------------------------------------------------------
    /*!
     * @if jp
     * @brief 桼ؿؤΥݥ
     * @else
     * @brief User's initialization function's pointer
     * @endif
     */
    ModuleInitProc m_initProc;
    
    /*!
     * @if jp
     * @brief Manager  configuration Ǽ Properties
     * @else
     * @brief Managaer's configuration Properties
     * @endif
     */
    coil::Properties m_config;
    
    /*!
     * @if jp
     * @brief ModuleManager ؤΥݥ
     * @else
     * @brief The pointer to the ModuleManager
     * @endif
     */
    ModuleManager* m_module;
    
    /*!
     * @if jp
     * @brief NamingManager ؤΥݥ
     * @else
     * @brief The pointer to the NamingManager
     * @endif
     */
    NamingManager* m_namingManager;
    
    /*!
     * @if jp
     * @brief Timer Object
     * @else
     * @brief Timer Object
     * @endif
     */
    coil::Timer* m_timer;
    
    //------------------------------------------------------------
    // Logger
    //------------------------------------------------------------
    /*!
     * @if jp
     * @brief Хåե
     * @else
     * @brief Logger buffer
     * @endif
     */
    LogStreamBuf m_logStreamBuf;
    
    /*!
     * @if jp
     * @brief ȥ꡼
     * @else
     * @brief Logger stream
     * @endif
     */
    Logger rtclog;

    /*!
     * @if jp
     * @brief ϥե
     * @else
     * @brief Files for log output
     * @endif
     */
    std::vector<std::filebuf*> m_logfiles;
    
    //============================================================
    // ݡͥȥޥ͡
    //============================================================
    // ObjectManager ϤҸ쥯饹
    struct InstanceName
    {
      InstanceName(RTObject_impl* comp);
      InstanceName(const char* name);
      InstanceName(const std::string name);
      bool operator()(RTObject_impl* comp);
      std::string m_name;
    };
    
    typedef ObjectManager<std::string, RTObject_impl, InstanceName> ComponentManager;

    /*!
     * @if jp
     * @brief ݡͥȥޥ͡
     * @else
     * @brief ComponentManager
     * @endif
     */
    ComponentManager m_compManager;
    
    //============================================================
    // ݡͥȥեȥ
    //============================================================
    // ݡͥȥեȥϤҸ쥯饹
    class FactoryPredicate
    {
    public:
      FactoryPredicate(const char* imple_id)
        : m_vendor(""), m_category(""), m_impleid(imple_id), m_version("")
      {
      }
      FactoryPredicate(const coil::Properties& prop)
        : m_vendor(prop["vendor"]),
          m_category(prop["category"]),
          m_impleid(prop["implementation_id"]),
          m_version(prop["version"])
      {
      }
      FactoryPredicate(FactoryBase* factory)
	: m_vendor(factory->profile()["vendor"]),
          m_category(factory->profile()["category"]),
          m_impleid(factory->profile()["implementation_id"]),
          m_version(factory->profile()["version"])
      {
      }
      bool operator()(FactoryBase* factory)
      {
        // implementation_id must not be empty
        if (m_impleid.empty()) return false;

        const coil::Properties& prop(factory->profile());

        if (m_impleid != prop["implementation_id"])
          return false;
        if (!m_vendor.empty()   && m_vendor != prop["vendor"])
          return false;
        if (!m_category.empty() && m_category != prop["category"])
          return false;
        if (!m_version.empty()  && m_version != prop["version"])
          return false;
          
        return true;
      }
    private:
      std::string m_vendor;
      std::string m_category;
      std::string m_impleid;
      std::string m_version;
    };
    
    class ModulePredicate
    {
      coil::Properties& m_prop;
    public:
      ModulePredicate(coil::Properties& prop)
      : m_prop(prop)
      {
      }
      bool operator()(coil::Properties& prop)
      {
        if (m_prop["implementation_id"] != prop["implementation_id"])
          {
            return false;
          }
        if (!m_prop["vendor"].empty() &&
            m_prop["vendor"] != prop["vendor"])     { return false; }
        if (!m_prop["category"].empty() &&
            m_prop["category"] != prop["category"]) { return false; }
        if (!m_prop["version"].empty() && 
            m_prop["version"] != prop["version"])   { return false; }
        return true;
      }
    };

    /*!
     * @if jp
     * @brief ݡͥȥեȥ
     * @else
     * @brief ComponentFactory
     * @endif
     */
    typedef ObjectManager<const coil::Properties, FactoryBase,
			  FactoryPredicate> FactoryManager;

    /*!
     * @if jp
     * @brief ComponentManager
     * @else
     * @brief ComponentManager
     * @endif
     */
    FactoryManager m_factory;
    
    //============================================================
    // ExecutionContextեȥ
    //============================================================
    // ECեȥϤҸ쥯饹
    struct ECFactoryPredicate
    {
      ECFactoryPredicate(const char* name) : m_name(name){};
      ECFactoryPredicate(ECFactoryBase* factory)
	: m_name(factory->name()) {};
      bool operator()(ECFactoryBase* factory)
      {
	return m_name == factory->name();
      }
      std::string m_name;
    };
    typedef ObjectManager<const char*,
			  ECFactoryBase,
			  ECFactoryPredicate> ECFactoryManager;
    
    /*!
     * @if jp
     * @brief ExecutionContext ޥ͡
     * @else
     * @brief ExecutionContext Manager
     * @endif
     */
    ECFactoryManager m_ecfactory;
    
    /*!
     * @if jp
     * @brief ExecutionContext ꥹ
     * @else
     * @brief ExecutionContext list
     * @endif
     */
    std::vector<ExecutionContextBase*> m_ecs;
    
    // եȥ̾ꥹȥåפ뤿Υե󥯥
    struct ModuleFactories
    {
      void operator()(FactoryBase* f)
      {
	modlist.push_back(f->profile().getProperty("implementation_id"));
      }
      std::vector<std::string> modlist;
    };
    
    //------------------------------------------------------------
    // ORB runner
    //------------------------------------------------------------
    /*!
     * @if jp
     * @class OrbRunner
     * @brief OrbRunner 饹
     *
     * ORB ¹ѥإѡ饹
     *
     * @since 0.4.0
     *
     * @else
     * @class OrbRunner
     * @brief OrbRunner class
     *
     * ORB exrcution helper class
     *
     * @since 0.4.0
     *
     * @endif
     */
    class OrbRunner
      : public coil::Task
    {
    public:
      /*!
       * @if jp
       * @brief 󥹥ȥ饯
       *
       * 󥹥ȥ饯
       *
       * @else
       * @brief Constructor
       *
       * Constructor
       *
       * @endif
       */
      OrbRunner(CORBA::ORB_ptr orb) : m_pORB(orb)
      {
	open(0);
      };
      
      /*!
       * @if jp
       * @brief ORB 
       *
       * ORB 
       *
       * @param args 
       *
       * @return 
       *
       * @else
       * @brief ORB activation processing
       *
       * ORB activation processing.
       *
       * @param args ORB activation processing
       *
       * @return Activation result
       *
       * @endif
       */
      virtual int open(void *args)
      {
	activate();
	return 0;
      }
      
      /*!
       * @if jp
       * @brief ORB Ͻ
       *
       * ORB Ͻ
       *
       * @return Ͻ
       *
       * @else
       * @brief ORB start processing
       *
       * ORB start processing
       *
       * @return Starting result
       *
       * @endif
       */
      virtual int svc(void)
      {
	m_pORB->run();
//	Manager::instance().shutdown();
	return 0;
      }
      
      /*!
       * @if jp
       * @brief ORB λ
       *
       * ORB λ
       *
       * @param flags λե饰
       *
       * @return λ
       *
       * @else
       * @brief ORB close processing
       *
       * ORB close processing.
       *
       * @param flags Flag of close processing
       *
       * @return Close result
       *
       * @endif
       */
      virtual int close(unsigned long flags)
      {
	return 0;
      }
    private:
      CORBA::ORB_ptr m_pORB;
    };
    /*!
     * @if jp
     * @brief ORB إѡ饹ؤΥݥ
     * @else
     * @brief The pointer to ORB helper class
     * @endif
     */
    OrbRunner* m_runner;
    
    //------------------------------------------------------------
    // Manager Terminator
    //------------------------------------------------------------
    /*!
     * @if jp
     * @class Terminator
     * @brief Terminator 饹
     *
     * ORB λѥإѡ饹
     *
     * @since 0.4.0
     *
     * @else
     * @class Terminator
     * @brief Terminator class
     *
     * ORB termination helper class.
     *
     * @since 0.4.0
     *
     * @endif
     */
    class Terminator
      : public coil::Task
    {
    public:
      /*!
       * @if jp
       * @brief 󥹥ȥ饯
       *
       * 󥹥ȥ饯
       *
       * @param manager ޥ͡㡦֥
       *
       * @else
       * @brief Constructor
       *
       * Constructor
       *
       * @param manager Manager object
       *
       * @endif
       */
      Terminator(Manager* manager) : m_manager(manager) {};
      
      /*!
       * @if jp
       * @brief λ
       *
       * ORBޥ͡㽪λ򳫻Ϥ롣
       *
       * @else
       * @brief Termination processing
       *
       * Start ORB and manager's termination processing.
       *
       * @endif
       */
      void terminate()
      {
	open(0);
      }
      
      /*!
       * @if jp
       * @brief λ
       *
       * λ
       *
       * @param args 
       *
       * @return 
       *
       * @else
       * @brief Termination processing activation
       *
       * Termination processing activation.
       *
       * @param args Activation argument
       *
       * @return Activation result
       *
       * @endif
       */
      virtual int open(void *args)
      {
	activate();
	return 0;
      }
      
      /*!
       * @if jp
       * @brief ORBޥ͡㽪λ
       *
       * ORBޥ͡㽪λ
       *
       * @return λ
       *
       * @else
       * @brief ORB and manager's termination processing
       *
       * ORB and manager's termination processing.
       *
       * @return Termination result
       *
       * @endif
       */
      virtual int svc(void)
      {
	Manager::instance().shutdown();
	return 0;
      }
      Manager* m_manager;
    };
    
    /*!
     * @if jp
     * @brief ORB λѥإѡ饹ؤΥݥ
     * @else
     * @brief The pointer to ORB termination helper class.
     * @endif
     */
    Terminator* m_terminator;
    
    struct Term
    {
      int waiting;
      Mutex mutex;
    };
    /*!
     * @if jp
     * @brief ޥ͡㽪λƱե饰
     *
     * ޥ͡㽪λԤ礻Ʊ뤿Υե饰
     *
     * @else
     * @brief Synchronous flag for manager termination
     *
     * Flag used to take synchronization by join(). 
     * 
     * @endif
     */
    Term m_terminate;

    struct Finalized
    {
      Mutex mutex;
      std::vector<RTObject_impl*> comps;
    };
    Finalized m_finalized;


  }; // class Manager
}; // namespace RTC

#endif // RTC_MANAGER_H
