// -*- C++ -*-
/*!
 * @file PortProfileHelper.h
 * @brief RTC's PortProfile helper class
 * @date $Date: 2007-04-26 15:32:25 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef RTC_PORTPROFILEHELPER_H
#define RTC_PORTPROFILEHELPER_H


// RTC header include
#include <rtm/RTC.h>
#include <rtm/Util.h>

// ACE includes
#include <coil/Mutex.h>

// CORBA header include
#include <rtm/idl/RTCSkel.h>




namespace RTC
{
  /*!
   * @if jp
   *
   * @class PortProfileHelper
   * @brief PortProfile إѡ饹
   *
   * RTC::Port μΥץեݻ PortProfile 륯饹
   * Ȥ PortBase ǻѤ롣
   *
   * @else
   *
   * @class PortProfileHelper
   * @brief PortProfile helper class
   *
   * This class manages the PortProfile that is profiles of the RTC:Port.
   * This is mainly used in PortBase class.
   *
   * @endif
   */
  class PortProfileHelper
  {
    typedef coil::Mutex Mutex;

  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *
     * @else
     * @brief Constructor
     *
     * Constructor
     *
     * @endif
     */
    PortProfileHelper();
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * Destructor
     *
     * @endif
     */
    virtual ~PortProfileHelper(void);


    /*!
     * @if jp
     *
     * @brief PortProfile ꤹ
     *
     * Υ֥Ȥݻ PortProfile Ϳ줿 PortProfile
     * 򥳥ԡ񤭤¸롣
     *
     * @param PortProfile 񤭤 PortProfile
     *
     * @else
     *
     * @brief Set PortProfile
     *
     * This operation copies the given PortProfile and overwrites the existent
     * PortProfile by the given ProtProfile.
     *
     * @param PortProfile The PortProfile to be stored.
     *
     * @endif
     */
    void setPortProfile(const PortProfile& profile);


    /*!
     * @if jp
     *
     * @brief PortProfile 
     *
     * Υ֥Ȥݻ PortProfile ֤
     *
     * @return Υ֥Ȥݻ PortProfile
     *
     * @else
     *
     * @brief Get PortProfile
     *
     * This operation returns the PortProfile.
     *
     * @return The PortProfile stored by the object.
     *
     * @endif
     */
    PortProfile* getPortProfile();


    /*!
     * @if jp
     *
     * @brief PortProfile.name ꤹ
     *
     * Υڥ졼ϰͿ줿ʸ򥳥ݡ
     * PortProfile.name Ȥݻ롣
     *
     * @param name PortProfile.name ˳Ǽ Port ̾
     *
     * @else
     *
     * @brief Set PortProfile.name
     *
     * This operation stores a copy of given name to the PortProfile.name.
     *
     * @param name The name of Port to be stored to the PortProfile.name.
     *
     * @endif
     */
    void setName(const char* name);


    /*!
     * @if jp
     *
     * @brief PortProfile.name 
     *
     * Υڥ졼 PortProfile.name 롣
     *
     * @return PortProfile.name ؤΥݥ
     *
     * @else
     *
     * @brief Get PortProfile.name
     *
     * This operation returns a pointer to the PortProfile.name.
     *
     * @return The pointer to PortProfile.name.
     *
     * @endif
     */
    const char* getName() const;


    /*!
     * @if jp
     *
     * @brief PortInterfaceProfile ɲä
     *
     * Υڥ졼 PortProfile  PortInterfaceProfile ɲä롣
     *
     * @param if_profile PortProfile ɲä PortInterfaceProfile
     *
     * @else
     *
     * @brief Append PortInterfaceProfile to the PortProfile
     *
     * This operation appends the PortInterfaceProfile to the PortProfile
     *
     * @param if_profile PortInterfaceProfile to be appended the PortProfile
     *
     * @endif
     */
    void appendPortInterfaceProfile(PortInterfaceProfile if_prof);


    /*!
     * @if jp
     *
     * @brief PortInterfaceProfileList 
     *
     * Υڥ졼 PortInterfaceProfileList ֤
     *
     * @return PortInterfaceProfileList
     *
     * @else
     *
     * @brief Get PortInterfaceProfileList
     *
     * This operation returns the PortInterfaceProfileList.
     *
     * @return PortInterfaceProfileList
     *
     * @endif
     */
    const PortInterfaceProfileList& getPortInterfaceProfiles() const;


    /*!
     * @if jp
     *
     * @brief PortInterfaceProfile 
     *
     * Υڥ졼 instance_name ǻꤵ줿 PortInterfaceProfile
     * ֤
     *
     * @param instance_name PortInterfaceProfile  instance_name
     * @return PortInterfaceProfile
     *
     * @else
     *
     * @brief Get PortInterfaceProfile
     *
     * This operation returns the PortInterfaceProfile specified
     * by instance_name.
     *
     * @param instance_name instance_name of the PortInterfaceProfile
     * @return PortInterfaceProfile
     *
     * @endif
     */
    const PortInterfaceProfile
    getPortInterfaceProfile(const char* instance_name) const;


    /*!
     * @if jp
     *
     * @brief PortInterfaceProfile 
     *
     * Υڥ졼 instance_name ǻꤵ줿PortInterfaceProfile
     * 롣ꤷ̾ PortInterfaceProfile ¸ߤʤˤϡ
     * NotFound exception ֤
     *
     * @param instance_name  PortInterfaceProfile ̾
     *
     * @else
     *
     * @brief Erase PortInterfaceProfile from the PortProfile
     *
     * This operation erases the PortInterfaceProfile from the PortProfile
     *
     * @param instance_name PortInterfaceProfile to be erased from the
     *        PortProfile
     *
     * @endif
     */
    void erasePortInterfaceProfile(const char* instance_name);


    /*!
     * @if jp
     *
     * @brief Port Υ֥ȻȤ򥻥åȤ
     *
     * Υڥ졼 PortProfile ˡϢ Port Υ֥Ȼ
     * ꤹ롣
     *
     * @param port ꤹ Port Υ֥ȥե
     *
     * @else
     *
     * @brief Set Port's object reference
     *
     * This operation set the object reference of the Port.
     *
     * @param port Port's object reference to be set.
     *
     * @endif
     */
    void setPortRef(PortService_ptr port);


    /*!
     * @if jp
     *
     * @brief Port Υ֥ȻȤ
     *
     * Υڥ졼 PortProfile ˴Ϣդ줿 Port 
     * ֥ȻȤ֤
     *
     * @return Ϣդ줿 Port Υ֥Ȼ
     *
     * @else
     *
     * @brief Get Port's object reference
     *
     * This operation returns the object reference of the PortProfile.
     *
     * @return Port's object reference associated with the PortProfile.
     *
     * @endif
     */
    PortService_ptr getPortRef() const;


    /*!
     * @if jp
     *
     * @brief ConnectorProfile ɲä
     *
     * Υڥ졼 PortProfile  ConnectorProfile ɲä롣
     *
     * @param conn_profile ConnectorProfile 
     *
     * @else
     *
     * @brief Append ConnectorProfile
     *
     * This operation appends the ConnectorProfile to the PortProfile.
     *
     * @param conn_profile ConnectorProfile to be added.
     *
     * @endif
     */
    void appendConnectorProfile(ConnectorProfile conn_profile);


    /*!
     * @if jp
     *
     * @brief ConnectorProfileList 
     *
     * Υڥ졼 PortProfile ˴Ϣդ줿 ConnectorProfile 
     * ꥹ ConnectorProfileList ֤
     *
     * @return Ϣդ줿 ConnectorProfileList
     *
     * @else
     *
     * @brief Get ConnectorProfileList
     *
     * This operation returns the list of ConnectorProfile of the PortProfile.
     *
     * @return Port's ConnectorProfileList.
     *
     * @endif
     */
    const ConnectorProfileList getConnectorProfiles() const;


    /*!
     * @if jp
     *
     * @brief ConnectorProfile 
     *
     * Υڥ졼ϰǻꤵ줿̾ ConnectorProfile ֤
     *
     * @param name ConnectorProfile ̾
     * @return ConnectorProfile
     *
     * @else
     *
     * @brief Get ConnectorProfile
     *
     * This operation returns the ConnectorProfile specified by name.
     *
     * @param name The name of ConnectorProfile
     * @return ConnectorProfile.
     *
     * @endif
     */
    const ConnectorProfile getConnectorProfile(const char* name) const;


     /*!
     * @if jp
     *
     * @brief ConnectorProfile 
     *
     * Υڥ졼ϰǻꤵ줿ID ConnectorProfile ֤
     *
     * @param id ConnectorProfile ID
     * @return ConnectorProfile
     *
     * @else
     *
     * @brief Get ConnectorProfile
     *
     * This operation returns the ConnectorProfile specified by ID.
     *
     * @param id The ID of ConnectorProfile
     * @return ConnectorProfile.
     *
     * @endif
     */
    const ConnectorProfile getConnectorProfileById(const char* id) const;


     /*!
     * @if jp
     *
     * @brief ConnectorProfile 
     *
     * Υڥ졼 PortProfile  ConnectorProfile 
     * ̾ǻꤷƺ롣
     *
     * @param naem ConnectorProfile ̾
     *
     * @else
     *
     * @brief Erase ConnectorProfile
     *
     * This operation erases the ConnectorProfile from the PortProfile.
     *
     * @param name The name of the ConnectorProfile to be erased.
     *
     * @endif
     */
    void eraseConnectorProfile(const char* name);


     /*!
     * @if jp
     *
     * @brief ConnectorProfile 
     *
     * Υڥ졼 PortProfile  ConnectorProfile 
     * ID ǻꤷƺ롣
     *
     * @param id ConnectorProfile ID
     *
     * @else
     *
     * @brief Erase ConnectorProfile
     *
     * This operation erases the ConnectorProfile from the PortProfile.
     *
     * @param id The ID of the ConnectorProfile to be erased.
     *
     * @endif
     */
    void eraseConnectorProfileById(const char* id);


     /*!
     * @if jp
     *
     * @brief PortProfile  owner ꤹ
     *
     * Υڥ졼 PortProfile  owner ꤹ롣
     *
     * @param owner PortProfile  owner Υ֥Ȼ
     *
     * @else
     *
     * @brief Set owner's object reference to the PortProfile
     *
     * This operation sets the owner's object reference to the PortProfile.
     *
     * @param owner The owner's object reference of PortProfile.
     *
     * @endif
     */
    void setOwner(RTObject_ptr owner);


    /*!
     * @if jp
     *
     * @brief PortProfile  owner 
     *
     * Υڥ졼 PortProfile  owner Υ֥ȻȤ֤
     *
     * @return PortProfile  owner Υ֥Ȼ
     *
     * @else
     *
     * @brief Get owner's object reference from the PortProfile
     *
     * This operation returns the owner's object reference of the PortProfile.
     *
     * @return The owner's object reference of PortProfile.
     *
     * @endif
     */
    RTObject_ptr getOwner() const;


    /*!
     * @if jp
     *
     * @brief PortProfile  properties ꤹ
     *
     * Υڥ졼 PortProfile  properties ꤹ롣
     *
     * @param prop PortProfile  properties  NVList
     *
     * @else
     *
     * @brief Set properties to the PortProfile
     *
     * This operation set the properties to the PortProfile.
     *
     * @param prop The NVList of PortProfile's properties.
     *
     * @endif
     */
    void setProperties(NVList& prop);


    /*!
     * @if jp
     *
     * @brief PortProfile  properties 
     *
     * Υڥ졼 PortProfile  properties֤
     *
     * @return PortProfile  properties  NVList
     *
     * @else
     *
     * @brief Get properties of the PortProfile
     *
     * This operation returns the properties of the PortProfile.
     *
     * @return The NVList of PortProfile's properties.
     *
     * @endif
     */
    const NVList& getProperties() const;



  private:
    // Specialization of SequenceEx template class


    // PortProfile.name
    std::string m_name;

    // PortProfile.interfaces
    typedef SequenceEx<PortInterfaceProfileList,
		       PortInterfaceProfile,
		       Mutex> IfProfiles;
    IfProfiles m_ifProfiles;

    // PortProfile.port_ref
    PortService_var m_portRef;

    // PortProfile.connector_profile
    typedef SequenceEx<ConnectorProfileList,
		       ConnectorProfile,
		       Mutex> ConnProfiles;
    ConnProfiles m_connProfiles ;

    // PortProfile.owner
    RTObject_var m_owner;

    // PortProfile.properties
    NVList m_properties;

    // Mutex
    mutable Mutex m_mutex;


    // Functor to find PortInterfaceProfile by name
    struct if_name
    {
      if_name(const char* name) :  m_name(name) {};
      bool operator()(const PortInterfaceProfile& p)
      {
	std::string name(p.instance_name);
	return m_name == name;
      }
      const std::string m_name;
    };

    // Functor to find ConnectorProfile by name
    struct conn_name
    {
      conn_name(const char* name) :  m_name(name) {};
      bool operator()(const ConnectorProfile& c)
      {
	std::string name(c.name);
	return m_name == name;
      }
      const std::string m_name;
    };
    
    // Functor to find ConnectorProfile by id
    struct conn_id
    {
      conn_id(const char* id) :  m_id(id) {};
      bool operator()(const ConnectorProfile& c)
      {
	std::string id(c.connector_id);
	return m_id == id;
      }
      const std::string m_id;
    };

  };   // class PortProfileHelper
};     // namespace RTC
#endif // RTC_PORTPROFILEHELPER_H
