// -*- C++ -*-
/*!
 * @file SdoConfiguration.cpp
 * @brief SDO's Configuration implementation class
 * @date $Date: 2008-01-14 07:49:37 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2008
 *     Noriaki Ando
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#include <coil/UUID.h>
#include "rtm/SdoConfiguration.h"
#include <rtm/CORBA_SeqUtil.h>
#include <rtm/NVUtil.h>
#include <rtm/ExecutionContextBase.h>
#include <memory>
#include <iostream>
// ACE

namespace SDOPackage
{
  /*!
   * @if jp
   *
   * @brief SDO ConfigurationSet ץѥƥ˳Ǽ
   * 
   * SDO ConfigurationSet RTCѥץѥƥ˳Ǽ롣
   *
   * @param prop Ǽץѥƥ
   * @param conf SDO ConfigurationSet
   * 
   * @else
   *
   * @brief Store SDO ConfigurationSet to properties
   * 
   * Store SDO ConfigurationSet to properties for RTC.
   *
   * @param prop Properties to store
   * @param conf SDO ConfigurationSet
   * 
   * @endif
   */
  void 
  toProperties(coil::Properties& prop, const SDOPackage::ConfigurationSet& conf)
  {
    NVUtil::copyToProperties(prop, conf.configuration_data);
  }
  
  /*!
   * @if jp
   *
   * @brief ץѥƥ SDO ConfigurationSetѴ
   * 
   * RTCѥץѥƥSDO ConfigurationSetѴ롣
   *
   * @param conf Ǽ SDO ConfigurationSet
   * @param prop ץѥƥ
   * 
   * @else
   *
   * @brief Convert properties into SDO ConfigurationSet
   * 
   * Convert properties for RTC into SDO ConfigurationSet.
   *
   * @param conf SDO ConfigurationSet to store
   * @param prop Properties
   * 
   * @endif
   */
  void
  toConfigurationSet(SDOPackage::ConfigurationSet& conf,
		     const coil::Properties& prop)
  {
#ifndef ORB_IS_RTORB
    conf.description = CORBA::string_dup(prop["description"].c_str());
    conf.id = CORBA::string_dup(prop.getName());
#else // ORB_IS_RTORB
    conf.description = (char *)prop["description"].c_str();
    conf.id = (char *)prop.getName();
#endif // ORB_IS_RTORB
    NVUtil::copyFromProperties(conf.configuration_data, prop);
  }
  
  //============================================================
  // Ctor and Dtor
  //============================================================
  
  /* @if jp
   * @brief 󥹥ȥ饯
   * @else
   * @brief Constructor
   * @endif
   */
  Configuration_impl::Configuration_impl(RTC::ConfigAdmin& configsets,
                                         RTC::SdoServiceAdmin& sdoServiceAdmin)
    : rtclog("sdo_config"), m_configsets(configsets),
      m_sdoservice(sdoServiceAdmin)
  {
    m_objref = this->_this();
  }
  
  /* @if jp
   * @brief ۥǥȥ饯
   * @else
   * @brief Virtual destructor
   * @endif
   */
  Configuration_impl::~Configuration_impl()
  {
  }

  
  //============================================================
  // Basic Configuration 
  //============================================================
  /* @if jp
   * @brief [CORBA interface] SDO  DeviceProfile 򥻥åȤ
   * @else
   * @brief [CORBA interface] Set DeviceProfile of SDO
   * @endif
   */
  CORBA::Boolean
  Configuration_impl::set_device_profile(const DeviceProfile& dProfile)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("set_device_profile()"));
    try
      {
	Guard gurad(m_dprofile_mutex);
	m_deviceProfile = dProfile;
      }
    catch (...)
      {
	throw InternalError("Unknown Error");
	// never reach here
	return false;
      }
    return true;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] SDO  ServiceProfile Υå
   * @else
   * @brief [CORBA interface] Set SDO's ServiceProfile
   * @endif
   */
  CORBA::Boolean
  Configuration_impl::add_service_profile(const ServiceProfile& sProfile)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("add_service_profile()"));
    // SDO specification defines that InvalidParameter() exception
    // is thrown when sProfile is null.
    // But sProfile is reference and it becomes never null.
    // So this operation does not throw InvalidParameter exception.
    //    if (CORBA::is_nil(sProfile.service)) throw InvalidParameter();
    try
      {
        return m_sdoservice.addSdoServiceConsumer(sProfile);
      }
    catch (...)
      {
	throw InternalError("Configuration::set_service_profile");
      }
    return false;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] Organization ɲ
   * @else
   * @brief [CORBA interface] Add Organization
   * @endif
   */ 
  CORBA::Boolean
  Configuration_impl::add_organization(Organization_ptr org)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("add_organization()"));
    try
      {
	CORBA_SeqUtil::push_back(m_organizations,
                                 ::SDOPackage::Organization::_duplicate(org));
      }
    catch (...)
      {
	throw InternalError("Configuration::set_service_profile");
	// never reach here
	return false;
      }
    return true;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ServiceProfile κ
   * @else
   * @brief [CORBA interface] Remove ServiceProfile
   * @endif
   */  
  CORBA::Boolean
  Configuration_impl::remove_service_profile(const char* id)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("remove_service_profile(%s)", id));
    try
      {
        return m_sdoservice.removeSdoServiceConsumer(id);
      }
    catch (...)
      {
	throw InternalError("Configuration::remove_service_profile");
      }
    return false;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] Organization λȤκ
   * @else
   * @brief [CORBA interface] Remove the reference of Organization 
   * @endif
   */
  CORBA::Boolean
  Configuration_impl::remove_organization(const char* organization_id)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("remove_organization(%s)", organization_id));
    try
      {
	Guard gurad(m_org_mutex);
	CORBA_SeqUtil::erase_if(m_organizations, org_id(organization_id));
      }
    catch (...)
      {
	throw InternalError("Configuration::remove_organization");
	// never reach here
	return false;
      }
    return true;
  }
  
  //============================================================
  // Configuration Parameter manipulation
  //============================================================
  /*!
   * @if jp
   * @brief [CORBA interface] ѥ᡼ΥꥹȤμ
   * @else
   * @brief [CORBA interface] Get a list of configuration parameters
   * @endif
   */ 
  ParameterList*
  Configuration_impl::get_configuration_parameters()
    throw (CORBA::SystemException,
	   NotAvailable, InternalError)
  {
    RTC_TRACE(("get_configuration_parameters()"));
    try
      {
	Guard gaurd(m_params_mutex);
	ParameterList_var param;
	param = new ParameterList(m_parameters);
	return param._retn();
      }
    catch (...)
      {
	throw InternalError("Configuration::get_configuration_parameters()");
	//never reach here
      }
    // never reach here
    return new ParameterList(0);
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] Configuration parameter ͤΥꥹȤμ
   * @else
   * @brief [CORBA interface] Get a list of the value of configuration parameters
   * @endif
   */
  NVList*
  Configuration_impl::get_configuration_parameter_values()
    throw (CORBA::SystemException,
	   NotAvailable, InternalError)
  {
    RTC_TRACE(("get_configuration_parameter_values()"));
    Guard guard(m_config_mutex);
    NVList_var nvlist;
    nvlist = new NVList((CORBA::ULong)0);
    
    /*
      CORBA::Long index;
      index = getActiveConfigIndex();
      if (index >= 0)
      {
      nvlist = new NVList(m_configurations[index].configuration_data);
      }
    */
    return nvlist._retn();
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] Configuration parameter ͤμ
   * @else
   * @brief [CORBA interface] Get the value of configuration parameter
   * @endif
   */
  CORBA::Any*
  Configuration_impl::get_configuration_parameter_value(const char* name)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("get_configuration_parameter_value(%s)", name));
    if (std::string(name).empty()) throw InvalidParameter("Name is empty.");
    
    //    CORBA::Long index;
    CORBA::Any_var value;
    value = new CORBA::Any();
    /*
      index = getActiveConfigIndex();
      if (index < 0) throw InternalError("No active configuration.");
      
      CORBA::Long item;
      item = CORBA_SeqUtil::find(m_configurations[index].configuration_data,
      nv_name(name));
      if (item < 0) throw InvalidParameter("No such name."); 
      
      value = new CORBA::Any(m_configurations[index].configuration_data[item].value);
    */
    return value._retn();
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] Configuration ѥ᡼ѹ
   * @else
   * @brief [CORBA interface] Modify the configuration parameter value
   * @endif
   */
  CORBA::Boolean
  Configuration_impl::set_configuration_parameter(const char* name,
						  const CORBA::Any& value)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("set_configuration_parameter(%s, value)", name));
    /*
      if (name == "") throw InvalidParameter("Name is empty.");
      
      CORBA::Long index(getActiveConfigIndex());
      if (index < 0) throw InternalError("No active config.");
      
      CORBA::Long item;
      item = CORBA_SeqUtil::find(m_configurations[index].configuration_data,
      nv_name(name));
      if (item < 0) throw InvalidParameter("No such name.");
      
      m_configurations[index].configuration_data[item].value = value;
    */
    return true;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ConfigurationSet ꥹȤμ 
   * @else
   * @brief [CORBA interface] Get a list of ConfigurationSet
   * @endif
   */
  ConfigurationSetList*
  Configuration_impl::get_configuration_sets()
    throw (CORBA::SystemException,
	   NotAvailable, InternalError)
  {
    RTC_TRACE(("get_configuration_sets()"));
    try
      {
	Guard guard(m_config_mutex);
	
	std::vector<coil::Properties*> cf(m_configsets.getConfigurationSets());
	ConfigurationSetList_var config_sets = 
          new ConfigurationSetList((CORBA::ULong)cf.size());
        // Ctor's first arg is max length. Actual length has to be set.
        config_sets->length((CORBA::ULong)cf.size());

	for (CORBA::ULong i(0), len(cf.size()); i < len; ++i)
	  {
	    toConfigurationSet(config_sets[i], *(cf[i]));
	  }
	
	return config_sets._retn();
      }
    catch (CORBA::SystemException& e)
      {
#ifndef ORB_IS_RTORB
        RTC_ERROR(("CORBA::SystemException cought: %s", e._name()));
#else
        RTC_ERROR(("CORBA::SystemException cought."));
#endif
	throw InternalError("Configuration::get_configuration_sets()");

      }
    catch (...)
      {
        RTC_ERROR(("Unknown exception cought."));
	throw InternalError("Configuration::get_configuration_sets()");
      }
    // never reach here
    return new ConfigurationSetList(0);
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ConfigurationSet μ
   * @else
   * @brief [CORBA interface] Get a ConfigurationSet
   * @endif
   */
  ConfigurationSet*
  Configuration_impl::get_configuration_set(const char* id)
    throw (CORBA::SystemException,
	   NotAvailable, InternalError)
  {
    RTC_TRACE(("get_configuration_set(%s)", id));
    if (std::string(id).empty()) throw InternalError("ID is empty");
    // Originally getConfigurationSet raises InvalidParameter according to the 
    // SDO specification. However, SDO's IDL lacks InvalidParameter.
    
    Guard guard(m_config_mutex);
    
    try
      {
        if (!m_configsets.haveConfig(id))
          {
            RTC_ERROR(("No such ConfigurationSet"));
            throw InvalidParameter("No such ConfigurationSet");
          }
      }
    catch(...)
      {
        RTC_ERROR(("Unknown exception"));
        throw InternalError("Unknown exception");
      }
    
    const coil::Properties& configset(m_configsets.getConfigurationSet(id));
    
    try
      {
	ConfigurationSet_var config;
	config = new ConfigurationSet();
	toConfigurationSet(config, configset);

	return config._retn();
      }
    catch (...)
      {
	throw InternalError("Configuration::get_configuration_set()");
      }

    // never reach here
    RTC_FATAL(("never reach here"));
    return new ConfigurationSet();
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ConfigurationSet 򥻥åȤ
   * @else
   * @brief [CORBA interface] Set ConfigurationSet
   * @endif
   */
  CORBA::Boolean
  Configuration_impl::
  set_configuration_set_values(const ConfigurationSet& configuration_set)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("set_configuration_set_values()"));
    std::string id(configuration_set.id);
    if (id.empty()) throw InvalidParameter("ID is empty.");
    
    try
      {
        coil::Properties conf(id.c_str());
	toProperties(conf, configuration_set);
        
	//------------------------------------------------------------
	// Because the format of port-name had been changed from
	// <port_name> to <instance_name>.<port_name>, the following
	// processing was added.  (since r1648)
        if (conf.findNode("exported_ports") != 0)
          {
            coil::vstring
              exported_ports(coil::split(conf["exported_ports"], ","));
            std::string exported_ports_str("");
            for (size_t i(0), len(exported_ports.size()); i < len; ++i)
              {
                coil::vstring keyval(coil::split(exported_ports[i], "."));
                if (keyval.size() > 2)
                  {
                    exported_ports_str += (keyval[0] + "." + keyval.back());
                  }
                else
                  {
                    exported_ports_str += exported_ports[i];
                  }
                if (i != exported_ports.size() - 1)
                  {
                    exported_ports_str += ",";
                  }
              }
            conf["exported_ports"] = exported_ports_str;
          }
	//------------------------------------------------------------

	return m_configsets.setConfigurationSetValues(conf);
      }
    catch (...)
      {
	throw InternalError("Configuration::set_configuration_set_values()");
      }
    return true;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ƥ֤ ConfigurationSet 
   * @else
   * @brief [CORBA interface] Get active ConfigurationSet
   * @endif
   */ 
  ConfigurationSet*
  Configuration_impl::get_active_configuration_set()
    throw (CORBA::SystemException,
	   NotAvailable, InternalError)
  {
    RTC_TRACE(("get_active_configuration_set()"));
    // activeConfigurationSet̵
    if (!m_configsets.isActive()) throw NotAvailable();    
    
    try
      {
	Guard gurad(m_config_mutex);
	// activeConfigurationSet֤
	ConfigurationSet_var config;
	config = new ConfigurationSet();
	toConfigurationSet(config, m_configsets.getActiveConfigurationSet());
	return config._retn();
      }
    catch (...)
      {
	throw InternalError("Configuration::get_active_configuration_set()");
      }
    // never reach here
    return new ConfigurationSet();
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ConfigurationSet ɲä
   * @else
   * @brief [CORBA interface] Add ConfigurationSet
   * @endif
   */
  CORBA::Boolean
  Configuration_impl::
  add_configuration_set(const ConfigurationSet& configuration_set)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("add_configuration_set()"));
    try
      {
	Guard gurad(m_config_mutex);
	const char* config_id(configuration_set.id);
//	const char* config_value(configuration_set.description);
//	RTC::Properties config(config_id);
	coil::Properties config(config_id);
//	coil::Properties config(config_id,config_value);
	toProperties(config, configuration_set);
//        config["description"] = configuration_set.description;
	return m_configsets.addConfigurationSet(config);
      }
    catch (...)
      {
	throw InternalError("Configuration::add_configuration_set()");
	return false;
      }
    return true;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ConfigurationSet 
   * @else
   * @brief [CORBA interface] Remove ConfigurationSet
   * @endif
   */
  CORBA::Boolean
  Configuration_impl::remove_configuration_set(const char* id)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("remove_configuration_set(%s)", id));
    if (std::string(id).empty())
      throw InvalidParameter("ID is empty.");
    
    try
      {
	Guard guard(m_config_mutex);
	return m_configsets.removeConfigurationSet(id);
      }
    catch (...)
      {
	throw InternalError("Configuration::remove_configuration_set()");
	return false;
      }
    return false;
  }
  
  /*!
   * @if jp
   * @brief [CORBA interface] ConfigurationSet Υƥֲ
   * @else
   * @brief [CORBA interface] Activate ConfigurationSet
   * @endif
   */ 
  CORBA::Boolean
  Configuration_impl::activate_configuration_set(const char* id)
    throw (CORBA::SystemException,
	   InvalidParameter, NotAvailable, InternalError)
  {
    RTC_TRACE(("activate_configuration_set(%s)", id));
    if (std::string(id).empty())
      throw InvalidParameter("ID is empty.");
    
    if(m_configsets.activateConfigurationSet(id))
      {
        return true;
      }
    else
      {
	throw InvalidParameter("Configuration::activate_configuration_set()");
      }
/*
    try
      {
	return m_configsets.activateConfigurationSet(id);
      }
    catch (...)
      {
	throw InternalError("Configuration::activate_configuration_set()");
	return false;
      }
*/
    return false;
  }
  
  //============================================================
  // Local interfaces
  //============================================================
  
  /*!
   * @if jp
   * @brief ֥ȡե󥹤
   * @else
   * @brief Get object reference
   * @endif
   */
  Configuration_ptr Configuration_impl::getObjRef()
  {
    return m_objref;
  }
  
  /*!
   * @if jp
   * @brief SDO  DeviceProfile 
   * @else
   * @brief Get the DeviceProfile of SDO
   * @endif
   */
  const DeviceProfile Configuration_impl::getDeviceProfile()
  {
    return m_deviceProfile;
  }
  
  /*!
   * @if jp
   * @brief SDO  Organization ꥹȤ
   * @else
   * @brief Get a list of Organization of SDO
   * @endif
   */
  const OrganizationList Configuration_impl::getOrganizations()
  {
    return m_organizations;
  }
  
  /*!
   * @if jp
   * @brief UUID
   * @else
   * @brief Generate UUID
   * @endif
   */
  const std::string Configuration_impl::getUUID() const
  {
    coil::UUID_Generator uugen;
    uugen.init();
    std::auto_ptr<coil::UUID> uuid(uugen.generateUUID(2,0x01));
    
    return (const char*) uuid->to_string();
  }
};
