//
// Copyright 2020-2021 Picovoice Inc.
//
// You may not use this file except in compliance with the license. A copy of the license is located in the "LICENSE"
// file accompanying this source.
//
// Unless required by applicable law or agreed to in writing, software distributed under the License is distributed on
// an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the License for the
// specific language governing permissions and limitations under the License.
//
"use strict";
const path = require("path");

const { PvArgumentError } = require("./errors");
const { getPlatform } = require("./platforms");

const ALEXA = 0;
const AMERICANO = 1;
const BLUEBERRY = 2;
const BUMBLEBEE = 3;
const COMPUTER = 4;
const GRAPEFRUIT = 5;
const GRASSHOPPER = 6;
const HEY_GOOGLE = 7;
const HEY_SIRI = 8;
const JARVIS = 9;
const OK_GOOGLE = 10;
const PICOVOICE = 11;
const PORCUPINE = 12;
const TERMINATOR = 13;

const ALEXA_STRING = "alexa";
const AMERICANO_STRING = "americano";
const BLUEBERRY_STRING = "blueberry";
const BUMBLEBEE_STRING = "bumblebee";
const COMPUTER_STRING = "computer";
const GRAPEFRUIT_STRING = "grapefruit";
const GRASSHOPPER_STRING = "grasshopper";
const HEY_GOOGLE_STRING = "hey google";
const HEY_SIRI_STRING = "hey siri";
const JARVIS_STRING = "jarvis";
const OK_GOOGLE_STRING = "ok google";
const PICOVOICE_STRING = "picovoice";
const PORCUPINE_STRING = "porcupine";
const TERMINATOR_STRING = "terminator";

const BUILTIN_KEYWORDS_ENUMS = new Set();
BUILTIN_KEYWORDS_ENUMS.add(ALEXA);
BUILTIN_KEYWORDS_ENUMS.add(AMERICANO);
BUILTIN_KEYWORDS_ENUMS.add(BLUEBERRY);
BUILTIN_KEYWORDS_ENUMS.add(BUMBLEBEE);
BUILTIN_KEYWORDS_ENUMS.add(COMPUTER);
BUILTIN_KEYWORDS_ENUMS.add(GRAPEFRUIT);
BUILTIN_KEYWORDS_ENUMS.add(GRASSHOPPER);
BUILTIN_KEYWORDS_ENUMS.add(HEY_GOOGLE);
BUILTIN_KEYWORDS_ENUMS.add(HEY_SIRI);
BUILTIN_KEYWORDS_ENUMS.add(JARVIS);
BUILTIN_KEYWORDS_ENUMS.add(OK_GOOGLE);
BUILTIN_KEYWORDS_ENUMS.add(PICOVOICE);
BUILTIN_KEYWORDS_ENUMS.add(PORCUPINE);
BUILTIN_KEYWORDS_ENUMS.add(TERMINATOR);

const BUILTIN_KEYWORDS_STRINGS = new Set();
BUILTIN_KEYWORDS_STRINGS.add(ALEXA_STRING);
BUILTIN_KEYWORDS_STRINGS.add(AMERICANO_STRING);
BUILTIN_KEYWORDS_STRINGS.add(BLUEBERRY_STRING);
BUILTIN_KEYWORDS_STRINGS.add(BUMBLEBEE_STRING);
BUILTIN_KEYWORDS_STRINGS.add(COMPUTER_STRING);
BUILTIN_KEYWORDS_STRINGS.add(GRAPEFRUIT_STRING);
BUILTIN_KEYWORDS_STRINGS.add(GRASSHOPPER_STRING);
BUILTIN_KEYWORDS_STRINGS.add(HEY_GOOGLE_STRING);
BUILTIN_KEYWORDS_STRINGS.add(HEY_SIRI_STRING);
BUILTIN_KEYWORDS_STRINGS.add(JARVIS_STRING);
BUILTIN_KEYWORDS_STRINGS.add(OK_GOOGLE_STRING);
BUILTIN_KEYWORDS_STRINGS.add(PICOVOICE_STRING);
BUILTIN_KEYWORDS_STRINGS.add(PORCUPINE_STRING);
BUILTIN_KEYWORDS_STRINGS.add(TERMINATOR_STRING);

const BUILTIN_KEYWORDS_ENUM_TO_STRING = new Map();
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(ALEXA, ALEXA_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(AMERICANO, AMERICANO_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(BLUEBERRY, BLUEBERRY_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(BUMBLEBEE, BUMBLEBEE_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(COMPUTER, COMPUTER_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(GRAPEFRUIT, GRAPEFRUIT_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(GRASSHOPPER, GRASSHOPPER_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(HEY_GOOGLE, HEY_GOOGLE_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(HEY_SIRI, HEY_SIRI_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(JARVIS, JARVIS_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(OK_GOOGLE, OK_GOOGLE_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(PICOVOICE, PICOVOICE_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(PORCUPINE, PORCUPINE_STRING);
BUILTIN_KEYWORDS_ENUM_TO_STRING.set(TERMINATOR, TERMINATOR_STRING);

const BUILTIN_KEYWORDS_STRING_TO_ENUM = new Map();
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(ALEXA_STRING, ALEXA);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(AMERICANO_STRING, AMERICANO);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(BLUEBERRY_STRING, BLUEBERRY);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(BUMBLEBEE_STRING, BUMBLEBEE);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(COMPUTER_STRING, COMPUTER);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(GRAPEFRUIT_STRING, GRAPEFRUIT);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(GRASSHOPPER_STRING, GRASSHOPPER);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(HEY_GOOGLE_STRING, HEY_GOOGLE);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(HEY_SIRI_STRING, HEY_SIRI);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(JARVIS_STRING, JARVIS);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(OK_GOOGLE_STRING, OK_GOOGLE);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(PICOVOICE_STRING, PICOVOICE);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(PORCUPINE_STRING, PORCUPINE);
BUILTIN_KEYWORDS_STRING_TO_ENUM.set(TERMINATOR_STRING, TERMINATOR);

function getBuiltinKeywordPath(builtinKeyword) {
  if (!Number.isInteger(builtinKeyword)) {
    throw new PvArgumentError(
      `getBuiltinKeywordPath argument '${builtinKeyword}' is not an integer (enum) value`
    );
  }

  let platform = getPlatform();
  let keywordString;
  if (!BUILTIN_KEYWORDS_ENUMS.has(builtinKeyword)) {
    throw new PvArgumentError(
      `Keyword argument ${builtinKeyword} does not map to one of the built-in keywords: [${Array.from(
        BUILTIN_KEYWORDS_STRINGS
      )}]`
    );
  } else {
    keywordString = BUILTIN_KEYWORDS_ENUM_TO_STRING.get(builtinKeyword);
  }

  return path.resolve(
    __dirname,
    `resources/keyword_files/${platform}/${keywordString}_${platform}.ppn`
  );
}

module.exports = {
  ALEXA: ALEXA,
  AMERICANO: AMERICANO,
  BLUEBERRY: BLUEBERRY,
  BUMBLEBEE: BUMBLEBEE,
  COMPUTER: COMPUTER,
  GRAPEFRUIT: GRAPEFRUIT,
  GRASSHOPPER: GRASSHOPPER,
  HEY_GOOGLE: HEY_GOOGLE,
  HEY_SIRI: HEY_SIRI,
  JARVIS: JARVIS,
  OK_GOOGLE: OK_GOOGLE,
  PICOVOICE: PICOVOICE,
  PORCUPINE: PORCUPINE,
  TERMINATOR: TERMINATOR,
  BUILTIN_KEYWORDS_ENUMS: BUILTIN_KEYWORDS_ENUMS,
  BUILTIN_KEYWORDS_STRINGS: BUILTIN_KEYWORDS_STRINGS,
  BUILTIN_KEYWORDS_ENUM_TO_STRING: BUILTIN_KEYWORDS_ENUM_TO_STRING,
  BUILTIN_KEYWORDS_STRING_TO_ENUM: BUILTIN_KEYWORDS_STRING_TO_ENUM,
  getBuiltinKeywordPath: getBuiltinKeywordPath,
};
