# Porcupine Binding for Unity

## Porcupine

Porcupine is a highly accurate and lightweight wake word engine. It enables building always-listening voice-enabled applications using cutting edge voice AI.

Porcupine is:

- private and offline
- [accurate](https://github.com/Picovoice/wake-word-benchmark)
- [resource efficient](https://www.youtube.com/watch?v=T0tAnh8tUQg) (runs even on microcontrollers)
- data efficient (wake words can be easily generated by simply typing them, without needing thousands of hours of bespoke audio training data and manual effort)
- scalable to many simultaneous wake-words / always-on voice commands
- cross-platform

To learn more about Porcupine, see the [product](https://picovoice.ai/products/porcupine/), [documentation](https://picovoice.ai/docs/), and [GitHub](https://github.com/Picovoice/porcupine/) pages.

### Custom wake words

Porcupine includes several built-in keywords, which are stored as `.ppn` files. To train custom PPN files, see the [Picovoice Console](https://picovoice.ai/console/).

Unlike the built-in keywords, custom PPN files generated with the Picovoice Console carry restrictions including (but not limited to): training allowance, time limits, available platforms, and commercial usage.

## Compatibility

[Porcupine unity package](./porcupine-2.1.0.unitypackage) is for running Porcupine on **Unity 2017.4+** on the following platforms:

- Android 4.4+ (API 19+) (ARM only)
- iOS 10.0+
- Windows (x86_64)
- macOS (x86_64)
- Linux (x86_64)

For running Porcupine on **macOS M1 (arm64)**, use the [Apple silicon](./porcupine-2.1.0-Apple-silicon.unitypackage) version on **Unity 2021.2+**.

## Installation

The easiest way to install the Porcupine Unity SDK is to import [porcupine-2.0.0.unitypackage](/binding/unity/porcupine-2.0.0.unitypackage) into your Unity projects by either dropping it into the Unity editor or going to _Assets>Import Package>Custom Package..._

**NOTE:** On macOS, the Porcupine library may get flagged as having come from an unverified source if you've downloaded the  `.unitypackage` directly from GitHub. This should only come up when running your project in the Editor. To disable this warning, go to Security & Preferences and choose to allow pv_porcupine.dylib to run.

## AccessKey

All bindings require a valid Picovoice `AccessKey` at initialization. `AccessKey`s act as your credentials when using Porcupine SDKs.
You can create your `AccessKey` for free. Make sure to keep your `AccessKey` secret.

To obtain your `AccessKey`:
1. Login or Signup for a free account on the [Picovoice Console](https://picovoice.ai/console/).
2. Once logged in, go to the [`AccessKey` tab](https://console.picovoice.ai/access_key) to create one or use an existing `AccessKey`.

## Packaging
To build the package from source, first you have to clone the repo with submodules:
```console
git clone --recurse-submodules git@github.com:Picovoice/porcupine.git
# or 
git clone --recurse-submodules https://github.com/Picovoice/porcupine.git
```

You then have to run the `copy.sh` file to copy the package resources from various locations in the repo to the Unity project located at [/binding/unity](/binding/unity) (**NOTE:** on Windows, Git Bash or another bash shell is required, or you will have to manually copy the resources into the project.). Then, open the Unity project, right-click the Assets folder and select Export Package. The resulting Unity package can be imported into other Unity projects as desired.

## Usage

The module provides you with two levels of API to choose from depending on your needs.

#### High-Level API

[PorcupineManager](/binding/unity/Assets/Porcupine/PorcupineManager.cs) provides a high-level API that takes care of audio recording. This class is the quickest way to get started.

>**NOTE:** If running on iOS, you must fill in the Microphone Usage Description under Project Settings>Other Settings in order to enable audio recording.

Using the constructor `PorcupineManager.FromBuiltInKeywords` will create an instance of the PorcupineManager using one or more of the built-in keywords.
```csharp
using Pv.Unity;

string accessKey = "${ACCESS_KEY}"; // // AccessKey obtained from Picovoice Console (https://picovoice.ai/console/)

try {
    List<Porcupine.BuiltInKeywords> keywords = new List<BuiltInKeywords>(){ Porcupine.BuiltInKeyword.PICOVOICE, Porcupine.BuiltInKeyword.PORCUPINE };
    PorcupineManager _porcupineManager = PorcupineManager.FromBuiltInKeywords(
                                            accessKey,
                                            keywords,
                                            OnWakeWordDetected);
}
catch (Exception ex)
{
    // handle porcupine init error
}
```
The `wakeWordCallback` parameter is a function that you want to execute when Porcupine has detected one of the keywords.
The function should accept a single integer, keywordIndex, which specifies which wake word has been detected.

```csharp
private void OnWakeWordDetected(int keywordIndex){
    if(keywordIndex == 0){
        // picovoice detected
    }
    else if (keywordIndex === 1){
        // porcupine detected
    }
}
```

Available built-in keywords are stored in the constants `Porcupine.BuiltInKeyword`.

To create an instance of PorcupineManager that detects custom keywords, you can use the `PorcupineManager.FromKeywordPaths`
static constructor and provide the paths to the `.ppn` file(s).
```csharp
string accessKey = "${ACCESS_KEY}"; // // AccessKey obtained from Picovoice Console (https://picovoice.ai/console/)

List<string> keywordPaths = new List<string>(){ "/path/to/keyword.ppn" };
PorcupineManager _porcupineManager = PorcupineManager.FromKeywordPaths( 
                                        accessKey,
                                        keywordPaths, 
                                        OnWakeWordDetected);
```

In addition to custom keywords, you can override the default Porcupine model file and/or keyword sensitivities, as well as add an error callback you want to trigger if there's a problem encountered while Porcupine is processing frames.

These optional parameters can be passed in like so:
```csharp
string accessKey = "${ACCESS_KEY}"; // // AccessKey obtained from Picovoice Console (https://picovoice.ai/console/)

List<string> keywordPaths = new List<string>()
{ 
    "/path/to/keyword/file/one.ppn", 
    "/path/to/keyword/file/two.ppn"
};
string modelPath = "path/to/model/file.pv";
List<float> sensitivites = new List<float>(){ 0.25f, 0.6f };

PorcupineManager _porcupineManager = PorcupineManager.FromKeywordPaths(
                                        accessKey,
                                        keywordPaths,
                                        OnWakeWordDetected,
                                        modelPath: modelPath,
                                        sensitivities: sensitivities,
                                        errorCallback: OnError);

void OnError(Exception ex){
    Debug.LogError(ex.ToString());
}
```

Once you have instantiated a PorcupineManager, you can start audio capture and wake word detection by calling:

```csharp
_porcupineManager.Start();
```

And then stop it by calling:

```csharp
_porcupineManager.Stop();
```

Once the app is done with using an instance of PorcupineManager, you can explicitly release the audio resources and the resources allocated to Porcupine:
```csharp
_porcupineManager.Delete();
```

There is no need to deal with audio capture to enable wake word detection with PorcupineManager.
This is because it uses our
[unity-voice-processor](https://github.com/Picovoice/unity-voice-processor/)
Unity package to capture frames of audio and automatically pass it to the wake word engine.

#### Low-Level API

[Porcupine](/binding/unity/Assets/Porcupine/Porcupine.cs) provides low-level access to the wake word engine for those who want to incorporate wake word detection into an already existing audio processing pipeline.

To create an instance of `Porcupine`, use the `.Create` static constructor. You can pass a list of built-in keywords as its `keywords` argument or a list or paths to custom keywords using its `keywordPaths` argument. 

```csharp
using Pv.Unity;

string accessKey = "${ACCESS_KEY}"; // // AccessKey obtained from Picovoice Console (https://picovoice.ai/console/)

try
{
    List<Porcupine.BuiltInKeyword> keywords = new List<Porcupine.BuiltInKeyword>(){ Porcupine.BuiltInKeyword.PORCUPINE, Porcupine.BuiltInKeyword.PICOVOICE };
    Porcupine _porcupine = Porcupine.FromBuiltInKeywords(accessKey: accessKey, keywords: keywords);
} 
catch (Exception ex) 
{
    // handle porcupine init error
}
```

To search for a keyword in audio, you must pass frames of audio to Porcupine using the `Process` function. The `keywordIndex` returned will either be -1 if no detection was made or an integer specifying which keyword was detected.

```csharp
short[] frame = getAudioFrame();

try 
{
    int keywordIndex = _porcupine.Process(frame);
    if (keywordIndex >= 0) 
    {
        // detection made!
    }
}
catch (Exception ex)
{
    Debug.LogError(ex.ToString());
}  
```

For process to work correctly, the audio data must be in the audio format required by Picovoice.
The required sample rate is specified by the `SampleRate` property and the required number of audio samples in each frame is specified by the `FrameLength` property. Audio must be single-channel and 16-bit linearly-encoded.

Porcupine implements the `IDisposable` interface, so you can use Porcupine in a `using` block. If you don't use a `using` block, resources will be released by the garbage collector automatically, or you can explicitly release the resources like so:

```csharp
_porcupine.Dispose();
```

## Custom Wake Word Integration

To add a custom wake word to your Unity app, you'll need to add it to your project root under `/StreamingAssets`. Then, in a script, retrieve them like so:
```csharp
string keywordPath = Path.Combine(Application.streamingAssetsPath, "keyword.ppn");
```

## Non-English Wake Words

In order to detect non-English wake words you need to use the corresponding model file. The model files for all supported languages are available [here](/lib/common).

## Demo

The Porcupine Unity demo can be imported along with the SDK when you import the Porcupine Unity package. Browse the source of the demo [here](/demo/unity).