/***
 *  Software License Agreement: BSD 3-Clause License
 *
 *  Copyright (c) 2016-2022, qbrobotics®
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
 *  following conditions are met:
 *
 *  * Redistributions of source code must retain the above copyright notice, this list of conditions and the
 *    following disclaimer.
 *
 *  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other materials provided with the distribution.
 *
 *  * Neither the name of the copyright holder nor the names of its contributors may be used to endorse or promote
 *    products derived from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 *  INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 *  DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 *  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 *  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef QB_SOFTHAND_INDUSTRY_COMMUNICATION_HANDLER_H
#define QB_SOFTHAND_INDUSTRY_COMMUNICATION_HANDLER_H

// Standard libraries
#include <mutex>
#include <regex>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

// ROS libraries
#include <ros/ros.h>

// internal libraries
#include <qb_softhand_industry_msgs/ConnectionState.h>
#include <qb_softhand_industry_srvs/qb_softhand_industry_srvs.h>


// qbSoftHand Industry macros
#define QBSOFTHAND_INDUSTRY_ERROR_BADPARAMS -1
#define QBSOFTHAND_INDUSTRY_ERROR_RECVAGAIN -2
#define QBSOFTHAND_INDUSTRY_ERROR_NOTRESPONDING -3
#define QBSOFTHAND_INDUSTRY_ERROR_NOTCOMPATIBLE -4

#define QBSOFTHAND_INDUSTRY_GET_CURRENT_CMD "IQ"
#define QBSOFTHAND_INDUSTRY_GET_POSITION_CMD "PU"
#define QBSOFTHAND_INDUSTRY_GET_VELOCITY_CMD "VU"

#define QBSOFTHAND_INDUSTRY_POS_MIN 0
#define QBSOFTHAND_INDUSTRY_POS_MAX 3800
#define QBSOFTHAND_INDUSTRY_VEL_MIN 400
#define QBSOFTHAND_INDUSTRY_VEL_MAX 3200
#define QBSOFTHAND_INDUSTRY_CURR_MIN 250
#define QBSOFTHAND_INDUSTRY_CURR_MAX 450

namespace qb_softhand_industry_communication_handler {
/**
 * The Communication Handler class is aimed to instantiate a ROS node which provides several ROS services to
 * communicate with one - or many - qbrobotics SoftHand Industry connected to the ROS ecosystem.
 */
class qbSoftHandIndustryCommunicationHandler {
  public:
    qbSoftHandIndustryCommunicationHandler();

    virtual ~qbSoftHandIndustryCommunicationHandler();

  protected:
    ros::NodeHandle node_handle_;
    std::string ip_;

    // socket variables
    std::unique_ptr<std::mutex> socket_protector_;
    int socket_descriptor_;
    sockaddr_in socket_address_;
    socklen_t address_length_;

    /**
     * Activate the motors of the device. Do nothing if the device is not connected in the Communication Handler.
     * \param max_repeats The maximum number of consecutive repetitions to mark retrieved data as corrupted.
     * \sa activateCallback(), activate(const int &, const bool &, const int &), isActive()
     */
    virtual int activate(const bool &command, const int &max_repeats);

    /**
     * Activate the motors of the SoftHand Industry relative to the node requesting the service.
     * \param request The request of the given service (see qb_softhand_industry_srvs::Trigger for details).
     * \param response The response of the given service (see qb_softhand_industry_srvs::Trigger for details).
     * \return \p true if the call succeed (actually \p response.success may be false).
     * \sa activate(const int &, const int &)
     */
    bool activateCallback(qb_softhand_industry_srvs::TriggerRequest &request, qb_softhand_industry_srvs::TriggerResponse &response);

    /**
     * Deactivate the motors of the SoftHand Industry relative to the node requesting the service.
     * \param request The request of the given service (see qb_softhand_industry_srvs::Trigger for details).
     * \param response The response of the given service (see qb_softhand_industry_srvs::Trigger for details).
     * \return \p true if the call succeed (actually \p response.success may be false).
     * \sa activate(const int &, const int &)
     */
    bool deactivateCallback(qb_softhand_industry_srvs::TriggerRequest &request, qb_softhand_industry_srvs::TriggerResponse &response);

    /**
     * @brief Get measurements from SoftHand Industry
     * 
     * @param command the measurements to be retrieved command
     * @param max_repeats the attempts to be performed
     * @return the measurements 
     */
    float getMeasurement(const std::string &measurement_command, const int &max_repeats);

    /**
     * Get measurements from SoftHand Industry relative to the node requesting the service.
     * \param request The request of the given service (see qb_softhand_industry_srvs::GetMeasurements for details).
     * \param response The response of the given service (see qb_softhand_industry_srvs::GetMeasurements for details).
     * \return \p true if the call succeed (actually \p response.success may be false).
     */
    bool getMeasurementsCallback(qb_softhand_industry_srvs::GetMeasurementsRequest &request, qb_softhand_industry_srvs::GetMeasurementsResponse &response);

    /**
     * Initialize socket to connect to the SoftHand Industry and verify the basic requirements.
     */
    int init();

    /**
     * Initialize socket to connect to the SoftHand Industry relative to the node requesting the service.
     */
    int initSocket();

    /**
     * @brief Validate current
     * @param current parameter to validate
     * 
     * @return true if valid, false otherwise
     */
    bool isValidCurrent(const int &current);  

    /**
     * @brief Validate position
     * @param position parameter to validate
     * 
     * @return true if valid, false otherwise
     */
    bool isValidPosition(const int &position);

    /**
     * @brief Validate velocity
     * @param velocity parameter to validate
     * 
     * @return true if valid, false otherwise
     */
    bool isValidVelocity(const int &velocity);

    /**
     * @brief Read N bytes into BUF through socket FD.
     * 
     * @param buffer 
     * @return Returns the number of bytes read or -1 for errors. 
     */
    ssize_t recvfrom(std::string &buffer);

    /**
     * @brief Send a movement command to SoftHand Industry
     * 
     * @param command the position reference command [ticks]
     * @return the response status  
     */
    int setCommand(const int &command, const int &max_repeats);

    /**
     * @brief Send a movement command to SoftHand Industry
     * 
     * @param command the command [ticks]
     * @param velocity [ticks/s]
     * @param current [mA]
     * @return the response status  
     */
    int setCommands(const int &position, const int &velocity, const int &current, const int &max_repeats);
    /**
     * Send a command to SoftHand Industry.
     * \param request The request of the given service (see qb_softhand_industry_srvs::setCommand for details).
     * \param response The response of the given service (see qb_softhand_industry_srvs::setCommand for details).
     * \return \p true if the call succeed (actually \p response.success may be false).
     */
    bool setCommandCallback(qb_softhand_industry_srvs::SetCommandRequest &request, qb_softhand_industry_srvs::SetCommandResponse &response);

    /**
     * Send a command to SoftHand Industry.
     * \param request The request of the given service (see qb_softhand_industry_srvs::setCommands for details).
     * \param response The response of the given service (see qb_softhand_industry_srvs::setCommands for details).
     * \return \p true if the call succeed (actually \p response.success may be false).
     */
    bool setCommandsCallback(qb_softhand_industry_srvs::SetCommandsRequest &request, qb_softhand_industry_srvs::SetCommandsResponse &response);

    /**
     * @brief Send N bytes of BUF on socket FD to peer at address ADDR (which is ADDR_LEN bytes long). 
     * 
     * @return Returns the number sent, or -1 for errors. 
     */
    ssize_t sendto(const std::string &buffer);

    /**
     * @brief Send a command to qbSoftHand Industry
     * 
     * @param command the command to be exectuted
     * @return the response status
     */
    int sendAndParseResponse(const std::string &command);

    /**
     * @brief Send a command to qbSoftHand Industry and parse the response
     * 
     * @param command the command to be exectuted
     * @param parsed_response the parsed response
     * @return the response status
     */
    int sendAndParseResponse(const std::string &command, float &parsed_response);

    /**
     * @brief Send a command to qbSoftHand Industry and parse the response for \p max_repeats attempts
     * 
     * @param command the command to be exectuted
     * @param max_repeats the attempts to be performed
     * @param parsed_response the parsed response
     * @return the response status 
     */
    int sendAndParseResponse(const std::string &command, const int &max_repeats, float &parsed_response);

    /**
     * @brief Send a command to qbSoftHand Industry for \p max_repeats attempts
     * 
     * @param command the command to be exectuted
     * @param max_repeats the attempts to be performed
     * @return the response status 
     */
    int sendAndParseResponse(const std::string &command, const int &max_repeats);

    /**
     * @brief parse the response of the \p command
     * 
     * @param command the command whose response has to be parsed
     * @param response the response
     * @param parsed_response the parsed response
     * @return the response status  
     */
    int parseResponse(const std::string &command, const std::string &response, float &parsed_response);    

  private:
    ros::ServiceServer activate_motors_;
    ros::ServiceServer deactivate_motors_;
    ros::ServiceServer set_command_;
    ros::ServiceServer set_commands_;
    ros::ServiceServer get_measurements_;
};
}

#endif //QB_SOFTHAND_INDUSTRY_COMMUNICATION_HANDLER_H