import argparse
import numpy as np
import matplotlib.pyplot as plt
from pprint import pprint
from reach import ReachDatabase, load
from scipy.interpolate import Rbf


def evaluate(positions, scores, func, eps=None):
    rbf = Rbf(positions[:, 0], positions[:, 1], positions[:, 2], scores, function=func, epsilon=eps)
    Ainv = np.linalg.inv(rbf.A)
    e = rbf.nodes / np.diag(Ainv)
    return np.linalg.norm(e, ord=1)/len(e)


def main():
    parser = argparse.ArgumentParser(
        description="Calculate and display the errors generated by varying parametrizations of RBF interpolation")
    parser.add_argument(type=str, dest="db_file", help="Filepath of the reach database")
    parser.add_argument('--max_eps', type=float, default=1.0, help="Maximum value of epsilon to sample")
    parser.add_argument('--n_eps_samples', type=int, default=50, help="Number of epsilon values to sample on [5e-3, max_eps]")
    args = parser.parse_args()

    # Load database from ROS message
    db = load(args.db_file)
    result = db.results[-1]

    positions = np.array([r.goal()[0:3, 3] for r in result])
    scores = np.array([r.score for r in result])

    # Evaluate the functions that use the epsilon parameter over a range of values
    epsilon = np.linspace(0.0005, args.max_eps, args.n_eps_samples)
    functions_eps = ['multiquadric', 'inverse', 'gaussian']
    error_eps = np.empty((len(functions_eps), len(epsilon)))
    for i, func in enumerate(functions_eps):
        for j, eps in enumerate(epsilon):
            error_eps[i, j] = evaluate(positions, scores, func, eps)

    # Evaluate the functions that do not use epsilon
    functions_no_eps = ['thin_plate', 'linear', 'cubic', 'quintic']
    error_no_eps = np.empty((len(functions_no_eps), len(epsilon)))
    for i, func in enumerate(functions_no_eps):
        error_no_eps[i] = np.repeat(evaluate(positions, scores, func), len(epsilon))

    # Combine the results
    functions = functions_eps + functions_no_eps
    error = np.vstack([error_eps, error_no_eps])

    # Print the error for each method and its corresponding epsilon value
    print('Method - Error - Epsilon')
    pprint(list(zip(functions, zip(np.min(error, axis=1), epsilon[np.argmin(error, axis=1)]))))

    # Plot the results
    _,ax = plt.subplots()
    lines = ax.plot(epsilon, error.T)
    for i, line in enumerate(lines):
        line.set_label(functions[i])
    ax.scatter(epsilon[np.argmin(error, axis=1)], np.min(error, axis=1))
    ax.set(
        xlabel='Epsilon',
        ylabel='Error'
    )
    ax.grid()
    ax.legend()
    plt.show()


if __name__ == '__main__':
    main()
