//======================================================================
/*!
  \file
  \section sdhlibrary_cpp_rs232_vcc_h_general General file information
    \author   Martin
    \date     2008-05-23


  \brief
    Implementation of class #SDH::cRS232, a class to access serial RS232 port with VCC compiler on Windows.

  \section sdhlibrary_cpp_rs232_vcc_h_cpp_copyright Copyright
    Code kindly provided by Martin from the RoboCluster project Denmark.

  <HR>
  \internal

    \subsection sdhlibrary_cpp_rs232_vcc_h_details SVN related, detailed file specific information:
      $LastChangedBy: Osswald2 $
      $LastChangedDate: 2011-03-09 11:55:11 +0100 (Mi, 09 Mrz 2011) $
      \par SVN file revision:
        $Id: rs232-vcc.h 6526 2011-03-09 10:55:11Z Osswald2 $

  \subsection sdhlibrary_cpp_rs232_vcc_h_changelog Changelog of this file:
      \include rs232-vcc.h.log
*/
//======================================================================

#ifndef RS232_VCC_h
#define RS232_VCC_h

#include "sdhlibrary_settings.h"

#if SDH_USE_VCC
# pragma warning(disable : 4996)
# pragma warning(disable : 4290)
# pragma once
#endif

//----------------------------------------------------------------------
// System Includes - include with <>
//----------------------------------------------------------------------

#include <windows.h>

//----------------------------------------------------------------------
// Project Includes - include with ""
//----------------------------------------------------------------------

#include "sdhexception.h"
#include "serialbase.h"
#include "sdhlibrary_settings.h"


//#define RS_232_TEST
#define SDH_RS232_VCC_ASYNC 0

NAMESPACE_SDH_START

class VCC_EXPORT cRS232Exception: public cSerialBaseException
{
public:
    cRS232Exception( cMsg const & _msg )
    : cSerialBaseException( "cRS232Exception", _msg )
    {}
};

/*!
\brief Low-level communication class to access a serial port on VCC Windows
*/
class VCC_EXPORT cRS232 : public cSerialBase
{
private:
#ifndef RS_232_TEST
    HANDLE       _hCOM;
#endif
#if SDH_RS232_VCC_ASYNC
    OVERLAPPED   o;
#endif
    COMMTIMEOUTS comm_timeouts;
    long         read_timeout_us; // for caching the read timeout in ms

protected:
    //! the RS232 port number to use (port 0 is COM1)
    int port;

    //! the baudrate in bit/s
    unsigned long baudrate;


public:
    /*!
    * Constructor: constructs an object to communicate with an %SDH via RS232

    * \param _port     - rs232 device number: 0='COM1'='/dev/ttyS0', 1='COM2'='/dev/ttyS1', ...
    * \param _baudrate - the baudrate in bit/s
    * \param _timeout  - the timeout in seconds
    * \param _device_format_string - ignored for this VCC version
    */
    cRS232( int _port, unsigned long _baudrate, double _timeout, char const* _device_format_string = "" );
    ~cRS232(void);

    void Open( void );
    void Close( void );

    virtual void SetTimeout( double _timeout );

#ifndef RS_232_TEST
    bool IsOpen()
    throw()
    { return _hCOM != NULL; }
#else
    bool IsOpen() { return true; }
#endif
    int write( char const *ptr, int len=0 );

    /*!
    * Read data from device. This function waits until \a max_time_us us passed or
    * the expected number of bytes are received via serial line.
    * if (\a return_on_less_data is true (default value), the number of bytes
    * that have been received are returned and the data is stored in \a data
    * If the \a return_on_less_data is false, data is only read from serial line, if at least
    * \a size bytes are available.
    */
    ssize_t Read( void *data, ssize_t size, long timeout_us, bool return_on_less_data );

    char* readline(char* line, int size, char* eol, bool return_on_less_data);

    //! overloaded from cSerialBase::UseCRC16 since we want to use a CRC16 to protect binary RS232 communication
    virtual bool UseCRC16()
    {
        return true;
    }
};

NAMESPACE_SDH_END

#endif
