/** **************************************************************************************
 * @file       sensor_lib.h
 * @ingroup    Toposens Sensor Library
 * @copyright  Copyright (c) Toposens GmbH 2021. All rights reserved.
 * @brief      This file contains the highlevel interface prototypes to the low-level Protocol.
 ******************************************************************************************/

#ifndef TOPOSENS_SENSOR_LIB_H /* include control, avoid multiple includes of header file */
#define TOPOSENS_SENSOR_LIB_H

#ifdef __cplusplus
extern "C"
{
#endif

/*---- "library includes" -------------------------------------------------------------*/

/*---- "project includes" -------------------------------------------------------------*/
#include "config/sensor_lib_config.h"
#include "custom_structs.h"
#include "custom_structs_internal.h"
#include "custom_types.h"
#include "custom_types_internal.h"
#include "message_flags.h"
#include "toposens/errors.h"
/*---- <system includes> --------------------------------------------------------------*/
#include <stdbool.h>
#include <stdint.h>

/*---- public macros and constants ----------------------------------------------------*/
#define UID_SIZE                   4
#define MAX_PARSED_LOG_MESSAGE_LEN 200
#define TIMEOUT_MICRO_SECONDS      200000

    /*---- public types -------------------------------------------------------------------*/
    typedef struct UID_t
    {
        uint8_t values[UID_SIZE];
    } UID_t;

    typedef struct Point3D_t
    {
        int16_t X_i16;
        int16_t Y_i16;
        int16_t Z_i16;
        uint8_t Intensity_u8;
        uint8_t Confidence_u8;
    } Point3D_t;

    typedef struct Point1D_t
    {
        uint16_t VectorLength_u16;
        uint8_t Intensity_u8;
    } Point1D_t;

    typedef enum SessionState_t
    {
        INIT,
        EXPECTED,
        RUNNING,
        COMPLETE,
    } SessionState_t;

    typedef struct Sensor_Session_t
    {
        SessionState_t SessionState;
        uint16_t SenderId_u16;
        uint8_t NumberOfPoints_u8;
        uint16_t NoiseLevel_u16;
        bool NearfieldPoint_b;
        float NearfieldLevel_f;
        uint8_t NumberOf1DPoints;
        uint8_t NumberOf3DPoints;
        bool silentFrame;
        bool alienSensor;
        bool blind;
        Point1D_t Point1D_tp[MAX_NUMBER_OF_POINTS_PER_SESSION];
        Point3D_t Point3D_tp[MAX_NUMBER_OF_POINTS_PER_SESSION];
    } Sensor_Session_t;

    typedef struct ADCDump_t
    {
        SessionState_t SessionState;
        uint16_t SenderId_u16;
        uint32_t ExpectedDumpSize_u32;
        uint32_t ReceivedDumpSize_u32;
        uint8_t* DumpBlob_pu8;
    } ADCDump_t;

    typedef struct Sensor_t
    {
        uint16_t InterfaceSensorId_u16;
        uint8_t InterfaceIndex_u8;
        enum SensorMode_t SensorMode;
    } Sensor_t;

    /**
     * A structure to store ACK-Status Data
     */
    typedef struct ACK_Status_t
    {
        uint16_t SenderId_u16; /**< Sender Id */
        bool WaitForACK_b;     /**< Wait for ACK-Status will turn false if responce was received */
        bool ACK_b;            /**< ACK-Status */
        uint8_t ResponsePayload_pu8[CAN_MAX_FRAME_LEN]; /**< Response Payload that was received from
                                                           this sender */
    } ACK_Status_t;

    /*---- public variables prototypes ----------------------------------------------------*/

    /*---- public functions prototypes ----------------------------------------------------*/

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Returns pointer to array containing all known sensors. With
    GetNumberOfKnownSensors() users can get length of array.
      @return       Sensor_t* Pointer to array containing all currently known Sensors on Bus
    *******************************************************************************/
    Sensor_t*
    GetKnownSensors();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Returns number of known sensors. With GetKnownSensors() users an pointer to an
    array containing the ID's of the sensors.
      @return       uint8_t Number of currently known sensors on Bus
    *******************************************************************************/
    uint8_t
    GetNumberOfKnownSensors();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        With this function, the user can set the current target-id, all comming messages
    will be send to
      @param[in]    TargetSensor_u16 Target Id
    *******************************************************************************/
    void
    SetTargetSensor(uint16_t TargetSensor_u16);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        With this function, the user can get the current target-id, all comming messages
    will be send to
      @return       uint16_t Target Id
    *******************************************************************************/
    uint16_t
    GetTargetSensor();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        With this function, the user can set command timeout time.
      @param[in]    timeout Timeout in microseconds
    *******************************************************************************/
    void
    SetTimeout(int timeout);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        With this function, the user can get the current command timeout time.
      @return       int Timeout in microseconds
    *******************************************************************************/
    int
    GetTimeout();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking - Request transmition of the current version of selected component from
    selected sensor
      @param[in]    TargetComponent_t Component whose version number is required
      @return       Version_t Received version
    *******************************************************************************/
    Version_t
    RequestVersion_t(VersionByte_t TargetComponent_t);

    // Actions

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to reboot
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestReboot();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to reply with a heartbeat message
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestHeartbeat();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to shutdown. Used only for SIL testing.
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestShutdown();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Send's a request to start a new measurement
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestMeasurement();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to reset to factory defaults
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestFactoryReset();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to store current settings
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestStoreSettings();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to load saved settings
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestLoadSettings();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to create an ADC-Dump
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestADCDump();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to calibrate nearfield
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestCalibrateNearfield();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Request selected sensor to calibrate transducer
      @return       bool true if ACK
                    bool false in case of NACK / unexpected
    CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
    *******************************************************************************/
    bool
    RequestCalibrateTransducer();

    // Setter

    // ADC

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to use the fixed framerate for current target sensor
      @param[in]    UseFixedFrameRate_b
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCUseFixedFrameRate(bool UseFixedFrameRate_b);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the fixed framerate for current target sensor
      @param[in]    FrameRate_u8
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCFixedFrameRate(uint8_t FrameRate_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the silent frame period for current target sensor
      @param[in]    Period_u8
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSilentFramePeriod_u8(uint8_t Period_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the silent frame mode for current target sensor
      @param[in]    Mode_u8. SILENT_FRAME_MODE_SINGLE or SILENT_FRAME_MODE_WHILE_BLINDED
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSilentFrameMode_u8(uint8_t Mode_u8);

    // Transducer

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Transducer Volume for current target sensor
      @param[in]    Volume_u8
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterTransducerVolume(uint8_t Volume_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Number of Transducer Pulses for current target sensor
      @param[in]    NumOfPulses_u8
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterTransducerNumOfPulses(uint8_t NumOfPulses_u8);

    // SignalProcessing

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Temperature for current target sensor
      @param[in]    Temperature_f
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingTemperature(float Temperature_f);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Humidity for current target sensor
      @param[in]    Humidity_u8
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingHumidity(uint8_t Humidity_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Noise Level Threshold Factor for current target sensor
      @param[in]    Factor_f
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingNoiseLevelThresholdFactor(float Factor_f);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Noise Ratio Threshold for current target sensor
      @param[in]    Threshold_u8
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingNoiseRatioThreshold(uint8_t Threshold_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to enable near field detection for current target sensor
      @param[in]    Enable_b
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingEnableNearfieldDetection(bool Enable_b);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the threshold for nearfield detection
      @param[in]    Enable_b
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingNearfieldThreshold(uint16_t Threshold_u16);

    // /** ******************************************************************************
    //   @ingroup      Toposens Sensor Library
    //   @brief        Blocking Request to enable near the multipath filter for current target sensor
    //   @param[in]    Enable_b
    //   @return       bool true if setting was successful
    //                 bool false in case of an error
    // *******************************************************************************/
    // bool SetParameterSignalProcessingEnableMultipathFilter(bool Enable_b);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to enable auto gain for current target sensor
      @param[in]    Enable_b
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingEnableAutoGain(bool Enable_b);

    // System

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set a new NodeID for current target sensor
      @param[in]    NodeID_u16
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSystemNodeID(uint16_t NodeID_u16);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Loglevel for current target sensor
      @param[in]    LogLevel_t LogLevel_t
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSystemLogLevel(LogLevel_t LogLevel_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the CAN speed for current target sensor
      @param[in]    CANSpeed_t CANSpeed_t
      @return       bool true if setting was successful
                    bool false in case of an error
      @note         This settings needs to be stored and the device needs to be
                    rebooted to be effective
    *******************************************************************************/
    bool
    SetParameterSystemCANSpeed(CANSpeed_t CANSpeed_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Sensor Mode for current target sensor
      @param[in]    Mode_t
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSystemSensorMode(SensorMode_t Mode_t);

    // Getter

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the "Use fixed framerate" value from current sensor
      @return       bool true if fixed framerate is used
    *******************************************************************************/
    bool
    GetParameterADCUseFixedFrameRate_b();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current set ADC-Framerate from current sensor
      @return       uint8_t framerate
    *******************************************************************************/
    uint8_t
    GetParameterADCFixedFrameRate_u8();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current set Silent Frame period from current sensor
      @return       uint8_t period in frames. 0 means disabled.
    *******************************************************************************/
    uint8_t
    GetParameterSilentFramePeriod_u8();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current set Silent Frame mode from current sensor
      @return       uint8_t mode. SILENT_FRAME_MODE_SINGLE or SILENT_FRAME_MODE_WHILE_BLINDED
    *******************************************************************************/
    uint8_t
    GetParameterSilentFrameMode_u8();

    // Transducer

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Transducer Volume from current sensor
      @return       uint8_t current Transducer Volume
    *******************************************************************************/
    uint8_t
    GetParameterTransducerVolume_u8();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Transducer Number of Pulses from current
    sensor
      @return       uint8_t Transducer Number of Pulses
    *******************************************************************************/
    uint8_t
    GetParameterTransducerNumOfPulses_u8();

    // SignalProcessing

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Transducer Temperature setting from current
    sensor
      @return       float Set Temperature
    *******************************************************************************/
    float
    GetParameterSignalProcessingTemperature_f();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Transducer Humidity setting from current
    sensor
      @return       uint8_t Set Humidity
    *******************************************************************************/
    uint8_t
    GetParameterSignalProcessingHumidity_u8();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Noise Level Threshold Factor from current
    sensor
      @return       float Noise Level Threshold Factor
    *******************************************************************************/
    float
    GetParameterSignalProcessingNoiseLevelThresholdFactor_f();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Noise Ratio Threshold from current sensor
      @return       uint8_t Noise Ratio Threshold
    *******************************************************************************/
    uint8_t
    GetParameterSignalProcessingNoiseRatioThreshold_u8();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read whether near field detection is enabled in the current
    sensor
      @return       bool Enabled status
    *******************************************************************************/
    bool
    GetParameterSignalProcessingEnableNearfieldDetection_b();

    /** ******************************************************************************
       @ingroup      Toposens Sensor Library
       @brief        Blocking Request to read the nearfield threshold value
       @return       uint16_t Nearfield threshold value
     *******************************************************************************/
    uint16_t
    GetParameterSignalProcessingNearfieldThreshold_u16();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read whether auto gain is enabled in the current sensor
      @return       bool Enabled status
    *******************************************************************************/
    bool
    GetParameterSignalProcessingEnableAutoGain_b();

    // System

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current NodeID setting from current sensor
      @return       uint16_t NodeID
    *******************************************************************************/
    uint16_t
    GetParameterSystemNodeID_u16();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current CAN speed from current sensor
      @return       CANSpeed_t SystemCANSpeed
    *******************************************************************************/
    CANSpeed_t
    GetParameterSystemCANSpeed_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current System Log Level from current sensor
      @return       LogLevel_t SystemLogLevel
    *******************************************************************************/
    LogLevel_t
    GetParameterSystemLogLevel_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the last Reset Reason from current sensor
      @return       ResetReason_t ResetReason
    *******************************************************************************/
    ResetReason_t
    GetParameterSystemResetReason_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the Sensor State from current sensor
      @return       SensorState_t Sensor State
    *******************************************************************************/
    SensorState_t
    GetParameterSystemSensorState_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the Unique ID from current sensor
      @return       UID_t Unique ID
    *******************************************************************************/
    UID_t
    GetParameterSystemUniqueID_t(uint8_t Index_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the Sensor Mode from current sensor
      @return       SensorMode_t Sensor Mode
    *******************************************************************************/
    SensorMode_t
    GetParameterSystemSensorMode_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read whether the transducer has been calibrated
      @return       bool Calibration status
    *******************************************************************************/
    bool
    GetParameterSystemCalibTransducerState_b();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read whether the nearfield detection has been calibrated
      @return       bool Calibration status
    *******************************************************************************/
    bool
    GetParameterSystemCalibNearfieldState_b();

    // Sessions

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Function call to wait for SessionStart Payload
      @return       uint16_t SenderId
    *******************************************************************************/
    uint16_t
    WaitForSessionStart();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Function call to wait for SessionEnd Payload
      @return       uint16_t SenderId
    *******************************************************************************/
    uint16_t
    WaitForSessionEnd();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Function call to wait for SessionEnd Payload
      @return       uint16_t SenderId
    *******************************************************************************/
    uint16_t
    WaitForADCSessionEnd();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to query current session-data (ongoing and closed session)
      @param[in]    Sender_u16 Sender-Id the user wants to get current session-data from
      @return       Sensor_Session_t*
    *******************************************************************************/
    Sensor_Session_t*
    GetSessionData(uint16_t Sender_u16);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to get current ADC-Dump from SenderId
      @param[in]    Sender_u16
      @return       ADCDump_t* ADC-Dump that belongs to SenderId. NULL if not found.
    *******************************************************************************/
    ADCDump_t*
    GetADCDumpData(uint16_t Sender_u16);

    // Ready

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Function call to wait for Ready Payload
      @return       uint16_t SenderId
    *******************************************************************************/
    uint16_t
    WaitForReady();

    // Connections

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Instruction to initialize the specified interface with the specified bit rate.
    Will also initialize the semaphores if necessary.
      @param[in]    *InterfaceName name of the linux /dev/  device that should be used for
    communication
      @param[in]    DataRate_u32 Datarate the device should use. 1000000 for example on CAN Bus
      @param[in]    Interface_t The kind of interface to be used. Attention! A wrong interfacename
    might lead to unexpected system behavior!
      @return       TOPO_OK or error code
    *******************************************************************************/
    TopoError_t
    InitInterface(const char* InterfaceName, uint32_t DataRate_u32, CommsInterfaceType_t Interface_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Instruction to deinit the specified interface type.
      @param[in]    Interface_t The kind of interface to be deinit.
      @return       TOPO_OK or error code
    *******************************************************************************/
    TopoError_t
    DeinitInterface(CommsInterfaceType_t Interface_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Instruction to deinit the specified interface name.
      @param[in]    Interface_t The kind of interface to be deinit.
      @return       TOPO_OK or error code
    *******************************************************************************/
    TopoError_t
    DeinitInterfaceByName(const char* InterfaceName, CommsInterfaceType_t Interface_t);
    // Callbacks

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to register a callback function that is called whenever an
    SessionStart Payload is received.
      @param[in]    Callback function pointer
    *******************************************************************************/
    void
    RegisterSessionStartCallback(void (*Callback)(uint16_t Sender_u16));

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to register a callback function that is called whenever an
    SessionEnd Payload is received.
      @param[in]    Callback function pointer
    *******************************************************************************/
    void
    RegisterPointSessionEndCallback(void (*Callback)(uint16_t Sender_u16));

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to register a callback function that is called whenever an
    SessionEnd Payload for an active ADCDump is received.
      @param[in]    Callback function pointer
    *******************************************************************************/
    void
    RegisterADCDumpSessionEndCallback(void (*Callback)(uint16_t Sender_u16));

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to register a callback function that is called whenever an
    Ready Payload is received.
      @param[in]    Callback function pointer
    *******************************************************************************/
    void
    RegisterRdyCallback(void (*Callback)(uint16_t Sender_u16));

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to register a callback function that is called whenever an
    ADC-Dump Start Payload is received.
      @param[in]    Callback function pointer
    *******************************************************************************/
    void
    RegisterADCDumpStartRequestCallback(void (*Callback)(uint16_t Sender_u16, uint32_t ADCDumpSize_32));

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Interface function to register a callback function that is called whenever a
    Logmessage Payload is received.
      @param[in]    Callback function pointer
    *******************************************************************************/
    void
    RegisterLogMsgCallback(void (*Callback)(uint16_t Sender_u16, uint8_t* ReceivedPayload_pu8));

    // Logging

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Parses log messages into human readable text.
      @param[out]   *Output_p8
      @param[in]    SenderId_u16
      @param[in]    *ReceivedPayload_pu8
    *******************************************************************************/
    void
    ParseLogMessageToText(char* Output_p8, uint16_t SenderId_u16, const uint8_t* ReceivedPayload_pu8);

    // Helpers

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        This function is called from blocking Set* Functions and blocks until the
  wait_ack_sem is unlocked by the EvaluateACKStatus_b function. The blocking behavior is achieved by
  trying to fetch the same semaphore twice. The second sem_wait blocks until sem_post is executed
  elsewhere. That happens in EvaluateACKStatus_b.
    @param[in]    *payload The payload is copied to wait_ack_payload_pu8. EvaluateACKStatus_b compares
  incoming ACKs whether they match the desired ACK.
    @param[in]    length_u16
    *******************************************************************************/
    void
    WaitForACK(uint8_t* payload, uint16_t length_u16);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called from Getter functions to extract values from payload. In this case a
  bool-value.
    @param[in]    *Payload_pu8
    @return       bool true value was set
                  bool false value was not set
    *******************************************************************************/
    bool
    GetBoolFromPayload_b(const uint8_t* Payload_pu8);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called from Getter functions to extract values from payload. In this case a
  uint8-value.
    @param[in]    *Payload_pu8
    @return       uint8_t value extracted from payload
    *******************************************************************************/
    uint8_t
    GetU8FromPayload_u8(const uint8_t* Payload_pu8);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called from Getter functions to extract values from payload. In this case a
  uint8-value.
    @param[in]    *Payload_pu8
    @return       uint8_t value extracted from payload
    *******************************************************************************/
    uint8_t
    GetU8FromTwoArgumentPayload_u8(const uint8_t* Payload_pu8);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called from Getter functions to extract values from payload. In this case a
  uint16-value.
    @param[in]    *Payload_pu8
    @return       uint16_t value extracted from payload
    *******************************************************************************/
    uint16_t
    GetU16FromPayload_u16(const uint8_t* Payload_pu8);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called from Getter functions to extract values from payload. In this case a
  uint16-value.
    @param[in]    *Payload_pu8
    @return       uint16_t value extracted from payload
    *******************************************************************************/
    uint16_t
    GetU16FromTwoArgumentPayload_u16(const uint8_t* Payload_pu8);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @return       bool
    *******************************************************************************/
    bool
    RequestWasSuccessful_b();

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called internally to extract the payload of a response.
    @param[in]    *ackstatus
    @return       uint8_t* pointer to the payload
    *******************************************************************************/
    uint8_t*
    GetResponsePayloadFromACKStatus(ACK_Status_t* ackstatus);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called internally to get the current ACKStatus.
    @param[in]    index
    @return       ACK_Status_t* pointer to the struct
    *******************************************************************************/
    ACK_Status_t*
    GetCurrentACKStatus(uint8_t index);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called from Getter functions to extract values from payload. In this case a
  i32-value.
    @param[in]    *Payload_pu8
    @return       int32_t value extracted from payload
    *******************************************************************************/
    int32_t
    GetI32FromPayload_i32(const uint8_t* Payload_pu8);

    /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Is called from Getter functions to extract values from two argument
  payload. In this case a i32-value.
    @param[in]    *Payload_pu8
    @return       int32_t value extracted from payload
    *******************************************************************************/
    int32_t
    GetI32FromTwoArgumentPayload_i32(const uint8_t* Payload_pu8);

    /** ******************************************************************************
        @ingroup      Toposens Sensor Library
        @brief        Blocking Request to read the current timeslot for a given step from current
      sensor
        @param[in]    Step_u8
        @return       uint16_t timeslot
      *******************************************************************************/
    uint16_t
    GetParameterADCStepGainTimeSlot_u16(uint8_t Step_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current adc-step-gain-factor for a given index from
    current sensor
      @param[in]    Index_u8
      @return       uint16_t timeslot
    *******************************************************************************/
    StepGainFactor_t
    GetParameterADCStepGainGainFactor_t(uint8_t Index_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current adc Number of step gains from current sensor
      @return       uint8_t number of steps
    *******************************************************************************/
    uint8_t
    GetParameterADCStepGainNumberOfSteps_u8();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Get the number of recorded adc samples.
      @return       number of recorded adc samples [0..65535]
    *******************************************************************************/
    uint32_t
    GetParameterADCNumberOfSamples_u32();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current ADC-Resolution from current sensor
      @return       ADCResolution_t Current Resolution
    *******************************************************************************/
    ADCResolution_t
    GetParameterADCSampleResolution_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current ADC Sample Time from current sensor
      @return       ADCSampleTime_t Current Sample Time
    *******************************************************************************/
    ADCSampleTime_t
    GetParameterADCSampleTime_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current ADC Sample Rate from current sensor
      @return       ADCSampleRate_t Current Sample Rate
    *******************************************************************************/
    ADCSampleRate_t
    GetParameterADCSampleRate_t();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the adc-step-gain time slot for current target sensor
      @param[in]    Index_u8 Step Index
      @param[in]    Value_u16 Timeslot
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCStepGainTimeSlot(uint8_t Index_u8, uint16_t Value_u16);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the gain factor for current target sensor
      @param[in]    Index_u8 Index
      @param[in]    GainFactor_t
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCStepGainGainFactor(uint8_t Index_u8, StepGainFactor_t GainFactor_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Number of step gains for current target sensor
      @param[in]    NumberOfSteps_u8
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCStepGainNumberOfSteps(uint8_t NumberOfSteps_u8);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the number of adc samples
      @param[in]    SampleDistanceMax_u16 number of recorded adc samples [0..65535]
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCNumberOfSamples(uint32_t NumberOfSamples_u32);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the ADC Resolution for current target sensor
      @param[in]    SampleResolution_t
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCSampleResolution(ADCResolution_t SampleResolution_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the ADC Sample Time for current target sensor
      @param[in]    SampleTime_t
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCSampleTime(ADCSampleTime_t SampleTime_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the ADC Sample Rate for current target sensor
      @param[in]    SampleRate_t
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterADCSampleRate(ADCSampleRate_t SampleRate_t);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Transducer Frequency from current sensor
      @return       uint16_t Transducer Frequency
    *******************************************************************************/
    uint16_t
    GetParameterTransducerFrequency_u16();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to set the Number of Transducer Frequency for current target
    sensor
      @param[in]    Frequency_u16
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterTransducerFrequency(uint16_t Frequency_u16);

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to read the current Transducer "use manual speed of sound"
    setting from current sensor
      @return       bool true if use manual speed of sound is enabled
    *******************************************************************************/
    bool
    GetParameterSignalProcessingVSound_b();

    /** ******************************************************************************
      @ingroup      Toposens Sensor Library
      @brief        Blocking Request to use manual speed of sound for current target sensor
      @param[in]    UseManualSetting_b
      @return       bool true if setting was successful
                    bool false in case of an error
    *******************************************************************************/
    bool
    SetParameterSignalProcessingVSound(bool UseManualSetting_b);

    /*---- public inline functions --------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif  // TOPOSENS_SENSOR_LIB_H
