#ifndef URG_UTILS_H
#define URG_UTILS_H

/*!
  \file
  \~japanese
  \brief URG ZTp̕⏕֐

  \~english
  \brief URG sensor utility

  \~
  \author Satofumi KAMIMURA

  $Id: urg_utils.h,v 630ee326c5ce 2011/02/19 08:06:25 Satofumi $
*/

#ifdef __cplusplus
extern "C" {
#endif

#include "urg_c/urg_sensor.h"


    /*!
      \~japanese
      \brief URG ̃G[Ԃ

      \param[in] urg URG ZTǗ

      \retval URG ̃G[

      \~
      Example
      \code
      if (!urg_open(&urg, "/dev/ttyACM0", 115200, URG_SERIAL)) {
      printf("urg_open: %s\n", urg_error(&urg));
      return -1;
      } \endcode
    */
    extern const char *urg_error(const urg_t *urg);


    /*!
      \~japanese
      \brief ZTԂ̍őlAŏlԂ

      ZTԂ [ŏl, ől] ŕԂ܂B

      \param[in] urg URG ZTǗ
      \param[out] min_distance ŏl [mm]
      \param[out] max_distance ől [mm]

      \~
      Example
      \code
      long min_distance, max_distance;
      urg_distance_min_max(&urg, &min_distance, &max_distance);

      for (int i = 0; i < n; ++i) {
      long distance = data[i];
      if ((distance < min_distance) || (distance > max_distance)) {
      continue;
      }
      ...
      } \endcode
    */
    extern void urg_distance_min_max(const urg_t *urg,
                                     long *min_distance, long *max_distance);


    /*!
      \~japanese
      \brief v step ̍őlAŏlԂ

      urg_set_scanning_parameter() Ŏwł͈͂ [ŏl, ől] ŕԂB

      \param[in] urg URG ZTǗ
      \param[out] min_step ŏl
      \param[out] max_step ől

      step ̓ZTʂ 0 łAZT㕔猩ꍇ̔v܂̕Av܂̕ step lƂȂB

      \image html sensor_step_image.png ZT step ̊֌W

      min_step, max_step ̒l̓ZTɂĈقȂB

      \~
      Example
      \code
      urg_step_min_max(&urg, &min_step, &max_step);

      printf("range first: %d [deg]\n", urg_step2deg(&urg, min_step));
      printf("range last : %d [deg]\n", urg_step2deg(&urg, max_step)); \endcode

      \see urg_set_scanning_parameter(), urg_step2rad(), urg_step2deg()
    */
    extern void urg_step_min_max(const urg_t *urg, int *min_step, int *max_step);


    /*! \~japanese PXLɂ鎞 [usec] Ԃ */
    extern long urg_scan_usec(const urg_t *urg);


    /*! \~japanese 擾f[^̍őlԂ */
    extern int urg_max_data_size(const urg_t *urg);


    /*!
      \~japanese
      \brief CfbNXƊpx(radian)̕ϊs

      CfbNƂ urg_get_distance() Ȃǂ̋f[^擾֐Ԃf[^zɂĂ̒lłB̊֐́AŌɍsf[^擾֐̃f[^zɂėLƂȂB

      \param[in] urg URG ZTǗ
      \param[in] index CfbNX

      \return px [radian]

      index ́A擾vf[^ɂĂ̒lł step pxƂ̊֌W͎擾ݒɂقȂB

      \image html sensor_index_image.png ZŤv͈͂ƃCfbNX̊֌W

      \~
      Example
      \code
      int n = urg_get_distance(&urg, data, NULL);
      for (int i = 0; i < n; ++i) {
      long distance = data[i];
      double radian = urg_index2rad(i);
      double x = distance * cos(radian);
      double y = distance * sin(radian);
      printf("%.1f, %.1f\n", x, y);
      } \endcode

      \see urg_index2deg(), urg_rad2index(), urg_deg2index()
    */
    extern double urg_index2rad(const urg_t *urg, int index);


    /*! \~japanese CfbNXƊpx(degree)̕ϊs */
    extern double urg_index2deg(const urg_t *urg, int index);


    /*! \~japanese px(radian)ƃCfbNX̕ϊs */
    extern int urg_rad2index(const urg_t *urg, double radian);


    /*! \~japanese px(degree)ƃCfbNX̕ϊs */
    extern int urg_deg2index(const urg_t *urg, double degree);


    /*!
      \~japanese
      \brief px(radian) step ̕ϊs

      urg_step_min_max() Œ`Ă step ɂāApx(radian) step ̕ϊsB

      \param[in] urg URG ZTǗ
      \param[in] radian px [radian]

      \return step

      \image html sensor_angle_image.png ZT step ƊpxƂ̊֌W

      px step ֕ϊʂłȂꍇAʂ 0 ̕ɐ؂̂ĂꂽlƂȂB

      \~
      \see urg_step_min_max(), urg_deg2step(), urg_step2rad(), urg_step2deg()
    */
    extern int urg_rad2step(const urg_t *urg, double radian);


    /*! \~japanese px(degree) step ̕ϊs */
    extern int urg_deg2step(const urg_t *urg, double degree);


    /*! \~japanese step  px(radian)̕ϊs */
    extern double urg_step2rad(const urg_t *urg, int step);


    /*! \~japanese step  px(degree)̕ϊs */
    extern double urg_step2deg(const urg_t *urg, int step);

    /*! \~japanese step ƃCfbNX̕ϊs */
    extern int urg_step2index(const urg_t *urg, int step);

#ifdef __cplusplus
}
#endif

#endif /* !URG_UTILS_H */
