#include <homer_nav_libs/tools/tools.h>
#include <algorithm>
#include <iostream>
#include <fstream>
#include <sstream>

#include <QMessageBox>
#include <QSpacerItem>
#include <QListWidget>
#include <QListWidgetItem>


#include "MapTab.h"

#include <ros/ros.h>
#include <tf/transform_datatypes.h>

// #include "GUI/Containers/SemanticMapTab/SemanticMapTab.h"

#include <geometry_msgs/Pose2D.h>
#include <homer_mapnav_msgs/PointOfInterest.h>
#include <homer_mapnav_msgs/ModifyPOI.h>
#include <homer_mapnav_msgs/DeletePointOfInterest.h>
#include <std_msgs/Empty.h>
#include <homer_mapnav_msgs/MapLayers.h>
#include <homer_mapnav_msgs/ModifyMap.h>
#include <homer_mapnav_msgs/StartNavigation.h>
#include <homer_mapnav_msgs/StopNavigation.h>
#include <std_msgs/Empty.h>
#include <std_msgs/String.h>

#include <homer_nav_libs/tools/loadRosConfig.h>

#include <QHeaderView>

#include <ros/package.h>

MapTab::MapTab (  QWidget *parent, ros::NodeHandle *nodeHandle) : QWidget ( parent )
{
  m_NodeHandle = nodeHandle;
  //add ros publishers here
  m_AddPOIPublisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::PointOfInterest>("/map_manager/add_POI", 10));
  m_UserDefPosePublisher = new ros::Publisher(m_NodeHandle->advertise<geometry_msgs::Pose>("/homer_mapping/userdef_pose", 10));
  m_ModifyPOIPublisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::ModifyPOI>("/map_manager/modify_POI", 10));
  m_DeletePOIPublisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::DeletePointOfInterest>("/map_manager/delete_POI", 10));
  m_SaveMapPublisher = new ros::Publisher(m_NodeHandle->advertise<std_msgs::String>("/map_manager/save_map", 10));
  m_LoadMapPublisher = new ros::Publisher(m_NodeHandle->advertise<std_msgs::String>("/map_manager/load_map", 10));
  m_DoMappingPublisher = new ros::Publisher(m_NodeHandle->advertise<std_msgs::Bool>("/homer_mapping/do_mapping", 10));
  m_ResetMapsPublisher = new ros::Publisher(m_NodeHandle->advertise<std_msgs::Empty>("/map_manager/reset_maps", 10));
  m_MapVisibilityPublisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::MapLayers>("/map_manager/toggle_map_visibility", 10));

  //path planning service client
  m_StartNavigationPublisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::StartNavigation>("/homer_navigation/start_navigation", 10));
  m_StopNavigationPublisher = new ros::Publisher(m_NodeHandle->advertise<std_msgs::Empty>("/homer_navigation/stop_navigation", 10));

  // creates the poi icons and fills the m_PoiIcons                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          QT vector
  createUserDefinedPoiIcons();
  // get the needed config entries (map size, cell size)
  getConfigEntries();
  // create and arrange the QT widgets
  createWidgetsAndSetupLayout();
  // install all signal slot connections between the widgets
  installSignalSlotConnections();
  // initialize the MapWidget
  // links
  m_MapWidget->linkPoiList ( m_PoiList );
  m_MapWidget->linkPoiIconList ( m_PoiIcons );
  m_MapWidget->setCurrentPoiIcon ( m_PoiIcons[0].getFileName() );

  m_MapWidget->setResolution( m_Resolution );
  m_MapWidget->setOrigin( m_Origin );
  m_MapWidget->setPixelSize( m_PixelSize );

  showZoomFactor( 1.0 ); // initial display

  m_CurrentlySelectedMapID = -1;

  std::string default_mapfile;
  loadConfigValue("/map_manager/default_mapfile", default_mapfile);
  if(default_mapfile != "")
  {
      toggleDoMappingState(0);
      m_DoMappingCheckBox->setChecked( false );
  }
}

MapTab::~MapTab ()
{
}

void MapTab::createUserDefinedPoiIcons()
{
  std::string path = ros::package::getPath("homer_gui");
  m_PoiIcons.push_back ( PoiIcon ( (path + "/icons/DefaultPoi.png").c_str(), "Default", homer_mapnav_msgs::PointOfInterest::DEFAULT ) );
  m_PoiIcons.push_back ( PoiIcon ( (path + "/icons/softdrink.png").c_str(), "Object", homer_mapnav_msgs::PointOfInterest::OBJECT ) ); // TODO add icon file
  m_PoiIcons.push_back ( PoiIcon ( (path + "/icons/softdrink2.png").c_str(), "Grippable Object", homer_mapnav_msgs::PointOfInterest::GRIPPABLE_OBJECT ) ); // TODO add icon file
  m_PoiIcons.push_back ( PoiIcon ( (path + "/icons/WhoIsWho.png").c_str(), "Person", homer_mapnav_msgs::PointOfInterest::PERSON ) );
  m_PoiIcons.push_back ( PoiIcon ( (path + "/icons/startpos.png").c_str(), "Start", homer_mapnav_msgs::PointOfInterest::START_POSITION ) );
}


void MapTab::createWidgetsAndSetupLayout()
{
  // start of the poi groupbox implementation
  QLabel* poiNameLabel = new QLabel ( tr ( "Name" ) );
  QLabel* poiMiscLabel = new QLabel ( tr ( "Remarks" ) );
  QLabel* poiSymbolLabel = new QLabel ( tr ( "Type" ) );
  m_PoiNameLineEdit = new QLineEdit();
  m_PoiMiscLineEdit = new QLineEdit();
  m_PoiNameLineEdit->setEnabled ( false );
  m_PoiMiscLineEdit->setEnabled ( false );
  m_PoiApplyButton = new QPushButton ( tr ( "Apply" ) );
  m_PoiDeleteButton = new QPushButton ( tr ( "Delete" ) );
  m_PoiApplyButton->setEnabled ( false );
  m_PoiDeleteButton->setEnabled ( false );
  m_DriveToPoiButton = new QPushButton ( tr ( "Drive to" ) );
  m_DriveToPoiButton->setEnabled ( false );
  m_StopNavigationButton = new QPushButton ( tr ( "Stop Navigation" ) );
  m_StopNavigationButton->setEnabled ( false );

  m_PoiSymbolsComboBox = new QComboBox();
  for ( int i=0; i<m_PoiIcons.size(); i++ )
  {
    m_PoiSymbolsComboBox->addItem ( QIcon ( QPixmap ( m_PoiIcons[i].getFileName() ) ), m_PoiIcons[i].getIdentifier(), QVariant ( m_PoiIcons[i].getPoiType() ) );
  }

  m_PoiTreeWidget = new QTreeWidget ( this );
  m_PoiTreeWidget->setColumnCount ( 2 );
  QStringList headers;
  headers << "Name" << "Value";
  m_PoiTreeWidget->setHeaderLabels ( headers );
  m_PoiTreeWidget->setColumnWidth ( 0, 160 );


  QGridLayout* poiEditGridLayout = new QGridLayout;

  //1st column
  poiEditGridLayout->addWidget ( poiNameLabel, 0, 0 );
  poiEditGridLayout->addWidget ( poiMiscLabel, 1, 0 );
  poiEditGridLayout->addWidget ( poiSymbolLabel, 2, 0 );

  //2nd column
  poiEditGridLayout->addWidget ( m_PoiNameLineEdit, 0, 1 );
  poiEditGridLayout->addWidget ( m_PoiMiscLineEdit, 1, 1 );
  poiEditGridLayout->addWidget ( m_PoiSymbolsComboBox, 2, 1 );

  //3rd column
  poiEditGridLayout->addWidget ( m_PoiApplyButton, 0, 2 );
  poiEditGridLayout->addWidget ( m_PoiDeleteButton, 1, 2 );
  poiEditGridLayout->addWidget ( m_DriveToPoiButton, 2, 2 );
  poiEditGridLayout->addWidget ( m_StopNavigationButton, 3, 2 );



  //Tree view
  poiEditGridLayout->addWidget ( m_PoiTreeWidget, 4, 0, 1, 3 );

  poiEditGridLayout->setRowStretch ( 3,10 );

  QGroupBox* poiEditGroupBox = new QGroupBox ( tr ( "Points of Interest" ) );
  poiEditGroupBox->setLayout ( poiEditGridLayout );

  // start of the map modify options
  QGridLayout* mapModifyOptionsGridLayout = new QGridLayout();
  m_MaskModeButtonGroup = new QButtonGroup();
  QRadioButton* btn;

  btn = new QRadioButton ( tr ( "block" ) );
  btn->setProperty ( "maskAction", QVariant ( homer_mapnav_msgs::ModifyMap::SET_BLOCKED) );
  btn->setChecked ( true );
  m_MaskModeButtonGroup->addButton ( btn );
  mapModifyOptionsGridLayout->addWidget ( btn, 0, 0 );

  btn = new QRadioButton ( tr ( "free" ) );
  btn->setProperty ( "maskAction", QVariant ( homer_mapnav_msgs::ModifyMap::SET_FREE ) );
  m_MaskModeButtonGroup->addButton ( btn );
  mapModifyOptionsGridLayout->addWidget ( btn, 1, 0 );

  btn = new QRadioButton ( tr ( "obstacle" ) );
  btn->setProperty ( "maskAction", QVariant ( homer_mapnav_msgs::ModifyMap::SET_OBSTACLE ) );
  m_MaskModeButtonGroup->addButton ( btn );
  mapModifyOptionsGridLayout->addWidget ( btn, 0, 1 );
  
  btn = new QRadioButton ( tr ( "sensitiv" ) );
  btn->setProperty ( "maskAction", QVariant ( homer_mapnav_msgs::ModifyMap::SET_HIGH_SENSITIV ) );
  m_MaskModeButtonGroup->addButton ( btn );
  mapModifyOptionsGridLayout->addWidget ( btn, 1, 1 );

  QPushButton* maskButton = new QPushButton ( tr ( "apply" ) );
  QPushButton* clearButton = new QPushButton ( tr ( "cancel" ) );

  mapModifyOptionsGridLayout->addItem( new QSpacerItem( 0,0,QSizePolicy::Expanding, QSizePolicy::Minimum ), 0, 3 );
  mapModifyOptionsGridLayout->addItem( new QSpacerItem( 0,0,QSizePolicy::Expanding, QSizePolicy::Minimum ), 1, 3 );

  mapModifyOptionsGridLayout->addWidget ( maskButton, 0, 4 );
  mapModifyOptionsGridLayout->addWidget ( clearButton, 1, 4 );

  QGroupBox* mapModifyOptionsGroupBox = new QGroupBox ( tr ( "Edit map (press Shift)" ) );
  mapModifyOptionsGroupBox->setLayout ( mapModifyOptionsGridLayout );

  // start of the options groupbox implementation
  m_DoMappingCheckBox = new QCheckBox ( tr ( "Do mapping" ) );
  m_DoMappingCheckBox->setEnabled ( true );
  m_DoMappingCheckBox->setChecked ( true );

  // start of the main buttons groupbox implementation
  QPushButton* saveMapButton = new QPushButton ( tr ( "Save selected maps" ) );
  saveMapButton->setIcon ( QIcon ( "icons/save.png" ) );
  QPushButton* openMapButton = new QPushButton ( tr ( "Open..." ) );
  openMapButton->setIcon ( QIcon ( "icons/open.png" ) );
  QPushButton* resetMapsButton = new QPushButton ( "Reset selected maps", this );
  QGridLayout* mainButtonGridLayout = new QGridLayout;

  mainButtonGridLayout->addWidget ( m_DoMappingCheckBox, 0, 0 );
  mainButtonGridLayout->addWidget ( saveMapButton, 1, 0 );
  mainButtonGridLayout->addWidget( resetMapsButton, 1, 1);
  mainButtonGridLayout->addWidget ( openMapButton, 2, 0 );

  QGroupBox* mainButtonsGroupBox = new QGroupBox ( tr ( "Map" ) );
  mainButtonsGroupBox->setLayout ( mainButtonGridLayout );

  // right layout
  QVBoxLayout* rightLayout = new QVBoxLayout;

  rightLayout->addWidget ( poiEditGroupBox );
  rightLayout->addStretch();
  rightLayout->setStretchFactor ( poiEditGroupBox,1 );

  // left layout
  QVBoxLayout* leftLayout = new QVBoxLayout;

  QCheckBox* showPoiCheckBox = new QCheckBox ( "POIs" );
  showPoiCheckBox->setChecked ( true );
  QCheckBox* showRobotCheckBox = new QCheckBox ( "Robot position" );
  showRobotCheckBox->setChecked ( true );
  QCheckBox* showGridCheckBox = new QCheckBox ( "Grid (1x1 m)" );
  showGridCheckBox->setChecked ( false );
  QCheckBox* antiAliasingToggle = new QCheckBox( "Anti aliasing" );
  antiAliasingToggle->setChecked ( false );
  QCheckBox* movablePoiToggle = new QCheckBox ( "Movable POI" );
  movablePoiToggle->setChecked ( true );

  m_ShowMousePosition = new QLabel ( "" );
  m_ShowZoomFactor = new QLabel ( "" );

  m_ScrollArea = new MapScrollArea( this );
  m_MapWidget = new MapDisplay ( m_NodeHandle, this );
  m_ScrollArea->setWidget ( m_MapWidget );
  m_ScrollArea->setFocus();


  QHBoxLayout* displayOptions = new QHBoxLayout;
  displayOptions->addWidget ( showPoiCheckBox );
  displayOptions->addWidget ( showRobotCheckBox );
  displayOptions->addWidget ( showGridCheckBox );
  //displayOptions->addWidget ( antiAliasingToggle );
  displayOptions->addWidget ( movablePoiToggle );
  displayOptions->addStretch();
  displayOptions->addWidget ( m_ShowMousePosition );
  displayOptions->addWidget ( m_ShowZoomFactor );

  leftLayout->addWidget ( m_ScrollArea );
  leftLayout->addLayout ( displayOptions );

  // map layers layout
  QVBoxLayout* mapLayout = new QVBoxLayout;

  QGroupBox* mapLayerGroupBox = new QGroupBox ( tr ( "Map Layers" ) );
  m_MapLayersListWidget = new QListWidget;
  std::map< int, std::string > mapNames=m_MapWidget->getMapNames();

  for ( std::map< int, std::string >::iterator it=mapNames.begin(); it != mapNames.end(); it++ )
  {
      QListWidgetItem *layerItem = new QListWidgetItem(QString(it->second.c_str() ), m_MapLayersListWidget);
      layerItem->setCheckState(Qt::Checked);
      layerItem->setText(QString(it->second.c_str()));
      layerItem->setData(Qt::UserRole, it->first);
      m_MapLayersListWidget->addItem(layerItem);
  }
  QVBoxLayout* maplayerLayout = new QVBoxLayout;
  maplayerLayout->addWidget(m_MapLayersListWidget);
  mapLayerGroupBox->setLayout(maplayerLayout);
  mapLayout->addWidget(mapLayerGroupBox);
  mapLayout->addWidget ( mapModifyOptionsGroupBox );


  // Code added by Carmen Navarro Luzón for adding Semantics Tab
  // 2011 Masterarbeit
  QTabWidget* mapSubTabs = new QTabWidget();

  QWidget* POITab = new QWidget(0, 0);
  QWidget* MapLayerTab = new QWidget();
  // m_SemanticsTab = new SemanticMapTab(m_MapWidget, this); // TODO add later

  POITab->setLayout(rightLayout);
  MapLayerTab->setLayout(mapLayout);

  mapSubTabs->setContentsMargins ( 0, 0, 0, 0 );
  mapSubTabs->addTab(POITab, tr ("&POIs"));
  mapSubTabs->addTab(MapLayerTab, tr("&Map Layers"));
  //mapSubTabs->addTab(m_SemanticsTab, tr ("&Semantics")); // TODO add later

  QVBoxLayout* mainRightLayout = new QVBoxLayout();
  mainRightLayout->addWidget(mapSubTabs);
  mainRightLayout->addWidget(mainButtonsGroupBox);


  // main layout
  QGridLayout* mainHBoxLayout = new QGridLayout;
  mainHBoxLayout->addLayout ( leftLayout,0,0 );

  // Line added by Carmen Navarro Luzón for adding Semantics Tab
  // 2011 Masterarbeit
  mainHBoxLayout->addLayout ( mainRightLayout,0,1);
  //mainHBoxLayout->addWidget ( mapSubTabs,0,1 );
  //mainHBoxLayout->addLayout ( rightLayout,0,1 );

  mainHBoxLayout->setColumnStretch ( 0,4 );
  mainHBoxLayout->setColumnStretch ( 1,2 );
  setLayout ( mainHBoxLayout );

  connect( saveMapButton, SIGNAL( clicked() ), this, SLOT( clickedSaveMapButton() ) );
  connect( openMapButton , SIGNAL( clicked() ), this, SLOT( clickedOpenMapButton() ) );
  connect ( resetMapsButton, SIGNAL ( clicked() ), this, SLOT ( clickedResetMapsButton() ) );


  connect( maskButton, SIGNAL( clicked() ), this, SLOT( clickedMaskButton() ) );
  connect( clearButton, SIGNAL( clicked() ), m_MapWidget, SLOT( resetRegion() ) );


  connect( m_DoMappingCheckBox, SIGNAL( stateChanged( int ) ), this, SLOT( toggleDoMappingState( int ) ) );

  connect( showPoiCheckBox, SIGNAL( stateChanged( int ) ), this, SLOT( togglePoiView( int ) ) );
  connect( showRobotCheckBox, SIGNAL( stateChanged( int ) ), this, SLOT( toggleRobotView( int ) ) );
  connect( showGridCheckBox, SIGNAL( stateChanged( int ) ), m_MapWidget, SLOT( toggleGridVisibility(int) ) );
  connect( antiAliasingToggle, SIGNAL( stateChanged( int ) ), m_MapWidget, SLOT( toggleAntiAliasing( int ) ) );
  connect( movablePoiToggle, SIGNAL( stateChanged( int ) ), m_MapWidget, SLOT( toggleMovablePoi( int ) ) );

  connect(m_MapLayersListWidget, SIGNAL(itemChanged(QListWidgetItem*)), this, SLOT(toggleMapLayerVisibility(QListWidgetItem*)));
  connect(m_MapLayersListWidget, SIGNAL(itemSelectionChanged()), this, SLOT(changeCurrentMapLayer()));

  // views
  toggleRobotView ( showRobotCheckBox->isChecked() );
  togglePoiView ( showPoiCheckBox->isChecked() );

}


void MapTab::getConfigEntries()
{
  loadConfigValue("/homer_mapping/size", m_MapSize);
  loadConfigValue("/homer_mapping/resolution", m_CellSize);
  m_PixelSize = static_cast<double>(m_MapSize)/m_CellSize + 1;
  m_Origin.position.x = -m_PixelSize*m_Resolution/2;
  m_Origin.position.y = -m_PixelSize*m_Resolution/2;
  m_Origin.orientation.w = 1.0;
  m_Origin.orientation.x = 0.0;
  m_Origin.orientation.y = 0.0;
  m_Origin.orientation.z = 0.0;
}

void MapTab::installSignalSlotConnections() {
  connect( m_MapWidget,SIGNAL( robotPositionChanged( QPoint, double ) ),this, SLOT( userdefinedRobotorPos( QPoint, double ) ) );
  connect( m_MapWidget, SIGNAL( addedNewPoi( QPoint ) ), this, SLOT( addedNewPoi( QPoint ) ) );
  connect( m_PoiNameLineEdit, SIGNAL( editingFinished() ), this, SLOT( modifyPoiNameLineEdit() ) );
  connect( m_PoiNameLineEdit, SIGNAL( textChanged( const QString& ) ), this, SLOT( modifyPoiNameLineEdit() ) );
  connect( m_PoiApplyButton, SIGNAL( clicked() ), this, SLOT( clickedPoiApplyButton() ) );

  /* NOTE to catch <return> */
  connect( m_PoiNameLineEdit, SIGNAL( returnPressed() ), this, SLOT( clickedPoiApplyButton() ) );

  connect( m_PoiSymbolsComboBox, SIGNAL( activated( int ) ), this, SLOT( changedCurrentPoiIcon( int ) ) );
  connect( m_PoiTreeWidget, SIGNAL( itemSelectionChanged() ), this, SLOT( changedCurrentPoiSelectionByTreeWidget() ) );
  connect( m_MapWidget, SIGNAL( changedMousePosition( QPoint ) ), this, SLOT( showMousePoition( QPoint ) ) );
  connect( m_MapWidget, SIGNAL( changedZoomFactor( float ) ), this, SLOT( showZoomFactor( float ) ) );
  connect( m_DriveToPoiButton, SIGNAL( clicked() ), this, SLOT( clickedDriveToPositionButton() ) );
  connect( m_StopNavigationButton, SIGNAL( clicked() ), this, SLOT( clickedStopNavigationButton() ) );
  connect( m_MapWidget, SIGNAL( centerMap( QPoint ) ), this, SLOT( centerMap( QPoint ) ) );
  connect( m_MapWidget, SIGNAL( modifiedPoi( QString, QString ) ), this, SLOT( changedCurrentPoiSelectionByMapWidget( QString, QString ) ) );
  connect( m_MapWidget, SIGNAL( modifiedPoiPosition( QPoint ) ), this, SLOT( saveTmpPoiPosition( QPoint ) ) );
  connect( m_MapWidget, SIGNAL( modifiedPoiOrientation( double ) ), this, SLOT( saveTmpPoiOrientation( double ) ) );
  connect( m_PoiDeleteButton, SIGNAL( clicked() ), this, SLOT( clickedPoiDeleteButton() ) );
}

QSize MapTab::minimumSizeHint() const {
  return QSize( 750, 450 );
}

QSize MapTab::sizeHint() const {
  return QSize( 1024,768 );
}

void MapTab::userdefinedRobotorPos( QPoint robotPos, double robotOrientation ) {
  m_UserDefinedRobotPos = robotPos;
  m_UserDefinedRobotOrientation = robotOrientation;

  geometry_msgs::Pose poseMsg;
  poseMsg.position = map_tools::fromMapCoords(Eigen::Vector2i(m_PixelSize - 1 - m_UserDefinedRobotPos.y(), m_PixelSize - 1 - m_UserDefinedRobotPos.x()), m_Origin, m_Resolution);
  poseMsg.orientation = tf::createQuaternionMsgFromYaw(-m_UserDefinedRobotOrientation);
  ROS_INFO_STREAM("Setting user defined robot pose: " << poseMsg);
  m_UserDefPosePublisher->publish(poseMsg);
}

void MapTab::addedNewPoi ( QPoint addedNewPoiPos )
{
  // to unselect a selected poi if the user want to add a new Poi
  m_PoiTreeWidget->setCurrentItem ( 0 );

  // indicator for the save poi functionality that a new poi shall be save
  m_CurrentPoiSelectionName = "";

  // update layout
  m_PoiNameLineEdit->clear();
  m_PoiNameLineEdit->setEnabled ( true );
  m_PoiNameLineEdit->setFocus();
  m_PoiMiscLineEdit->clear();
  m_PoiMiscLineEdit->setEnabled ( true );

  // save new poi information for further usage (apply)
  saveTmpPoiPosition ( addedNewPoiPos );
  m_TmpPoiOrientation = 0.0;
}

void MapTab::saveTmpPoiPosition ( QPoint pos )
{
  m_TmpPoiPos = pos;
}

void MapTab::saveTmpPoiOrientation ( double orientation )
{
  m_TmpPoiOrientation = orientation;
  clickedPoiApplyButton();
}

void MapTab::modifyPoiNameLineEdit()
{

  if ( m_PoiNameLineEdit->text().size() != 0 )
  {
    m_PoiApplyButton->setEnabled ( true );
  }
  else
  {
    m_PoiApplyButton->setEnabled ( false );
  }
}

void MapTab::changedCurrentPoiIcon ( int index )
{
  m_MapWidget->setCurrentPoiIcon ( m_PoiIcons[index].getFileName() );
}

void MapTab::clickedSaveMapButton()
{
  QString fileName = QFileDialog::getSaveFileName ( this, tr ( "Save Map" ), ".", tr ( "All files (*.*)" ) );

  QDir dir ( "." );
  QString fileNameRelative = dir.relativeFilePath ( fileName );

  if ( fileName.size() != 0 )
  {
	  std_msgs::String saveMapMsg;
    saveMapMsg.data = fileNameRelative.toStdString();
    m_SaveMapPublisher->publish(saveMapMsg);
	std_msgs::Bool doMappingMsg;
    doMappingMsg.data = false;
    m_DoMappingPublisher->publish(doMappingMsg);
    m_DoMappingCheckBox->setChecked(false);
  }


  //string aux = fileNameRelative.toStdString();
  // ROS_INFO_STREAM("Saving to file pepe.txt"); // TODO add some sense to this line
  // m_SemanticsTab->saveToFile("pepe.txt");//fileNameRelative.toStdString()); // TODO later add semantics
}

void MapTab::clickedOpenMapButton()
{
  QString fileName = QFileDialog::getOpenFileName ( this, tr ( "Load Map" ), ".", tr ( "YAML file format (*.yaml)" ) );

  QDir dir ( "." );
  QString fileNameRelative = dir.relativeFilePath ( fileName );

  if ( fileName.size() != 0 )
  {
	  std_msgs::String loadMapMsg;
    loadMapMsg.data = fileNameRelative.toStdString();
    m_LoadMapPublisher->publish(loadMapMsg);

    toggleDoMappingState(0);
    m_DoMappingCheckBox->setChecked( false );
  }
}

void MapTab::clickedResetMapsButton()
{
    std_msgs::Empty msg;
    m_ResetMapsPublisher->publish(msg);
}

void MapTab::clickedPoiApplyButton()
{
  // convert a QT pixel to a proper world coordinate
  geometry_msgs::Point poiWorldCoordinate = map_tools::fromMapCoords(Eigen::Vector2i(m_PixelSize - 1 - m_TmpPoiPos.y(), m_PixelSize - 1 - m_TmpPoiPos.x()), m_Origin, m_Resolution);
  // poi Name + cast QString to StdString
  std::string poiName = m_PoiNameLineEdit->text().toStdString();

  // poiType
  int poiType = m_PoiSymbolsComboBox->itemData ( m_PoiSymbolsComboBox->currentIndex() ).toInt();

  // poi remarks + cast QString to StdString
  std::string poiRemarks = m_PoiMiscLineEdit->text().toStdString();

  // create actual poi (new or modified) + emit suitable message
  homer_mapnav_msgs::PointOfInterest poi;
  poi.type = poiType;
  poi.name = poiName;
  poi.remarks = poiRemarks;

  poi.pose.position.x = poiWorldCoordinate.x;
  poi.pose.position.y = poiWorldCoordinate.y;
  poi.pose.position.z = 0;
  poi.pose.orientation = tf::createQuaternionMsgFromYaw(m_TmpPoiOrientation);

  if ( m_CurrentPoiSelectionName != "" )
  {
      homer_mapnav_msgs::ModifyPOI mod_poi;
      mod_poi.old_name = m_CurrentPoiSelectionName;
      mod_poi.poi = poi;
      m_ModifyPOIPublisher->publish(mod_poi);
  }
  else
  {
      m_AddPOIPublisher->publish(poi);
  }

  // after apply no poi is selected
  m_CurrentPoiSelectionName = "";

  // update Layout
  deactivatePoiApply();
}

void MapTab::clickedPoiDeleteButton()
{
  deletePoi();
}

void MapTab::clickedMaskButton()
{
    if ( m_CurrentlySelectedMapID < 0 )
    {
      QMessageBox errorBox;
      errorBox.setText( "Select a map layer first!" );
      errorBox.exec();
      return;
    }
    QRadioButton* selectedMaskMode = static_cast<QRadioButton*> ( m_MaskModeButtonGroup->checkedButton() );
    if ( selectedMaskMode )
    {
       m_MapWidget->commitRegion ( selectedMaskMode->property ( "maskAction" ).toInt(), m_CurrentlySelectedMapID );
    }
}

void MapTab::keyPressEvent ( QKeyEvent * event )
{
  if ( event->key() == Qt::Key_Delete )
  {
    deletePoi();
  }
  else if ( event->key() == Qt::Key_Escape )
  {
    m_MapWidget->resetRegion();
    m_MapWidget->clearTmpPoi();
  }
  else if ( event->key() == Qt::Key_Return )
  {
    m_MapWidget->commitRegion ( homer_mapnav_msgs::ModifyMap::SET_BLOCKED , m_CurrentlySelectedMapID);
  }
}

void MapTab::wheelEvent( QWheelEvent* event ) {
event->accept();
  if ( event->delta() > 0 ) { // wheel goes up
    m_MapWidget->zoomIn();
  } else if ( event->delta() < 0 ) { // wheel goes down
    m_MapWidget->zoomOut();
  }
}

void MapTab::deletePoi()
{
  if ( m_CurrentPoiSelectionName != "" )
  {
      homer_mapnav_msgs::DeletePointOfInterest deletePoi;
      deletePoi.name = m_CurrentPoiSelectionName;
      m_DeletePOIPublisher->publish(deletePoi);
    deactivatePoiDelete();
    deactivateDriveToPosition();
  }
}

void MapTab::changedCurrentPoiSelectionByTreeWidget()
{

  QTreeWidgetItem* poiItem = m_PoiTreeWidget->currentItem();
  QTreeWidgetItem* m_CurrentPoiSelectionParentTreeWidgetItem;

  if ( poiItem )
  {
    if ( !poiItem->parent() )
    {
      m_CurrentPoiSelectionParentTreeWidgetItem = poiItem;
    }
    else
    {
      m_CurrentPoiSelectionParentTreeWidgetItem = poiItem->parent();
    }
    QString name = m_CurrentPoiSelectionParentTreeWidgetItem->text(0);
    // fetch the position to the selected tree widget item
    geometry_msgs::Point poiPositionWorld;
    poiPositionWorld.x = ( m_CurrentPoiSelectionParentTreeWidgetItem->child ( 1 )->text ( 1 ) ).toDouble();
    poiPositionWorld.y = ( m_CurrentPoiSelectionParentTreeWidgetItem->child ( 2 )->text ( 1 ) ).toDouble();
   // convert world to map
    Eigen::Vector2i poiPositionMap = map_tools::toMapCoords(poiPositionWorld, m_Origin, m_Resolution);
    // save it for further operations (apply)
    saveTmpPoiPosition ( QPoint ( m_PixelSize - 1 - poiPositionMap.y(), m_PixelSize - 1 - poiPositionMap.x() ) );
    m_TmpPoiOrientation = m_CurrentPoiSelectionParentTreeWidgetItem->child(3)->text(1).toDouble();
    switchToEditMode ( name.toStdString() );
  }
}

//FIXME: there is some error when selecting a poi after a poi was selected, the icon of the new poi gets the old icon
void MapTab::changedCurrentPoiSelectionByMapWidget ( QString name, QString poiIconFileName )
{
  QVectorIterator<PoiIcon > iconIt ( m_PoiIcons );
  int vecIndex = 0;
  while ( iconIt.hasNext() )
  {
    if ( iconIt.next().getFileName() == poiIconFileName )
    {
      m_PoiSymbolsComboBox->setCurrentIndex ( vecIndex );
      m_MapWidget->setCurrentPoiIcon ( poiIconFileName );
      break;
    }
    vecIndex++;
  }

  QList<QTreeWidgetItem *> itemList = m_PoiTreeWidget->findItems ( "Name:", Qt::MatchExactly | Qt::MatchRecursive, 0 );
  QListIterator<QTreeWidgetItem *> it ( itemList );
  while ( it.hasNext() )
  {
    if ( it.peekNext()->parent()->text ( 0 ) == name )
    {
      it.next()->parent()->setSelected ( true );
      continue;
    }
    it.next()->parent()->setSelected ( false );
  }
  switchToEditMode ( name.toStdString() );
}

void MapTab::deactivatePoiApply()
{
  // delete the new poi presentation informations in the MapWidget
  m_MapWidget->clearTmpPoi();

  // update layout
  m_PoiNameLineEdit->clear();
  m_PoiMiscLineEdit->clear();
  m_PoiNameLineEdit->setEnabled ( false );
  m_PoiMiscLineEdit->setEnabled ( false );
  m_PoiApplyButton->setEnabled ( false );
}

void MapTab::deactivatePoiDelete()
{
  // update layout if there isn't any usage for the delete functionality
  m_PoiNameLineEdit->clear();
  m_PoiMiscLineEdit->clear();
  m_PoiNameLineEdit->setEnabled ( false );
  m_PoiMiscLineEdit->setEnabled ( false );
  m_PoiDeleteButton->setEnabled ( false );
}

void MapTab::deactivateDriveToPosition()
{
  // update layout if there isn't any usage for the "drive to position" functionality
  m_DriveToPoiButton->setEnabled ( false );
}

void MapTab::fillPoiTreeWidget()
{

  m_PoiTreeWidget->setCurrentItem ( 0 );
  m_PoiTreeWidget->clear();

  std::vector<homer_mapnav_msgs::PointOfInterest>::iterator it;
  for ( it = m_PoiList.begin(); it != m_PoiList.end(); it++ )
  {

    QTreeWidgetItem* poi = new QTreeWidgetItem ( m_PoiTreeWidget );
    poi->setText ( 0, QString ( it->name.c_str() ) );
    poi->setData ( 0, QAccessible::TreeItem , QString ( it->name.c_str() ) ); //TODO check

    bool found=false;
    for ( int i=0; i<m_PoiIcons.size(); i++ )
    {
      if ( m_PoiIcons[i].getPoiType() == it->type )
      {
        poi->setIcon ( 0, QIcon ( QPixmap ( m_PoiIcons[i].getFileName() ) ) );
                                found=true;
                                break;
      }
    }
    if ( !found ) {
      ROS_ERROR_STREAM("POI icon for POI type " << it->type << " not found!");
      poi->setIcon ( 0, QIcon ( QPixmap ( m_PoiIcons[0].getFileName() ) ) );
    }


    QTreeWidgetItem* poiName = new QTreeWidgetItem ( poi );
    poiName->setFlags ( poiName->flags() & Qt::ItemIsEnabled );
    poiName->setText ( 0, tr ( "Name:" ) );
    poiName->setText ( 1, QString (it->name.c_str()) );

    QTreeWidgetItem* poiX = new QTreeWidgetItem ( poi );
    poiX->setFlags ( poiX->flags() & Qt::ItemIsEnabled );
    poiX->setText ( 0, tr ( "x:" ) );
    poiX->setText ( 1, QString ( QString::number ( it->pose.position.x ) ) );

    QTreeWidgetItem* poiY = new QTreeWidgetItem ( poi );
    poiY->setFlags ( poiY->flags() & Qt::ItemIsEnabled );
    poiY->setText ( 0, tr ( "y:" ) );
    poiY->setText ( 1, QString ( QString::number ( it->pose.position.y ) ) );

    QTreeWidgetItem* poiTheta = new QTreeWidgetItem ( poi );
    poiTheta->setFlags ( poiTheta->flags() & Qt::ItemIsEnabled );
    poiTheta->setText ( 0, tr ( "Theta:" ) );
    poiTheta->setText ( 1, QString ( QString::number ( tf::getYaw(it->pose.orientation) ) ) );

    QTreeWidgetItem* poiRemarks = new QTreeWidgetItem ( poi );
    poiRemarks->setFlags ( poiRemarks->flags() & Qt::ItemIsEnabled );
    poiRemarks->setText ( 0, tr ( "Remarks:" ) );
    poiRemarks->setText ( 1, QString ( it->remarks.c_str() ) );
  }
  m_MapWidget->updatePoiPresentation();
}

void MapTab::switchToEditMode ( std::string name )
{
  // fill the poi name and the poi remarks for editing in the line edit fields
  if ( m_CurrentPoiSelectionName != name )   // if a poi was selected by the tree widget don't override potential changes
  {

    m_MapWidget->resetPoiPresentation();

    std::vector<homer_mapnav_msgs::PointOfInterest>::iterator it;
    for ( it = m_PoiList.begin(); it != m_PoiList.end(); it++ )
    {
      if ( it->name == name )
      {
        m_PoiNameLineEdit->setText ( QString ( it->name.c_str() ) );
        m_PoiNameLineEdit->setEnabled ( true );
        m_PoiMiscLineEdit->setText ( QString ( it->remarks.c_str() ) );
        m_PoiMiscLineEdit->setEnabled ( true );
      }
    }
  }

  m_CurrentPoiSelectionName = name;

  m_MapWidget->clearTmpPoi();
  m_MapWidget->showActivePoiLabelSelection ( name.c_str() );

  m_PoiDeleteButton->setEnabled ( true );
  m_DriveToPoiButton->setEnabled ( true );
  m_StopNavigationButton->setEnabled ( true );
}

void MapTab::showMousePoition( QPoint mousePos ) {
    float x_map = m_PixelSize - mousePos.y();
    float y_map = m_PixelSize - mousePos.x();
    geometry_msgs::Point mousePosWorld = map_tools::fromMapCoords(Eigen::Vector2i(x_map,y_map), m_Origin, m_Resolution);
  m_ShowMousePosition->setText( "Mouse position: " + QString ( QString::number( mousePosWorld.x ) ) + "," + QString ( QString::number( mousePosWorld.y ) ) );
}

void MapTab::showZoomFactor( float zoomFactor ) {
  int zoom = int( ( zoomFactor * 100 ) + 0.5 );

  m_ShowZoomFactor->setText( "Zoom factor: " +  QString::number( zoom ) + "%" );
}

void MapTab::clickedDriveToPositionButton()
{
    std_msgs::Empty stop_msg;
    //m_StopNavigationPublisher->publish(stop_msg);
    geometry_msgs::Point poiWorldCoordinate = map_tools::fromMapCoords(Eigen::Vector2i(m_PixelSize - 1 - m_TmpPoiPos.y(), m_PixelSize - 1 - m_TmpPoiPos.x()), m_Origin, m_Resolution);

    homer_mapnav_msgs::StartNavigation start_msg;
    start_msg.goal.position.x = poiWorldCoordinate.x;
    start_msg.goal.position.y = poiWorldCoordinate.y;
    start_msg.goal.position.z = 0.0;
    start_msg.goal.orientation = tf::createQuaternionMsgFromYaw(m_TmpPoiOrientation);
    start_msg.distance_to_target = 0.1;
    m_StartNavigationPublisher->publish(start_msg);
    ROS_INFO_STREAM("Navigating to point " << poiWorldCoordinate.x << ", " << poiWorldCoordinate.y);
}

void MapTab::clickedStopNavigationButton()
{
    std_msgs::Empty stop_msg;
    m_StopNavigationPublisher->publish(stop_msg);
}

void MapTab::centerMap ( QPoint xy )
{
  m_ScrollArea->horizontalScrollBar()->setSliderPosition ( xy.x() - ( m_ScrollArea->horizontalScrollBar()->pageStep() / 2 ) );
  m_ScrollArea->verticalScrollBar()->setSliderPosition ( xy.y() - ( m_ScrollArea->verticalScrollBar()->pageStep() / 2 ) );
}

void MapTab::toggleRobotView ( int state )
{
  m_MapWidget->toggleRobotView ( ( bool ) state );
}

void MapTab::togglePoiView ( int state )
{
  m_MapWidget->togglePoiView ( ( bool ) state );
}

void MapTab::toggleDoMappingState ( int state )
{
  std_msgs::Bool msg;
  msg.data = (bool)state;
  m_DoMappingPublisher->publish(msg);
}

void MapTab::updatePose(double x, double y, double theta)
{
    m_LastPose.position.x = x;
    m_LastPose.position.y = y;
    m_LastPose.position.z = 0.0;
    m_LastPose.orientation = tf::createQuaternionMsgFromYaw(theta);

    m_MapWidget->updatePose(x, y, theta);
}

bool operator < ( const std::pair < std::pair < int, int> , int>& left, const std::pair < std::pair < int, int> , int> & right )
{
  return left.second < right.second;
}

void MapTab::updatePOIList(std::vector<homer_mapnav_msgs::PointOfInterest> poiList)
{
    m_PoiList = poiList;
    fillPoiTreeWidget();
}

void MapTab::updatePath(std::vector<geometry_msgs::PoseStamped> path)
{
    m_MapWidget->printPath(path);
}

void MapTab::updateFollowingPath(std::vector<geometry_msgs::PoseStamped> path)
{
    m_MapWidget->printFollowingPath(path);
}

void MapTab::updateMap(unsigned char * map, uint size, double resolution, geometry_msgs::Pose origin)
{
    m_MapWidget->updateMap(MapDisplay::SlamMap, map, size);
    m_Resolution = resolution;
    m_Origin = origin;
    m_PixelSize = size;
    m_MapWidget->setOrigin(origin);
    m_MapWidget->setResolution(resolution);
    m_MapWidget->setPixelSize(size);
}

void MapTab::updateMask(const nav_msgs::OccupancyGrid::ConstPtr &msg)
{
    m_MapWidget->updateMask(msg);
}

void MapTab::toggleMapLayerVisibility(QListWidgetItem* item)
{
    homer_mapnav_msgs::MapLayers msg;
    msg.layer = item->data(Qt::UserRole).toInt();
    if(item->checkState() == Qt::Checked)
    {
        msg.state = true;
        m_MapVisibilityPublisher->publish(msg);
        ROS_INFO_STREAM(item->text().toStdString() << " enabled");
    }
    else
    {
        msg.state = false;
        m_MapVisibilityPublisher->publish(msg);
        ROS_INFO_STREAM(item->text().toStdString() << " disabled");
    }
}

void MapTab::changeCurrentMapLayer()
{
    m_CurrentlySelectedMapID = m_MapLayersListWidget->currentRow();
}
