/*******************************************************************************
 *  MyStateMachineModule.cpp
 *
 *  (C) 2009 AG Aktives Sehen <agas@uni-koblenz.de>
 *           Universitaet Koblenz-Landau
 *
 * $Id: $
 *******************************************************************************/

#include "DetectOpeningDoorModule.h"

#include <robbie_architecture/Architecture/Tracer/Tracer.h>
#include <robbie_architecture/Architecture/Config/Config.h>
#include <robbie_architecture/Architecture/Singleton/Clock.h>

#include "Messages/DetectOpeningDoorM.h"
#include "Messages/SpeechOutM.h"
#include "Messages/StartGameM.h"
#include "Messages/LaserDataM.h"

#include <sstream>

using namespace std;

#define THIS DetectOpeningDoorModule

THIS::THIS()
{
  subscribeTo( MessageTypes::DETECT_OPENING_DOOR_M, MessageHandling::KeepOnlyNewest );
  subscribeTo( MessageTypes::LASER_DATA_M, MessageHandling::KeepOnlyNewest );
}

THIS::~THIS()
{
}


void THIS::init()
{
  //Read configuration etc.
  //Messages can already be sent here

  //setIdleInterval( millisec );

  //Register all possible machine states
  ADD_MACHINE_STATE( m_ModuleMachine, IDLE );
  ADD_MACHINE_STATE( m_ModuleMachine, WAITING );
  ADD_MACHINE_STATE( m_ModuleMachine, DOOR_IS_OPEN );
  
  m_ModuleMachine.setName( "Module State" );
  m_ModuleMachine.setState( IDLE );
}


set< Message* > THIS::processMessages()
{
  set<Message* > deadMessages;
  Message* newMessage = 0;
  bool foundMessage = false;
  
  newMessage = inbox( MessageTypes::DETECT_OPENING_DOOR_M );
  if ( newMessage )
  {
    foundMessage = true;
    if ( DetectOpeningDoorM* detectOpeningDoorM = Message::castTo<DetectOpeningDoorM>( newMessage ) )
    {
      TRACE_INFO( "Received DetectOpeningDoorM" );
      sendMessage(new SpeechOutM("Knock, knock!"));
      m_GameTime = detectOpeningDoorM->getGameTime();
      m_CurrentTimestamp = detectOpeningDoorM->getTimestamp();
      m_ModuleMachine.setState( WAITING );
    }
    deadMessages.insert( newMessage );
  }
  newMessage = inbox( MessageTypes::LASER_DATA_M );
  if ( newMessage )
  {
    foundMessage = true;
    if ( LaserDataM* laserDataM = Message::castTo<LaserDataM>( newMessage ) )
    {
      m_Ranges.clear();
      
      switch ( m_ModuleMachine.state() )
      {
        case IDLE:
        {
          if ( laserDataM->getScannerConfig()->getName() == "SickLMS100" )
          {
            m_Ranges = laserDataM->getRangeVector();
            int ranges_center = ( m_Ranges.size() / 2 ) + 1;
            //- m_CurrentRange = m_Ranges.at( ranges_center );
            m_CurrentRange = (m_Ranges.at( ranges_center -1 )        // mittelwert der mittleren 3 laser, noiseresistent
                              + m_Ranges.at( ranges_center )
                              + m_Ranges.at( ranges_center +1 ))/3;

          }
          break;
        }
        case WAITING:
        {
          string LRFName = laserDataM->getScannerConfig()->getName();
          m_NewTimestamp = laserDataM->getTimestamp();
          if ( LRFName == "SickLMS100" && m_NewTimestamp > m_CurrentTimestamp + 500 )
          {
            m_Ranges = laserDataM->getRangeVector();
            compareRanges();
          }
          break;
        }
        default:
	  break;
      }
    }
    deadMessages.insert( newMessage );
  }

  if ( !foundMessage ) { TRACE_ERROR( "Could not identify new message!" ) }
  return deadMessages;
}

void THIS::compareRanges()
{
  switch (m_ModuleMachine.state() )
  {
    case WAITING:
    {
      int ranges_center = ( m_Ranges.size() / 2 ) + 1; 
      //- m_NewRange = m_Ranges.at( ranges_center );
      m_NewRange = (m_Ranges.at( ranges_center -1 )        // mittelwert der mittleren 3 laser, noiseresistent
          + m_Ranges.at( ranges_center )
          + m_Ranges.at( ranges_center +1 ))/3;
      TRACE_INFO( "NewRange: " << m_NewRange );
      m_CurrentTimestamp = m_NewTimestamp;
      
      if (( m_NewRange > m_CurrentRange + 1000 ) && ( m_CurrentRange <= 2000)) // Tür nicht weiter als 2m weg, um hintergrundbewegungen zu ignorieren
      {
        usleep( 1000000 );
        TRACE_INFO ( "The door is open!" );
        sendMessage( new SpeechOutM( "The door is now open" ) );
        sendMessage( new StartGameM( m_GameTime ) );
        m_ModuleMachine.setState( DOOR_IS_OPEN );
      }
      else if (m_NewRange < m_CurrentRange)   // falls tür nach dem startbutton geschlossen wird
      {
        m_CurrentRange = m_NewRange;
      }
      break;
    }  
    default:
      break;
  }
}


void THIS::idleProcess()
{
  //Do something useful here, e.g. check for errors
  string errorMessage;

  switch ( m_ModuleMachine.state() )
  {
    //note: avoid using the 'default' case on state machines,
    //      as it easily leads to errors when adding new states
    case WAITING:
    case DOOR_IS_OPEN:
      break;

    case IDLE:
      //Check if the state machine got stuck
      /*if ( m_ModuleMachine.timeSinceStateChange() > 30000 )
      {
        errorMessage="DetectOpeningDoorM did not arrive for more than 30 sec";
      }*/
      break;
  }

  if ( errorMessage != "" )
  {
    TRACE_ERROR( errorMessage );
    m_ExtraStatusInfo = "\n" + errorMessage;
    m_ModuleMachine.setState( IDLE );
  }
}


void THIS::refreshStatusInfo()
{
  setStatusInfo ( m_ModuleMachine.stateString() + m_ExtraStatusInfo );
}


#undef THIS
