/*******************************************************************************
 *  TalkingHead - A talking head for robots
 *  Copyright (C) 2014 AG Aktives Sehen <agas@uni-koblenz.de>
 *                     Universitaet Koblenz-Landau
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 *  MA 02110-1301  USA or see <http://www.gnu.org/licenses/>.
 *******************************************************************************/

#include "ImageDisplay.h"
#include <sensor_msgs/Image.h> //for kinect RGB stream
#include <QPainter>
#include <cv.h>

#define MAX_QUEUE_SIZE 2 //maximum number of images to be buffered
ImageDisplay::ImageDisplay(QWidget *parent):
    QWidget(parent),
    window_rotation_(0)
{

}

ImageDisplay::ImageDisplay(const int window_rotation, QWidget *parent):
    QWidget( parent ),
    window_rotation_(window_rotation),
    image_queue_(),
    reset_timer_(new QTimer( this ))  // create internal timer
{
    this->setVisible( false );
    reset_timer_->setSingleShot( true );

    connect( reset_timer_, SIGNAL ( timeout() ), SLOT ( clearImage() ) );
    //connect( this, SIGNAL ( newImageToShow(unsigned int) ), SLOT ( showImage(unsigned int) ) );
}
ImageDisplay::~ImageDisplay()
{
    while(image_queue_.size() > 0)
    {
        pop_image_from_queue();
    }
    //note: reset_timer will be deleted by parent
}

void ImageDisplay::pop_image_from_queue()
{

    QImage* image = image_queue_.front();
    image_queue_.pop();
    if(image) delete image;
}

void ImageDisplay::showImage(const unsigned int milliseconds)
{

    this->show();
    update();
    reset_timer_->start( milliseconds);
}

void ImageDisplay::clearImage()
{
    this->hide();
    pop_image_from_queue();
}

void ImageDisplay::paintEvent(QPaintEvent*)
{
    QPainter qpainter(this);

    switch(image_queue_.size())
    {
    case 0:
        //no image to paint
        update();
        return;
    case 1:
        //if there is only one element in the queue we do not remove it here, because we have to wait until the timer fires;
        update();
        rotateAndDrawImage( &qpainter, image_queue_.front() );

        break;
    default:
        //more than one image in the queue, paint the image at the front of the queue and pop it then
        rotateAndDrawImage( &qpainter, image_queue_.front() );
        update();
        pop_image_from_queue();
        break;
    }
}

void ImageDisplay::rotateAndDrawImage(QPainter *painter, const QImage *toDraw )
{
    painter->save();
    QMatrix matrix;

    switch( window_rotation_ )
    {
    case 0:
        matrix.translate(0, 0);
        matrix.rotate(window_rotation_);
        painter->setMatrix(matrix);
        painter->drawImage(0, 0, *toDraw);
        break;
    case 90:
        matrix.translate(width(), 0);
        matrix.rotate(window_rotation_);
        painter->setMatrix(matrix);
        painter->drawImage(0, 0, *toDraw);
        break;
    case 180:
        matrix.translate(width(), height());
        matrix.rotate(window_rotation_);
        painter->setMatrix(matrix);
        painter->drawImage(0, 0, *toDraw);
        break;
    case 270:
        matrix.translate(0, height());
        matrix.rotate(window_rotation_);
        painter->setMatrix(matrix);
        painter->drawImage(0, 0, *toDraw);
        break;
    default:
        ROS_WARN("ImageDisplay::rotateAndDrawImage() - text_rotation type not 0, 90, 180 or 270");
        break;
    }
    painter->restore();
}


///Convert function from http://asmaloney.com/2013/11/code/converting-between-cvmat-and-qimage-or-qpixmap/
inline QImage ImageDisplay::cvMatToQImage( const cv::Mat &inMat )
{
    switch ( inMat.type() )
    {
    // 8-bit, 4 channel
    case CV_8UC4:
    {
        QImage image( inMat.data, inMat.cols, inMat.rows, inMat.step, QImage::Format_RGB32 );
        return image;
    }
        // 8-bit, 3 channel
    case CV_8UC3:
    {
        QImage image( inMat.data, inMat.cols, inMat.rows, inMat.step, QImage::Format_RGB888 );
        return image.rgbSwapped();
    }
        // 8-bit, 1 channel
    case CV_8UC1:
    {
        static QVector<QRgb>  sColorTable;

        // only create our color table once
        if ( sColorTable.isEmpty() )
        {
            for ( int i = 0; i < 256; ++i )
                sColorTable.push_back( qRgb( i, i, i ) );
        }

        QImage image( inMat.data, inMat.cols, inMat.rows, inMat.step, QImage::Format_Indexed8 );
        image.setColorTable( sColorTable );
        return image;
    }
    default:
        ROS_WARN_STREAM("ImageDisplay::cvMatToQImage() - cv::Mat image type not handled in switch: " << inMat.type());
        break;
    }
    return QImage();
}


void ImageDisplay::processImageData(const sensor_msgs::Image::ConstPtr& msg)
{
    if(image_queue_.size() < MAX_QUEUE_SIZE){
        //Get image into OpenCV format..
        cv_bridge::CvImagePtr sensor_img_cv;
        try
        {
            //TODO:msg->encoding="mono8" for debug_gray display delivers wrong images
            sensor_img_cv = cv_bridge::toCvCopy(msg,"bgr8");


        }
        catch (cv_bridge::Exception& e)
        {
            ROS_ERROR("cv_bridge exception: %s", e.what());
            return;
        }
        //scale image to fit in (max_image_width_, max_image_height_)
        cv::Mat resized_img_mat;

        unsigned int width;
        unsigned int height;



        if(sensor_img_cv->image.cols > sensor_img_cv->image.rows)
        {
            cv::resize( sensor_img_cv->image, resized_img_mat, cv::Size(sensor_img_cv->image.cols, this->height()),0,0, cv::INTER_AREA);
        }

        if(sensor_img_cv->image.cols >this->height())
        {
            width = this->width();
        } else {
            width = sensor_img_cv->image.cols;
        }
        if(sensor_img_cv->image.rows >this->width())
        {
            height = this->height();
        } else {
            height = this->height();
        }
        float aspectRatio=(float)sensor_img_cv->image.rows/sensor_img_cv->image.cols;
        if(width>height/aspectRatio){
            cv::resize( sensor_img_cv->image, resized_img_mat, cv::Size(height/aspectRatio, height),0,0, cv::INTER_AREA);

        }
        else if(height>width*aspectRatio){
            cv::resize( sensor_img_cv->image, resized_img_mat, cv::Size(width,width*aspectRatio),0,0, cv::INTER_AREA);

        }
        else{
            //CV_INTER_AREA looks best for shrinked images
            cv::resize( sensor_img_cv->image, resized_img_mat, cv::Size(width, height),0,0, cv::INTER_AREA);
        }
        //cv::imshow("test",sensor_img_cv->image);
        // and from OpenCV to QImage
        QImage* image_converted = new QImage( cvMatToQImage(resized_img_mat) );
        image_queue_.push(image_converted);
    }

}



#undef MAX_QUEUE_SIZE
