/*******************************************************************************
 *  Homer GUI
 *  Copyright (C) 2014 AG Aktives Sehen <agas@uni-koblenz.de>
 *                     Universitaet Koblenz-Landau
 *******************************************************************************/



#include <tf/tf.h>  //has to be called before QtRosNode.h, reason is unclear for me (M. v. Steimker)
#include "QtRosNode.h"
//messages:
#include <std_msgs/String.h>
#include <geometry_msgs/PoseStamped.h>
#include <nav_msgs/OccupancyGrid.h>
#include <homer_mapnav_msgs/DeletePointOfInterest.h>
#include <stdlib.h>

QtRosNode::QtRosNode(int argc, char *argv[], const char* node_name, MainWindow* mainWindow)
{
    //ros::init(argc, argv, node_name);
    ros::start();
    m_NodeHandle = new ros::NodeHandle;
    m_MainWindow = mainWindow;

    //m_Application = app;
    quitfromgui = false;

	advertiseTopics();
	subscribeToTopics();
    m_LastLaserStateUpdateTime = ros::Time::now();


    start();


    // Door Detection SM
    ADD_MACHINE_STATE( m_ModuleMachine, IDLE );
    ADD_MACHINE_STATE( m_ModuleMachine, WAITING );
    ADD_MACHINE_STATE( m_ModuleMachine, DOOR_IS_OPEN );

    m_ModuleMachine.setName( "Module State" );
    m_ModuleMachine.setState( IDLE );
    //end Door Detection

}

QtRosNode::~QtRosNode()
{
    if( m_NodeHandle ) delete m_NodeHandle;
    if(ros::isStarted()) {
        ros::shutdown(); // explicitly needed since we use ros::start();
        ros::waitForShutdown();
    }
    wait();
}


void QtRosNode::compareRanges()
{
    //ROS_INFO_STREAM("Compare Ranges");
    switch (m_ModuleMachine.state() )
    {
    case WAITING:
    {
        int ranges_center = ( m_Ranges.size() / 2 ) + 1;
        //- m_NewRange = m_Ranges.at( ranges_center );
        m_NewRange = (m_Ranges.at( ranges_center -1 )        // mittelwert der mittleren 3 laser, noiseresistent
                      + m_Ranges.at( ranges_center )
                      + m_Ranges.at( ranges_center +1 ))/3;
        ROS_DEBUG_STREAM( "NewRange: " << m_NewRange << " " << m_CurrentRange );
        m_CurrentTimestamp = m_NewTimestamp;

        if (( m_NewRange > m_CurrentRange +1 ) && ( m_CurrentRange <= 2)) // Tür nicht weiter als 2m weg, um hintergrundbewegungen zu ignorieren
        {
            usleep( 1000000 );
            ROS_INFO_STREAM( "The door is open!" );
            speak("The door is now open");
            //sendMessage( new SpeechOutM( "The door is now open" ) );
            //sendMessage( new StartGameM( m_GameTime ) );
            publishStartGame();
            m_ModuleMachine.setState( DOOR_IS_OPEN );
        }
        else if (m_NewRange < m_CurrentRange)   // falls tür nach dem startbutton geschlossen wird
        {
            m_CurrentRange = m_NewRange;
        }
        break;
    }
    default:
        break;
    }
}


void QtRosNode::subscribeToTopics()
{
    //    // subscribe to all topics here
    m_LaserSubscriber = m_NodeHandle->subscribe<sensor_msgs::LaserScan>("/scan", 10, &QtRosNode::callbackLaserScan, this);
    m_KinectRGBImageSubscriber = m_NodeHandle->subscribe<sensor_msgs::Image>("/camera/rgb/image_rect_color", 1, &QtRosNode::callbackKinectRGBImageDisplay, this);
    m_DebugGrayImageSubscriber = m_NodeHandle->subscribe<sensor_msgs::Image>("/or/debug_image_gray", 1, &QtRosNode::callbackDebugGrayDisplay, this);
    m_PrimaryRGBImageSubscriber = m_NodeHandle->subscribe<sensor_msgs::Image>("/or/obj_learn_primary_color", 1, &QtRosNode::callbackPrimaryRGBDisplay, this);

    m_AnyImageSubscriber = m_NodeHandle->subscribe<sensor_msgs::Image>("/image", 1, &QtRosNode::callbackAnyImageDisplay, this);
    m_PoseStampedSubscriber = m_NodeHandle->subscribe<geometry_msgs::PoseStamped>("/pose", 10, &QtRosNode::callbackPoseStamped, this);
    m_MapDataSubscriber = m_NodeHandle->subscribe<nav_msgs::OccupancyGrid>("/map", 10, &QtRosNode::callbackOccupancyGrid, this);
    m_MaskSlamSubscriber = m_NodeHandle->subscribe<nav_msgs::OccupancyGrid>("/map_manager/mask_slam", 1, &QtRosNode::callbackMaskSlam, this );
    m_POIsSubscriber = m_NodeHandle->subscribe<homer_mapnav_msgs::PointsOfInterest>("/map_manager/poi_list", 10, &QtRosNode::callbackPOIList, this);
    m_PathSubscriber = m_NodeHandle->subscribe<nav_msgs::Path>("/homer_navigation/path", 10, &QtRosNode::callbackPath, this);
    m_FollowingPathSubscriber = m_NodeHandle->subscribe<nav_msgs::Path>("/following_node/path", 10, &QtRosNode::callbackFollowingPath, this);

    door_detection_sub_ = m_NodeHandle->subscribe<std_msgs::Empty>("/detect_doors", 10, &QtRosNode::callbackDetectDoors, this);
    emergeny_button_pressed_sub_ = m_NodeHandle->subscribe<std_msgs::Bool>("/emergency_button_pressed", 10, &QtRosNode::callbackEmergencyButton, this);

    //m_OLPrimaryImageSubscriber = m_NodeHandle->subscribe<homer_mapnav_msgs::PointsOfInterest>("/or/obj_learn_primary", 10, &QtRosNode::callbackPOIList, this);
    //m_OLPrimaryImageSubscriber = m_NodeHandle->subscribe<sensor_msgs::Image>("/or/obj_learn_primary", 10, &QtRosNode::callbackGUIImage, this);
    //m_OLPrimaryImageSubscriber = m_NodeHandle->subscribe<or_msgs::OrImage>("/or/obj_learn_primary", 10, &QtRosNode::callbackPrimaryImage, this);
    m_ORLearningStatusSubscriber = m_NodeHandle->subscribe<or_msgs::OrLearningStatus>("/or/learning_status", 10, &QtRosNode::callbackLearningStatus, this);

    m_ORObjectNamesSubscriber = m_NodeHandle->subscribe<or_msgs::OrObjectNames>("/or/obj_names", 10, &QtRosNode::callbackObjectNames, this);

    //m_HardwareStatusSubscriber = m_NodeHandle->subscribe<hardware_state::HardwareState>("/hardware_state",10, &QtRosNode::callbackHardwareStatus, this);

    //m_HardwareTimer = m_NodeHandle->createTimer(ros::Duration(0.5), &QtRosNode::hardwareTimerCallback, this);
}

void QtRosNode::publishStartGame()
{
    std_msgs::Empty msg;
    start_game_pub_.publish(msg);
}

void QtRosNode::advertiseTopics()
{
    //    // TODO advertise topics here or find other solution
    //    ros::Publisher* text_publisher = new ros::Publisher(m_NodeHandle->advertise<std_msgs::String>("chatter", 10));
    //    m_MainWindow->setTextPublisher(text_publisher);
    //    ros::Publisher* pose2D_publisher = new ros::Publisher(m_NodeHandle->advertise<geometry_msgs::Pose2D>("/userdef_pose", 10));
    //    m_MainWindow->setPose2DPublisher(pose2D_publisher);
    //    ros::Publisher* addPointOfInterest_publisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::PointOfInterest>("/add_POI", 10));
    //    m_MainWindow->setAddPOIPublisher(addPointOfInterest_publisher);
    //    ros::Publisher* modifyPointOfInterest_publisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::PointOfInterest>("/modify_POI", 10));
    //    m_MainWindow->setModifyPOIPublisher(modifyPointOfInterest_publisher);
    //    ros::Publisher* deletePointOfInterest_publisher = new ros::Publisher(m_NodeHandle->advertise<homer_mapnav_msgs::DeletePointOfInterest>("/delete_POI", 10));
    //    m_MainWindow->setDeletePOIPublisher(deletePointOfInterest_publisher);

    start_game_pub_ = m_NodeHandle->advertise<std_msgs::Empty>("/start_game", 1);
    text_out_pub_ = m_NodeHandle->advertise<std_msgs::String>("/robot_face/speak", 1);

}


void QtRosNode::quitNow()
{
    quitfromgui = true;
}


void QtRosNode::run()
{
    ros::Rate loop_rate(3);

    while (ros::ok())
    {
        ros::spinOnce();
        loop_rate.sleep();
    }
    emit rosShutdown();

}

void QtRosNode::callbackPoseStamped(const geometry_msgs::PoseStamped::ConstPtr &msg)
{
    double theta = tf::getYaw(msg->pose.orientation);
    emit poseUpdated(msg->pose.position.x, msg->pose.position.y, theta);
}

void QtRosNode::callbackOccupancyGrid(const nav_msgs::OccupancyGrid::ConstPtr &msg)
{
    unsigned char* map = new unsigned char[msg->data.size()];
    for(int x = 0; x < msg->info.height; x++)
    {
        int xOffset = msg->info.height * x;
        for(int y = 0; y < msg->info.width; y++)
        {
            int i = xOffset + y;
            map[i] = msg->data[i] == -1 ? 50 : msg->data[i];
        }
    }

    emit mapUpdated(map, msg->info.width, msg->info.resolution, msg->info.origin);
}

void QtRosNode::callbackMaskSlam(const nav_msgs::OccupancyGrid::ConstPtr &msg)
{
    emit maskUpdated(msg);
}

void QtRosNode::callbackPOIList(const homer_mapnav_msgs::PointsOfInterest::ConstPtr& msg)
{
    emit poiListUpdated(msg->pois);

}

void QtRosNode::callbackPath(const nav_msgs::Path::ConstPtr& msg)
{
    emit pathUpdated(msg->poses);
}

void QtRosNode::callbackFollowingPath(const nav_msgs::Path::ConstPtr& msg)
{
    emit followingPathUpdated(msg->poses);
}

void QtRosNode::callbackEmergencyButton(const std_msgs::Bool::ConstPtr& msg)
{
    //ROS_INFO_STREAM("Emergency Button pressed");
    emit emergencyButtonPressed(msg->data);
}

void QtRosNode::callbackKinectRGBImageDisplay(const sensor_msgs::Image::ConstPtr& msg)
{
    emit kinectRGBImageUpdate(msg);
}
void QtRosNode::callbackDebugGrayDisplay(const sensor_msgs::Image::ConstPtr& msg)
{
    emit kinectDebugGrayUpdate(msg);

}
void QtRosNode::callbackPrimaryRGBDisplay(const sensor_msgs::Image::ConstPtr& msg)
{
    emit kinectPrimaryRGBUpdate(msg);
}

void QtRosNode::callbackAnyImageDisplay(const sensor_msgs::Image::ConstPtr &msg)
{
    emit anyImageUpdate(msg);
}

//void QtRosNode::callbackPrimaryImage(const or_msgs::OrImage::ConstPtr &msg ) {
//    ROS_INFO_STREAM("callbackPrimaryImage");
//}

//void QtRosNode::callbackGUIImage(const sensor_msgs::Image::ConstPtr& msg)
//{

//}


void QtRosNode::callbackLaserScan(const sensor_msgs::LaserScan::ConstPtr& msg)
{
    m_Ranges.clear();
    switch ( m_ModuleMachine.state() )
    {
    case IDLE:
    {
        //if ( laserDataM->getScannerConfig()->getName() == "SickLMS100" )
        //{
        m_Ranges = msg->ranges;//laserDataM->getRangeVector();
        int ranges_center = ( m_Ranges.size() / 2 ) + 1;
        //- m_CurrentRange = m_Ranges.at( ranges_center );
        m_CurrentRange = (m_Ranges.at( ranges_center -1 )        // mittelwert der mittleren 3 laser, noiseresistent
                          + m_Ranges.at( ranges_center )
                          + m_Ranges.at( ranges_center +1 ))/3;

        //}
        break;
    }
    case WAITING:
    {
        //string LRFName = laserDataM->getScannerConfig()->getName();
        m_NewTimestamp = msg->header.stamp;//laserDataM->getTimestamp();
        //if ( LRFName == "SickLMS100" &&
        //if ( m_NewTimestamp > m_CurrentTimestamp + 500 ) // TODO offset half a second
        {
            m_Ranges = msg->ranges;//laserDataM->getRangeVector();
            compareRanges();
        }
        break;
    }
    default:
        break;
    }

    // update SickLRF hardware state
    //if((ros::Time::now() - m_LastLaserStateUpdateTime).toSec() > 5)
    //{
        //emit hardwareStatus("SickLRF", true);
        //m_LastLaserStateUpdateTime = ros::Time::now();
		////std::system("xterm -e roslaunch lms1xx lms1xx.launch");
    //}





    //    if(m_SensorDataDisplay)
    //    {
    //         NewLaserDataPainter* painter = dynamic_cast<NewLaserDataPainter*> (m_SensorDataDisplay->getSensorDataGLWidget()->getPainter("2D Laser Data"));
    //        // TODO convert laser data to world coordinates here or in painter
    //        if(painter)
    //        {
    //            painter->updateData(msg);
    //        }
    //        else
    //        {
    //            ROS_WARN_STREAM("Could not get Painter Object \"2D Laser Data\"");
    //        }
    //    }

}


void QtRosNode::speak(std::string text)
{
    std_msgs::String msg;
    msg.data = text;
    if( ros::ok() )
        text_out_pub_.publish( msg );
}

void QtRosNode::callbackLearningStatus(const or_msgs::OrLearningStatusConstPtr &msg)
{
    emit learningStatus(msg->image_names, msg->object_type);
}

void QtRosNode::callbackObjectNames(const or_msgs::OrObjectNamesConstPtr &msg)
{
    emit objectNames(msg->object_names, msg->object_types);
}

//void QtRosNode::callbackHardwareStatus(const hardware_state::HardwareState::ConstPtr &msg)
//{
    //emit hardwareStatus(msg->deviceName, msg->isOk);
//}

void QtRosNode::callbackDetectDoors(const std_msgs::Empty::ConstPtr& msg)
{
    m_ModuleMachine.setState(WAITING);
    speak("Knock Knock");
}

//void QtRosNode::hardwareTimerCallback(const ros::TimerEvent&)
//{
    //emit checkHardwareStatus();
//}
