#ifndef MAPDISPLAY_H
#define MAPDISPLAY_H

#include <list>
#include <iostream>

#include <ros/ros.h>

/* Qt related includes */
#include <QtCore/QVector>
#include <QtCore/QPoint>
#include <QtCore/QString>

#include <QtGui/QApplication>
#include <QtGui/QWidget>
#include <QtGui/QPainter>
#include <QtGui/QMouseEvent>
#include <QtGui/QRgb>
#include <QtGui/QMessageBox>

#include "PoiIcon.h"
#include "PoiLabel.h"

#include <Eigen/Geometry>
//#include "Workers/Semantics/Location/Area.h"

#include <nav_msgs/Path.h>
#include <homer_mapnav_msgs/PointOfInterest.h>
#include <nav_msgs/OccupancyGrid.h>

/**
 * @author Malte Knauf (2014), Christian Feinen (RX), David Gossow (RX)
 * This widget displays the current map sent from the map_manager node.
 * Additionally all points of interest will be shown and the current robot pose.
 * @note The QT coordinate system has its origin in the upper left corner, whereas
 * the ROS coordinate system is in the lower right corner. Also x and y are swapped.
 * To convert simply invert the coordinates, i.e.
 *      ros_x = map_width  - 1 - qt_y,
 *      ros_y = map_height - 1 - qt_x
 */
class MapDisplay : public QWidget
{
    Q_OBJECT

  public:

    /**
     * @brief   Identifies the different maps.
     * @warning Also add a description string to m_MapNames when adding a new ID
     */
    enum MapId {
      SlamMap,
      Obstacle3dMap,
      MaskingMap,
      SickMap
    };

    MapDisplay ( ros::NodeHandle *nodeHandle, QWidget *parent = 0 );

    ~MapDisplay();

    void updatePose ( double x, double y, double theta );

    void setRobotPos ( QPoint position, double angle );
    void setOrigin( geometry_msgs::Pose origin) { m_Origin = origin; }
    void setResolution( float resolution ) {m_Resolution = resolution; }
    void setPixelSize( int size ) { m_PixelSize = size; }
    double computeOrientationAngle ( QPoint point, QPoint orientation );
    void linkPoiList ( std::vector<homer_mapnav_msgs::PointOfInterest >& poiList );
    void linkPoiIconList ( QVector< PoiIcon >& poiIconList );
    void setCurrentPoiIcon ( QString poiIconFileName );
    void clearTmpPoi();
    void toggleRobotView ( bool visibility );
    void togglePoiView ( bool visibility );

    void printPath(std::vector<geometry_msgs::PoseStamped> path);
    void printFollowingPath(std::vector<geometry_msgs::PoseStamped> path);

    // Added by Carmen Navarro Luzón, MasterArbeit 2011
    void toggleRoomView ( bool visibility );
    void toggleArtifactView ( bool visibility );
    void toggleInteractiveArtView ( bool visibility );
    std::vector<geometry_msgs::Point> getPolygonFromRegion();
    std::vector<geometry_msgs::Point> commitRoom();
    geometry_msgs::Point commitArtifact();

    geometry_msgs::Point getSelectedPOI();
    void updateSelectedItemIndex(unsigned i);
    void updateSelectedLayer(unsigned i);
    void deleteRoomByIndex(unsigned i);
    void deleteArtifactByIndex(unsigned i);
    void printQPolygon(QPolygon p);
    void printQPoint(QPoint p);


    void showActivePoiLabelSelection ( QString name );
    void resetPoiPresentation();
    void commitRegion ( int maskAction, int mapLayer );
    const QImage* getMapImagePtr();

    float zoomIn();
    float zoomOut();

    virtual QSize minimumSizeHint();
    virtual QSize sizeHint();

    std::map< int, std::string > getMapNames() { return m_MapNames; }

    static const unsigned char RED = 101; //for laser visualization
    static const unsigned char INACCESSIBLE = 100;
    static const unsigned char OBSTACLE = 99;
    static const unsigned char OCCUPIED = 98;
    static const unsigned char UNKNOWN = 50;
    static const unsigned char FREE = 0;

  public slots:
    /** @brief replace map image */
    void updateMap ( MapId mapId, unsigned char *pMap, unsigned size);
    void updateMask ( const nav_msgs::OccupancyGrid::ConstPtr &msg);


    void resetRegion();
    void resetRobotPos();
    void updatePoiPresentation();

    void selectMap( int mapId );
    void toggleAntiAliasing( int state );
    void toggleMovablePoi( int state );

    void toggleGridVisibility( int state );

  signals:

    void robotPositionChanged( QPoint robotPos, double robotOrientation );
    void addedNewPoi( QPoint addedNewPoiPos );
    void changedMousePosition( QPoint mousePos );
    void modifiedPoi( QString name, QString poiIconFileName );
    void modifiedPoiPosition( QPoint newPoiPos );
    void modifiedPoiOrientation( double theta);
    void centerMap( QPoint robotPos );
    void changedZoomFactor( float zoomFactor );

    void mapSelected( int mapId );

  private:



    /** @brief paint a neat checker texture into the unknown areas */
    void paintCheckerTexture ( QImage& mapImage );
    /** @brief paint a 1x1m grid on top of the map */
    void paintGrid( QImage& mapImage );

    QString getPoiIconFileName ( int poiType );

    virtual void paintEvent ( QPaintEvent* );
    virtual void mousePressEvent ( QMouseEvent* event );
    virtual void mouseMoveEvent ( QMouseEvent* event );
    virtual void mouseReleaseEvent ( QMouseEvent* event );

    MapId m_CurrentMap;

    /** map parameters used to convert world coordinates to map cells */
    float m_Resolution;
    geometry_msgs::Pose m_Origin;
    int m_PixelSize;

    //copy of incoming map data
    std::map< MapId, QImage > m_MapImages;
    //palette used for display of grayscale map images
    QVector<QRgb> m_MapPalette;
    //checker texture was already painted into map images
    std::map< MapId, bool > m_CheckerPainted;
    std::map< int, std::string > m_MapNames;
    
    //copy of incomming maskData
    unsigned short* m_HighSensitive;

    // boolean if a grid will be paited on top of the map
    bool m_PaintGrid;

    /** robot position and orientation */
    QPoint m_RobotPos;
    double m_RobotOrientation;

    /** size in pixels of the poi icons */
    QSize m_DefaultIconSize;

    /** current selected poi */
    QPixmap m_CurrentPoiIconSelection;
    QPoint m_CurrentlySelectedPoi;
    double m_CurrentPoiOrientation;
    std::string m_CurrentlySelectedPoiName;

    /** last position of currently modified poi */
    QPoint m_LastModifiedPoiPositionBackup;
    PoiLabel* m_TmpMouseOperationLabel;
    PoiLabel* m_LastSelectedPoiLabelAddress;

    /** current poi list */
    std::vector< homer_mapnav_msgs::PointOfInterest >* m_PoiList;
    QVector< PoiIcon >* m_PoiIcons;
    QVector< PoiLabel* > m_PoiLabelList;

    /** Visibility and setting states */
    bool m_RobotVisibility;
    bool m_PoiVisibility;
    bool m_RegionVisibility;
    bool m_MovablePoi;

    /** zoom factor of the map */
    float m_ZoomFactor;
    Qt::TransformationMode m_ScaleMode;

    /** modified region */
    QPolygon m_Region;
    int m_SelectedRegionIndex;
    bool m_ModifyRegion;

    // Added by Carmen Navarro Luzón for MasterArbeit 2011, for drawing regions in the map
    std::vector<QPolygon> m_Rooms;
    std::vector<QPoint> m_Centers;
    std::vector<QPoint> m_InteractiveArtifacts;
    std::vector<QPoint> m_NonInteractiveArtifacts;
    unsigned m_SelectedListItemIndex;
    unsigned m_SelectedLayer;
    //int m_SelectedRoomIndex;
    //int m_SelectedArtifactIndex;
    bool m_DisplayRooms;
    bool m_DisplayIntArtifacts;
    bool m_DisplayNonIntArtifacts;

    QPixmap m_RoomCenterIcon;
    QPixmap m_InteractiveIcon;
    QPixmap m_NonInteractiveIcon;

    // End of the code added by Carmen

    std::vector<geometry_msgs::PoseStamped> m_Path;
    std::vector<geometry_msgs::PoseStamped> m_FollowingPath;

    std::vector< float > m_XCoords;
    std::vector< float > m_YCoords;

    // Flag to determine wether the user wants to select two points in the map for a distance meassurement
    bool m_DistDetermination;
    QPoint m_DistDetermStart;
    QPoint m_DistDetermCurrent;

    //ROS related members
    ros::NodeHandle* m_NodeHandle;

    ros::Publisher* m_ModifyMapPublisher;
};

#endif
