///
/// @author Florian Sattler (R22), David Gossow (RX)
///

#ifndef STRING_TOOLS_
#define STRING_TOOLS_

#include <algorithm>
#include <string>
#include <vector>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <sstream>

namespace homer{

	/// @brief Gets a string and returns it in lowercase
	std::string lowercase(const std::string & input){
		std::string data(input);
		std::transform(data.begin(), data.end(), data.begin(), ::tolower);
		return data;
	}


	/// @brief Gets a string and returns it in uppercase
	std::string uppercase(const std::string & input){
		std::string data(input);
		std::transform(data.begin(), data.end(), data.begin(), ::toupper);
		return data;
	}

	/// @brief Returns true if keyword is part of source
	bool contains(const std::string & source, const std::string & keyword){
		return (lowercase(source).find(lowercase(keyword)) != std::string::npos);
	}

	/// @brief Returns true if at least one keyword is part of source
	bool contains (const std::string & source, const std::vector<std::string> & keywords){
		for(unsigned int i = 0; i < keywords.size(); ++i)
			if (contains(source,keywords.at(i)))
				return true;
		return false;
	}

	/// @brief Returns true if keyword is part of source
	bool contains(const std::vector<std::string> & source, const std::string & keyword){
		for(unsigned int i = 0; i < source.size(); ++i)
			if (contains(source.at(i),keyword))
				return true;
		return false;
	}

	/// @brief splits a given string into a vector ("explode")
	std::vector<std::string> splitString(std::string text, const char* separator){
		std::vector<std::string> result;
		boost::split(result, text , boost::is_any_of(separator));
		return result;
	}


	std::vector<std::string> explode(std::string text, const char* separator)
	{
		return splitString(text, separator);
	}

	/// @brief connects the elements of a given vector ("implode")
	std::string connectString(const std::vector<std::string> & strings,const char * separator){
		std::stringstream ss;
		if (strings.size() == 0)
			return "";

		ss << strings.at(0);

		for (unsigned int i = 1; i < strings.size(); ++i)
			ss << separator << strings.at(i);

		return ss.str();
	}

	/// @brief Returns true if all keywords are contained in source in the correct order
	/// @param Strings are meant to be space seperated
	bool isCorrectCommand(std::vector<std::string> & source, std::vector<std::string> & keywords){
		std::vector<int> positions;
		unsigned correctWords = 0;

		if ( source.size() < keywords.size() ) return false;

		//search first occurence
		for ( unsigned int i = 0; i < keywords.size(); i++ )
			for ( unsigned int j = 0; j < source.size(); j++ )
				if ( lowercase(source[j]) == lowercase(keywords[i]) ){
					correctWords++;
					positions.push_back ( j );
					break;
				}

		//check order of appearance
		for ( unsigned int i = 1; i < positions.size(); i++ )
			if ( positions[i-1] > positions[i] )
				return false;

		if ( correctWords != keywords.size() )
			return false;

		return true;
	}

	bool isCorrectCommand(std::string & source, std::string keywords){
		std::vector<std::string> key = splitString(keywords, " " );
		std::vector<std::string> src = splitString(source, " " );
		return isCorrectCommand( src, key );
	}

	bool isCorrectCommand(std::string & source, std::vector<std::string> & keywords){
		std::vector<std::string> src = splitString(source, " " );
		return isCorrectCommand( src, keywords );
	}

	bool isCorrectCommand(std::vector<std::string> & source, std::string keywords){
		std::vector<std::string> key = splitString(keywords, " " );
		return isCorrectCommand( source, key );
	}


    // @brief: remove all characters from text
    std::string removeCharacters( std::string text, std::string characters )
    {
      std::vector<std::string> parts = splitString ( text, characters.c_str() );

      std::
string result="";
      for ( unsigned i=0; i<parts.size(); i++ )
      {
        result += parts[i];
      }

      return result;
    }


    template<class T>
	inline static std::vector<T> convertTo ( std::vector<std::string> strVec )
	{
		std::vector<T> tVec;
		tVec.resize ( strVec.size() );

		for ( unsigned i = 0; i < strVec.size(); i++ )
		{
			std::stringstream s;
			s << strVec[i];
			s >> tVec[i];
		}
		return tVec;
	}

	/// @brief convert a string to some kind of variable
	/// example: int i = String::convertTo<int>( "16" );
	template<class T>
	inline static T convertTo ( std::string str )
	{
		std::stringstream s;
		T t;
		s << str;
		s >> t;
		return t;
	}

	/// convert a variable of arbitrary type T to a string (must be writable to a stream)
	template<class T>
	inline static std::string toString ( T value )
	{
		std::ostringstream s;
		s << value;
		return s.str();
	}
	
	/**
	 * Converts a list of names, so that it can be used in speech output (e.g. {a, b, c} => "a, b and c")
	 * @param items Items to list
	 * @return
	 */
	std::string listItems(const std::vector<std::string>& items) {
		if (items.size() == 0) {
			ROS_WARN_STREAM("Item list is empty") ;
			return "" ;
		}
		else {
			std::stringstream list ;
			list << items[0] ;

			for (unsigned int i=1; i<items.size(); i++) {
				if (i == items.size()-1) {
					list << " and " << items[i] ;
				}
				else {
					list << ", " << items[i] ;
				}
			}
			return list.str() ;
		}
	}

	/// @brief trims (removes leading and closing spaces) from a given string
	std::string trim(std::string s)
	{
		boost::trim(s);
		return s;
	}

}
#endif
