/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef VISIONTRANSFER_IMAGEPAIR_H
#define VISIONTRANSFER_IMAGEPAIR_H

#include <cassert>
#include <cstddef>
#include "visiontransfer/common.h"

/**
 * \brief A set of two images, which are usually the left camera image and the disparity map.
 *
 * Both images must be of equal width and height, but are allowed to have a
 * different pixel formats. Please note that the class does not manage the
 * pixel data but only keeps pointers. You thus need to ensure that the pixel
 * data remains valid for as long as this object persists.
 */
class VT_EXPORT ImagePair {
public:
    /**
     * \brief Image formats that can be transferred.
     */
    enum ImageFormat {
        /// 8-bit greyscale format
        FORMAT_8_BIT,

        /// 12-bit greyscale format plus 4 bits of padding
        /// (hence a total of 16 bits).
        FORMAT_12_BIT
    };

    /**
     * \brief Default constructor creating an image pair with no pixel data.
     */
    ImagePair(): width(0), height(0), qMatrix(NULL), timeSec(0), timeMicrosec(0),
            seqNum(0), minDisparity(0), maxDisparity(0) {
        formats[0] = FORMAT_8_BIT;
        formats[1] = FORMAT_8_BIT;
        data[0] = NULL;
        data[1] = NULL;
        rowStride[0] = 0;
        rowStride[1] = 1;
    }

    /**
     * \brief Sets a new width for both images.
     */
    void setWidth(int w) {width = w;}

    /**
     * \brief Sets a new width for both images.
     */
    void setHeight(int h) {height = h;}

    /**
     * \brief Sets a new row stride for the pixel data of one image.
     *
     * \param imageNumber Number of the image for which to set the
     *        row stride (0 or 1).
     * \param stride The row stride that shall be set.
     */
    void setRowStride(int imageNumber, int stride) {
        assert(imageNumber >= 0 && imageNumber <=1);
        rowStride[imageNumber] = stride;
    }

    /**
     * \brief Sets the pixel format for the given image.
     *
     * \param imageNumber Number of the image for which to set the
     *        pixel format (0 or 1).
     * \param format The pixel format that shall be set.
     */
    void setPixelFormat(int imageNumber, ImageFormat format) {
        assert(imageNumber >= 0 && imageNumber <=1);
        formats[imageNumber] = format;
    }

    /**
     * \brief Sets the pixel data for the given image.
     *
     * \param imageNumber Number of the image for which to set the
     *        pixel data (0 or 1).
     * \param pixelData The pixel data that shall be set.
     */
    void setPixelData(int imageNumber, unsigned char* pixelData) {
        assert(imageNumber >= 0 && imageNumber <=1);
        data[imageNumber] = pixelData;
    }

    /**
     * \brief Sets the pointer to the disparity-to-depth mapping matrix q.
     *
     * No data is copied. The data which q is pointing to has to remain valid
     * for as long as this object exists.
     */
    void setQMatrix(const float* q) {
        qMatrix = q;
    }

    /**
     * \brief Sets the sequence number for this image pair.
     */
    void setSequenceNumber(unsigned int num) {
        seqNum = num;
    }

    /**
     * \brief Sets the time at which this image pair has been captured.
     *
     * \param seconds The time stamp with a resolution of one second.
     * \param microsec The fractional seconds part of the time stamp with
     *        a resolution of 1 microsecond.
     */
    void setTimestamp(int seconds, int microsec) {
        timeSec = seconds;
        timeMicrosec = microsec;
    }

    /**
     * \brief Sets the value range for the disparity map contained in this
     *        image pair.
     *
     * \param minimum Minimum disparity value.
     * \param maximum Maximum disparity value.
     */
    void setDisparityRange(int minimum, int maximum) {
        minDisparity = minimum;
        maxDisparity = maximum;
    }

    /**
     * \brief Returns the width of each image.
     */
    int getWidth() const {return width;}

    /**
     * \brief Returns the height of each image.
     */
    int getHeight() const {return height;}

    /**
     * \brief Returns the row stride for the pixel data of one image.
     *
     * \param imageNumber Number of the image for which to receive the
     *        row stride (0 or 1).
     */
    int getRowStride(int imageNumber) const {
        assert(imageNumber >= 0 && imageNumber <=1);
        return rowStride[imageNumber];
    }

    /**
     * \brief Returns the pixel format for the given image.
     *
     * \param imageNumber Number of the image for which to receive the
     *        pixel format (0 or 1).
     */
    ImageFormat getPixelFormat(int imageNumber) const {
        assert(imageNumber >= 0 && imageNumber <=1);
        return formats[imageNumber];
    }

    /**
     * \brief Returns the pixel data for the given image.
     *
     * \param imageNumber Number of the image for which to receive the
     *        pixel data (0 or 1).
     */
    unsigned char* getPixelData(int imageNumber) const {
        assert(imageNumber >= 0 && imageNumber <=1);
        return data[imageNumber];
    }

    /**
     * \brief Returns a pointer to the disparity-to-depth mapping matrix q.
     */
    const float* getQMatrix() const {
        return qMatrix;
    }

    /**
     * \brief Returns the sequence number for this image pair.
     */
    unsigned int getSequenceNumber() const {return seqNum;}

    /**
     * \brief Returns the time at which this image pair has been captured.
     *
     * \param seconds The time stamp with a resolution of one second.
     * \param microsec The fractional seconds part of the time stamp with
     *        a resolution of 1 microsecond.
     */
    void getTimestamp(int& seconds, int& microsec) const {
        seconds = timeSec;
        microsec = timeMicrosec;
    }

    /**
     * \brief Gets the value range for the disparity map contained in this
     *        image pair. If the image pair does not contain any disparity data
     *        then the disparity range is undefined.
     *
     * \param minimum Minimum disparity value.
     * \param maximum Maximum disparity value.
     */
    void getDisparityRange(int& minimum, int& maximum) const {
        minimum = minDisparity;
        maximum = maxDisparity;
    }

    /**
     * \brief Writes one image of the pair to a PGM file
     *
     * \param imageNumber The number of the image that shall be written.
     * \param File name of the PGM file that shall be created.
     */
    void writePgmFile(int imageNumber, const char* fileName);

private:
    // No pimpl idiom here as almost everything is inlined.

    int width;
    int height;
    int rowStride[2];
    ImageFormat formats[2];
    unsigned char* data[2];
    const float* qMatrix;
    int timeSec;
    int timeMicrosec;
    unsigned int seqNum;
    int minDisparity;
    int maxDisparity;
};

#endif

