#!/usr/bin/env python
# -*- coding: euc-jp -*-

##
# @file RingBuffer.py
# @brief Defautl Buffer class
# @date $Date: 2007/09/12 $
# @author Noriaki Ando <n-ando@aist.go.jp> and Shinji Kurihara
#
# Copyright (C) 2006-2008
#     Noriaki Ando
#     Task-intelligence Research Group,
#     Intelligent Systems Research Institute,
#     National Institute of
#         Advanced Industrial Science and Technology (AIST), Japan
#     All rights reserved.

import threading
import OpenRTM_aist


##
# @if jp
# @class RingBuffer
# @brief 󥰥Хåե饹
# 
# ꤷĹΥ󥰾ХåեĥХåե饹
# ХåեΤ˥ǡǼ줿硢ʹߤΥǡϸŤǡ
# 缡񤭤롣
# äơХåեˤľΥХåեĹʬΥǡΤݻ롣
#
# )ߤμǤϡֺǸ˳ǼǡΤߥХåեɤ߽Фǽ
#
# @param DataType Хåե˳Ǽǡ
#
# @since 0.4.0
#
# @else
#
# @endif
class RingBuffer(OpenRTM_aist.BufferBase):
  """
  """

  RINGBUFFER_DEFAULT_LENGTH = 8

  ##
  # @if jp
  #
  # @brief 󥹥ȥ饯
  # 
  # 󥹥ȥ饯
  # ꤵ줿ХåեĹǥХåե롣
  #
  # @param length ХåեĹ
  # 
  # @else
  #
  # @brief Constructor
  # 
  # Constructor.
  # Initialize the buffer by specified buffer length.
  # However, if the specified length is less than two, the buffer should
  # be initialized by two in length.
  #
  # @param length Buffer length
  # 
  # @endif
  #
  # @endif
  def __init__(self, length=RINGBUFFER_DEFAULT_LENGTH):
    self._overwrite = True
    self._readback = True
    self._timedwrite = False
    self._timedread  = False
    self._wtimeout = OpenRTM_aist.TimeValue(1,0)
    self._rtimeout = OpenRTM_aist.TimeValue(1,0)
    self._length   = length
    self._wpos = 0
    self._rpos = 0
    self._fillcount = 0
    self._wcount = 0
    self._buffer = [None for i in range(self._length)]
    self._pos_mutex = threading.RLock()
    self._full_mutex = threading.RLock()
    self._empty_mutex = threading.RLock()
    self._full_cond = threading.Condition(self._full_mutex)
    self._empty_cond = threading.Condition(self._empty_mutex)


    self.reset()


  ##
  # @if jp
  # @brief Хåե
  #
  # Properties Ϳץѥƥˤꡢ
  # Хåե롣
  # ѤǤ륪ץỌ̇̄ϰʲ̤
  #
  # - buffer.length:
  #     ХåեĹʳοͤꤵƤ̵뤵롣
  #     Ǥ˥ХåեѾ֤Ǥ⡢Ĺꤵ줿Τ٤Ƥ
  #     ݥ󥿤롣
  #
  # - buffer.write.full_policy:
  #     񤭤뤫ɤΥݥꥷ
  #     overwrite (), do_nothing (⤷ʤ), block (֥å)
  #     block ꤷ硢 timeout ͤꤹСָ
  #     񤭹ԲǽǤХॢȤ롣
  #     ǥեȤ  overwrite ()
  #
  # - buffer.write.timeout:
  #     ॢȻ֤ [sec] ǻꤹ롣ǥեȤ 1.0 [sec]
  #     1 sec -> 1.0, 1 ms -> 0.001, ॢȤʤ -> 0.0
  #
  # - buffer.read.empty_policy:
  #     ХåեΤȤɤ߽Фݥꥷ
  #     readback (Ǹ), do_nothing (⤷ʤ), block (֥å)
  #     block ꤷ硢 timeout ͤꤹСָ
  #     ɤ߽ФԲǽǤХॢȤ롣
  #     ǥեȤ readback (Ǹ)
  #
  # - buffer.read.timeout:
  #     ॢȻ [sec] ǻꤹ롣ǥեȤ 1.0 [sec]
  #     1sec -> 1.0, 1ms -> 0.001, ॢȤʤ -> 0.0
  #
  # @else
  #
  # @endif
  #
  # void init(const coil::Properties& prop)
  def init(self, prop):
    self.__initLength(prop)
    self.__initWritePolicy(prop)
    self.__initReadPolicy(prop)


  ##
  # @if jp
  #
  # @brief ХåեĹ
  # 
  # ХåեĹ롣
  #
  # @param self
  # 
  # @return ХåեĹ
  # 
  # @else
  #
  # @brief Get the buffer length
  #
  # @endif
  #
  # size_t length(void) const
  def length(self, n = None):
    if n is None:
      guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
      return self._length

    if n < 1:
      return OpenRTM_aist.BufferStatus.NOT_SUPPORTED

    self._buffer = [None for i in range(n)]
    self._length = n
    self.reset()
    return OpenRTM_aist.BufferStatus.BUFFER_OK

  
  ##
  # @if jp
  #
  # @brief Хåեξ֤ꥻåȤ
  # 
  # Хåեɤ߽Фݥ󥿤Ƚ񤭹ߥݥ󥿤ΰ֤ꥻåȤ롣
  # μǤ BUFFER_OK ֤ʤ
  # 
  # @return BUFFER_OK: ｪλ
  #         NOT_SUPPORTED: ꥻåԲǽ
  #         BUFFER_ERROR: ۾ｪλ
  # 
  # @else
  #
  # @brief Get the buffer length
  #
  # Pure virtual function to get the buffer length.
  #
  # @return buffer length
  # 
  # @endif
  # 
  # ReturnCode reset()
  def reset(self):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    self._fillcount = 0
    self._wcount = 0
    self._wpos = 0
    self._rpos = 0
    return OpenRTM_aist.BufferStatus.BUFFER_OK


  ##
  # @if jp
  #
  # @brief ХåեθߤνǤΥݥ
  # 
  # ХåեθߤνǤΥݥ󥿤ޤϡnΥݥ󥿤֤
  # 
  # @param  n ߥݥ + n ΰ֤Υݥ 
  # @return ֤߰Υݥ
  # 
  # @else
  #
  # @brief Get the buffer length
  #
  # Pure virtual function to get the buffer length.
  #
  # @return buffer length
  # 
  # @endif
  # 
  # DataType* wptr(long int n = 0) 
  def wptr(self, n = 0):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    return self._buffer[(self._wpos + n + self._length) % self._length]

    
  ##
  # @if jp
  #
  # @brief ߥݥ󥿤ʤ
  # 
  # ߤν񤭹֤߰Υݥ󥿤 n Ŀʤ롣
  # 񤭹߲ǽǿʾοͤꤷ硢PRECONDITION_NOT_MET
  # ֤
  # 
  # @param  n ߥݥ + n ΰ֤Υݥ 
  # @return BUFFER_OK:            ｪλ
  #         PRECONDITION_NOT_MET: n > writable()
  # 
  # @else
  #
  # @brief Get the buffer length
  #
  # Pure virtual function to get the buffer length.
  #
  # @return buffer length
  # 
  # @endif
  # 
  # ReturnCode advanceWptr(long int n = 1)
  def advanceWptr(self, n = 1):
    # n > 0 :
    #     n satisfies n <= writable elements
    #                 n <= m_length - m_fillcout
    # n < 0 : -n = n'
    #     n satisfies n'<= readable elements
    #                 n'<= m_fillcount
    #                 n >= - m_fillcount
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    if (n > 0 and n > (self._length - self._fillcount)) or \
          (n < 0 and n < (-self._fillcount)):
      return OpenRTM_aist.BufferStatus.PRECONDITION_NOT_MET

    self._wpos = (self._wpos + n + self._length) % self._length
    self._fillcount += n
    return OpenRTM_aist.BufferStatus.BUFFER_OK


  ##
  # @if jp
  #
  # @brief Хåե˥ǡ񤭹
  # 
  # Хåե˥ǡ񤭹ࡣ񤭹ߥݥ󥿤ΰ֤ѹʤ
  # μǤϾ BUFFER_OK ֤
  # 
  # @param value 񤭹оݥǡ
  #
  # @return BUFFER_OK: ｪλ
  #         BUFFER_ERROR: ۾ｪλ
  # 
  # @else
  #
  # @brief Write data into the buffer
  #
  # Pure virtual function to write data into the buffer.

  #
  # @param value Target data to write.
  #
  # @return BUFFER_OK:    Successful
  #         BUFFER_ERROR: Failed
  #
  # @endif
  #
  # ReturnCode put(const DataType& value)
  def put(self, value):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    self._buffer[self._wpos] = value
    return OpenRTM_aist.BufferStatus.BUFFER_OK
    
  ##
  # @if jp
  #
  # @brief Хåե˽񤭹
  # 
  # Ϳ줿ǡХåե˽񤭹ࡣ
  #
  # 2(sec)3(nsec)ꤵƤʤ硢Хåեե
  # νߥ⡼ (overwrite, do_nothing, block)  init() 
  # 줿⡼ɤ˽
  #
  # 2(sec) ˰ꤵ줿ϡinit() ꤵ줿⡼
  # ˴ؤ餺block ⡼ɤȤʤꡢХåեե֤Ǥл
  # ֤ޤॢȤ롣3(nsec)ϻꤵʤ0Ȥ
  # 롣ॢԤˡɤ߽Фå¦ǥХåե
  # ɤ߽ФС֥å󥰤ϲǡ񤭹ޤ롣
  #
  # 񤭹߻˥Хåե(empty)֤ǡ̤Υåɤblock⡼
  # ɤ߽ФԤ򤷤Ƥ硢signalȯԤɤ߽Ф¦Υ֥
  # 󥰤롣
  # 
  # @param value 񤭹оݥǡ
  # @param sec   ॢȻ sec  (default -1: ̵)
  # @param nsec  ॢȻ nsec (default 0)
  # @return BUFFER_OK            ｪλ
  #         BUFFER_FULL          Хåեե
  #         TIMEOUT              ߤॢȤ
  #         PRECONDITION_NOT_MET ۾
  # 
  # @else
  #
  # @brief Write data into the buffer
  # 
  # Write data which is given argument into the buffer.
  #
  # @param value Target data for writing
  #
  # @return Writing result (Always true: writing success is returned)
  # 
  # @endif
  # 
  # ReturnCode write(const DataType& value,
  #                  long int sec = -1, long int nsec = 0)
  def write(self, value, sec = -1, nsec = 0):
    try:
      self._full_cond.acquire()
      if self.full():
        timedwrite = self._timedwrite # default is False
        overwrite  = self._overwrite  # default is True

        if not (sec < 0): # if second arg is set -> block mode
          timedwrite = True
          overwrite  = False

        if overwrite and not timedwrite:       # "overwrite" mode
          self.advanceRptr()

        elif not overwrite and not timedwrite: # "do_nothiong" mode
          self._full_cond.release()
          return OpenRTM_aist.BufferStatus.BUFFER_FULL

        elif not overwrite and timedwrite:     # "block" mode

          if sec < 0:
            sec = self._wtimeout.sec()
            nsec = self._wtimeout.usec() * 1000

          # true: signaled, false: timeout
          if not self._full_cond.wait(sec + (nsec/1000000000.0)):
            self._full_cond.release()
            return OpenRTM_aist.BufferStatus.TIMEOUT

        else: # unknown condition
          self._full_cond.release()
          return OpenRTM_aist.BufferStatus.PRECONDITION_NOT_MET
      
      self._full_cond.release()

      self.put(value)
      
      self._empty_cond.acquire()
      empty = self.empty()
      if empty:
        self.advanceWptr(1)
        self._empty_cond.notify()
      else:
        self.advanceWptr(1)
      self._empty_cond.release()

      return OpenRTM_aist.BufferStatus.BUFFER_OK
    except:
      return OpenRTM_aist.BufferStatus.BUFFER_OK

    
  ##
  # @if jp
  #
  # @brief Хåե˽߲ǽǿ
  # 
  # Хåե˽߲ǽǿ֤
  # 
  # @return 񤭹߲ǽǿ
  # 
  # @else
  #
  # @brief Write data into the buffer
  #
  # Pure virtual function to write data into the buffer.
  #
  # @param value Target data to write.
  #
  # @return Result of having written in data (true:Successful, false:Failed)
  #
  # @endif
  #
  # size_t writable() const
  def writable(self):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    return self._length - self._fillcount
    

  ##
  # @if jp
  #
  # @brief Хåեfullå
  # 
  # Хåեfullåѽ貾۴ؿ
  #
  # @return fullå(true:Хåեfullfalse:Хåե)
  # 
  # @else
  #
  # @brief Check on whether the buffer is full.
  #
  # Pure virtual function to check on whether the buffer is full.
  #
  # @return True if the buffer is full, else false.
  #
  # @endif
  #
  # bool full(void) const
  def full(self):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    return self._length == self._fillcount
    

  ##
  # @if jp
  #
  # @brief Хåեθߤɤ߽ФǤΥݥ
  # 
  # Хåեθߤɤ߽ФǤΥݥ󥿤ޤϡnΥݥ󥿤֤
  # 
  # @param  n ɤ߽Фݥ + n ΰ֤Υݥ 
  # @return ɤ߽Ф֤Υݥ
  # 
  # @else
  #
  # @brief Get the buffer length
  #
  # Pure virtual function to get the buffer length.
  #
  # @return buffer length
  # 
  # @endif
  def rptr(self, n = 0):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    return self._buffer[(self._rpos + n + self._length) % self._length]

    
  ##
  # @if jp
  #
  # @brief ɤ߽Фݥ󥿤ʤ
  # 
  # ߤɤ߽Ф֤Υݥ󥿤 n Ŀʤ롣
  # 
  # @param  n ɤ߽Фݥ + n ΰ֤Υݥ 
  # @return BUFFER_OK: ｪλ
  #         BUFFER_ERROR: ۾ｪλ
  # 
  # @else
  #
  # @brief Get the buffer length
  #
  # Pure virtual function to get the buffer length.
  #
  # @return buffer length
  # 
  # @endif
  # 
  # DataType* rptr(long int n = 0)
  def advanceRptr(self, n = 1):
    # n > 0 :
    #     n satisfies n <= readable elements
    #                 n <= m_fillcout 
    # n < 0 : -n = n'
    #     n satisfies n'<= m_length - m_fillcount
    #                 n >= m_fillcount - m_length
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    if (n > 0 and n > self._fillcount) or \
          (n < 0 and n < (self._fillcount - self._length)):
      return OpenRTM_aist.BufferStatus.PRECONDITION_NOT_MET

    self._rpos = (self._rpos + n + self._length) % self._length
    self._fillcount -= n
    return OpenRTM_aist.BufferStatus.BUFFER_OK


    
  ##
  # @if jp
  #
  # @brief Хåեǡɤ߽Ф
  # 
  # Хåեǡɤߤɤ߽Фݥ󥿤ΰ֤ѹʤ
  # 
  # @param value ɤ߽Фǡ
  #
  # @return BUFFER_OK: ｪλ
  #         BUFFER_ERROR: ۾ｪλ
  # 
  # @else
  #
  # @brief Write data into the buffer
  #
  # Pure virtual function to write data into the buffer.
  #
  # @param value Target data to write.
  #
  # @return Result of having written in data (true:Successful, false:Failed)
  #
  # @endif
  #
  # ReturnCode get(DataType& value)
  def get(self, value=None):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    if value is None:
      return self._buffer[self._rpos]

    value[0] = self._buffer[self._rpos]
    return OpenRTM_aist.BufferStatus.BUFFER_OK
    
    
  ##
  # @if jp
  #
  # @brief Хåեɤ߽Ф
  # 
  # Хåե˳Ǽ줿ǡɤ߽Ф
  #
  # 2(sec)3(nsec)ꤵƤʤ硢Хåե
  # ֤Ǥɤ߽Ф⡼ (readback, do_nothing, block)  init() 
  # ꤵ줿⡼ɤ˽
  #
  # 2(sec) ˰ꤵ줿ϡinit() ꤵ줿⡼
  # ˴ؤ餺block ⡼ɤȤʤꡢХåե֤Ǥл
  # ԤॢȤ롣3(nsec)ϻꤵʤ0Ȥư
  # 롣ॢԤˡߥå¦ǥХåեؽ
  # С֥å󥰤ϲǡɤߤ롣
  #
  # ɤ߽Ф˥Хåե(empty)֤ǡ̤Υåɤblock⡼
  # ǽԤ򤷤Ƥ硢signalȯԤƽ¦Υ֥å
  # 롣
  # 
  # @param value(list) ɤ߽Фоݥǡ
  # @param sec   ॢȻ sec  (default -1: ̵)
  # @param nsec  ॢȻ nsec (default 0)
  # @return BUFFER_OK            ｪλ
  #         BUFFER_EMPTY         Хåեե
  #         TIMEOUT              ߤॢȤ
  #         PRECONDITION_NOT_MET ۾
  # 
  # @else
  #
  # @brief Readout data from the buffer
  # 
  # Readout data stored into the buffer.
  # 
  # @param value(list) Readout data
  #
  # @return Readout result (Always true: readout success is returned)
  # 
  # @endif
  #
  # ReturnCode read(DataType& value,
  #                 long int sec = -1, long int nsec = 0)
  def read(self, value, sec = -1, nsec = 0):
    self._empty_cond.acquire()
      
    if self.empty():
      timedread = self._timedread
      readback  = self._readback

      if not (sec < 0):  # if second arg is set -> block mode
        timedread = True
        readback  = False
        sec = self._rtimeout.sec()
        nsec = self._rtimeout.usec() * 1000

      if readback and  not timedread:      # "readback" mode
        if not self._wcount > 0:
          self._empty_cond.release()
          return OpenRTM_aist.BufferStatus.BUFFER_EMPTY
        self.advanceRptr(-1)

      elif not readback and not timedread: # "do_nothiong" mode
        self._empty_cond.release()
        return OpenRTM_aist.BufferStatus.BUFFER_EMPTY

      elif not readback and timedread:     # "block" mode
        if sec < 0:
          sec = self._rtimeout.sec()
          nsec = self._rtimeout.usec() * 1000
        #  true: signaled, false: timeout
        if not self._empty_cond.wait(sec + (nsec/1000000000.0)):
          self._empty_cond.release()
          return OpenRTM_aist.BufferStatus.TIMEOUT

      else:                              # unknown condition
        self._empty_cond.release()
        return OpenRTM_aist.BufferStatus.PRECONDITION_NOT_MET
      
    self._empty_cond.release()

    val = self.get()

    if len(value) > 0:
      value[0] = val
    else:
      value.append(val)

    self._full_cond.acquire()
    full_ = self.full()

    if full_:
      self.advanceRptr()
      self._full_cond.notify()
    else:
      self.advanceRptr()

    self._full_cond.release()


    return OpenRTM_aist.BufferStatus.BUFFER_OK

    
  ##
  # @if jp
  #
  # @brief Хåեɤ߽Фǽǿ
  # 
  # Хåեɤ߽Фǽǿ֤
  # 
  # @return ɤ߽Фǽǿ
  #
  # @return BUFFER_OK: ｪλ
  #         BUFFER_ERROR: ۾ｪλ
  # 
  # @else
  #
  # @brief Write data into the buffer
  #
  # Pure virtual function to write data into the buffer.
  #
  # @param value Target data to write.
  #
  # @return Result of having written in data (true:Successful, false:Failed)
  #
  # @endif
  #
  # size_t readable() const
  def readable(self):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    return self._fillcount
    

  ##
  # @if jp
  #
  # @brief Хåեemptyå
  # 
  # Хåեemptyåѽ貾۴ؿ
  #
  # @return emptyå(true:Хåեemptyfalse:Хåեǡ)
  # 
  # @else
  #
  # @brief Check on whether the buffer is empty.
  #
  # Pure virtual function to check on whether the buffer is empty.
  #
  # @return True if the buffer is empty, else false.
  #
  # @endif
  #
  # bool empty(void) const
  def empty(self):
    guard = OpenRTM_aist.ScopedLock(self._pos_mutex)
    return self._fillcount == 0

    
  ## void initLength(const coil::Properties& prop)
  def __initLength(self, prop):
    if prop.getProperty("length"):
      n = [0]
      if OpenRTM_aist.stringTo(n, prop.getProperty("length")):
        n = n[0]
        if n > 0:
          self.length(n)

    
  ## void initWritePolicy(const coil::Properties& prop)
  def __initWritePolicy(self, prop):
    policy = OpenRTM_aist.normalize([prop.getProperty("write.full_policy")])

    if policy == "overwrite":
      self._overwrite  = True
      self._timedwrite = False
    
    elif policy == "do_nothing":
      self._overwrite  = False
      self._timedwrite = False

    elif policy == "block":
      self._overwrite  = False
      self._timedwrite = True

      tm = [0.0]
      if OpenRTM_aist.stringTo(tm, prop.getProperty("write.timeout")):
        tm = tm[0]
        if not (tm < 0):
          self._wtimeout.set_time(tm)

    
  ## void initReadPolicy(const coil::Properties& prop)
  def __initReadPolicy(self, prop):
    policy = prop.getProperty("read.empty_policy")

    if policy == "readback":
      self._readback  = True
      self._timedread = False

    elif policy == "do_nothing":
      self._readback  = False
      self._timedread = False

    elif policy == "block":
      self._readback  = False
      self._timedread = True
      tm = [0.0]
      if OpenRTM_aist.stringTo(tm, prop.getProperty("read.timeout")):
        self._rtimeout.set_time(tm[0])
