// -*- C++ -*-
/*!
 * @file ComponentActionListener.h
 * @brief component action listener class
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2011
 *     Noriaki Ando
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef RTC_COMPONENTACTIONLISTENER_H
#define RTC_COMPONENTACTIONLISTENER_H

#include <vector>
#include <utility>
#include <coil/Mutex.h>
#include <coil/Guard.h>
#include <rtm/RTC.h>
#include <rtm/idl/RTCSkel.h>
#include <rtm/ConnectorBase.h>

namespace RTC
{
  typedef ExecutionContextHandle_t UniqueId;
  //============================================================
  /*!
   * @if jp
   * @brief PreComponentActionListener Υ
   *
   * - PRE_ON_INITIALIZE:    onInitialize ľ
   * - PRE_ON_FINALIZE:      onFinalize ľ
   * - PRE_ON_STARTUP:       onStartup ľ
   * - PRE_ON_SHUTDOWN:      onShutdown ľ
   * - PRE_ON_ACTIVATED:     onActivated ľ
   * - PRE_ON_DEACTIVATED:   onDeactivated ľ
   * - PRE_ON_ABORTING:      onAborted ľ
   * - PRE_ON_ERROR:         onError ľ
   * - PRE_ON_RESET:         onReset ľ
   * - PRE_ON_EXECUTE:       onExecute ľ
   * - PRE_ON_STATE_UPDATE:  onStateUpdate ľ
   * - PRE_ON_RATE_CHANGED:  onRateChanged ľ
   *
   * @else
   * @brief The types of ConnectorDataListener
   * 
   * @endif
   */

  enum PreComponentActionListenerType
    {
      PRE_ON_INITIALIZE,
      PRE_ON_FINALIZE,
      PRE_ON_STARTUP,
      PRE_ON_SHUTDOWN,
      PRE_ON_ACTIVATED,
      PRE_ON_DEACTIVATED,
      PRE_ON_ABORTING,
      PRE_ON_ERROR,
      PRE_ON_RESET,
      PRE_ON_EXECUTE,
      PRE_ON_STATE_UPDATE,
      PRE_ON_RATE_CHANGED,
      PRE_COMPONENT_ACTION_LISTENER_NUM
    };

  /*!
   * @if jp
   * @class PreComponentActionListener 饹
   * @brief PreComponentActionListener 饹
   *
   * OMG RTCͤƤʲΥݡͥȥȤˤĤ
   * ơ
   *
   * - on_initialize()
   * - on_finalize()
   * - on_startup()
   * - on_shutdown()
   * - on_activated
   * - on_deactivated()
   * - on_aborted()
   * - on_error()
   * - on_reset()
   * - on_execute()
   * - on_state_update()
   * - on_rate_changed()
   *
   * ƥб桼ɤƤФľΥߥ
   * ǥ뤵ꥹʥ饹δ쥯饹
   *
   * - PRE_ON_INITIALIZE:
   * - PRE_ON_FINALIZE:
   * - PRE_ON_STARTUP:
   * - PRE_ON_SHUTDOWN:
   * - PRE_ON_ACTIVATED:
   * - PRE_ON_DEACTIVATED:
   * - PRE_ON_ABORTING:
   * - PRE_ON_ERROR:
   * - PRE_ON_RESET:
   * - PRE_IN_EXECUTE:
   * - PRE_ON_STATE_UPDATE:
   * - PRE_ON_RATE_CHANGED:
   *
   * @else
   * @class PreComponentActionListener class
   * @brief PreComponentActionListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events in rtobject.
   *
   * @endif
   */
  class PreComponentActionListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief PreComponentActionListenerType ʸѴ
     *
     * PreComponentActionListenerType ʸѴ
     *
     * @param type Ѵо PreComponentActionListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert PreComponentActionListenerType into the string.
     *
     * Convert PreComponentActionListenerType into the string.
     *
     * @param type The target PreComponentActionListenerType for transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(PreComponentActionListenerType type)
    {
      static const char* typeString[] =
        {
          "PRE_ON_INITIALIZE",
          "PRE_ON_FINALIZE",
          "PRE_ON_STARTUP",
          "PRE_ON_SHUTDOWN",
          "PRE_ON_ACTIVATED",
          "PRE_ON_DEACTIVATED",
          "PRE_ON_ABORTING",
          "PRE_ON_ERROR",
          "PRE_ON_RESET",
          "PRE_ON_EXECUTE",
          "PRE_ON_STATE_UPDATE",
          "PRE_ON_RATE_CHANGED",
          "PRE_COMPONENT_ACTION_LISTENER_NUM"
        };
      if (type < PRE_COMPONENT_ACTION_LISTENER_NUM) { return typeString[type]; }
      return "";
    }

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PreComponentActionListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * PreComponentActionListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for PreComponentActionListener.
     *
     * @endif
     */
    virtual void operator()(UniqueId ec_id) = 0;
  };


  //============================================================
  /*!
   * @if jp
   * @brief PostCompoenntActionListener Υ
   *
   * - POST_ON_INITIALIZE:
   * - POST_ON_FINALIZE:
   * - POST_ON_STARTUP:
   * - POST_ON_SHUTDOWN:
   * - POST_ON_ACTIVATED:
   * - POST_ON_DEACTIVATED:
   * - POST_ON_ABORTING:
   * - POST_ON_ERROR:
   * - POST_ON_RESET:
   * - POST_ON_EXECUTE:
   * - POST_ON_STATE_UPDATE:
   * - POST_ON_RATE_CHANGED:
   *
   * @else
   * @brief The types of ConnectorDataListener
   * 
   * @endif
   */
  enum PostComponentActionListenerType
    {
      POST_ON_INITIALIZE,
      POST_ON_FINALIZE,
      POST_ON_STARTUP,
      POST_ON_SHUTDOWN,
      POST_ON_ACTIVATED,
      POST_ON_DEACTIVATED,
      POST_ON_ABORTING,
      POST_ON_ERROR,
      POST_ON_RESET,
      POST_ON_EXECUTE,
      POST_ON_STATE_UPDATE,
      POST_ON_RATE_CHANGED,
      POST_COMPONENT_ACTION_LISTENER_NUM
    };


  /*!
   * @if jp
   * @class PostComponentActionListener 饹
   * @brief PostComponentActionListener 饹
   *
   * OMG RTCͤƤʲΥݡͥȥȤˤĤ
   * ơ
   *
   * - on_initialize()
   * - on_finalize()
   * - on_startup()
   * - on_shutdown()
   * - on_activated
   * - on_deactivated()
   * - on_aborted()
   * - on_error()
   * - on_reset()
   * - on_execute()
   * - on_state_update()
   * - on_rate_changed()
   *
   * ƥб桼ɤƤФľΥߥ
   * ǥ뤵ꥹʥ饹δ쥯饹
   *
   * - POST_ON_INITIALIZE:
   * - POST_ON_FINALIZE:
   * - POST_ON_STARTUP:
   * - POST_ON_SHUTDOWN:
   * - POST_ON_ACTIVATED:
   * - POST_ON_DEACTIVATED:
   * - POST_ON_ABORTING:
   * - POST_ON_ERROR:
   * - POST_ON_RESET:
   * - POST_ON_EXECUTE:
   * - POST_ON_STATE_UPDATE:
   * - POST_ON_RATE_CHANGED:
   *
   * @else
   * @class PostComponentActionListener class
   * @brief PostComponentActionListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events in rtobject.
   *
   * @endif
   */
  class PostComponentActionListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief PostComponentActionListenerType ʸѴ
     *
     * PostComponentActionListenerType ʸѴ
     *
     * @param type Ѵо PostComponentActionListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert PostComponentActionListenerType into the string.
     *
     * Convert PostComponentActionListenerType into the string.
     *
     * @param type The target PostComponentActionListenerType for transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(PostComponentActionListenerType type)
    {
      static const char* typeString[] =
        {
          "POST_ON_INITIALIZE",
          "POST_ON_FINALIZE",
          "POST_ON_STARTUP",
          "POST_ON_SHUTDOWN",
          "POST_ON_ACTIVATED",
          "POST_ON_DEACTIVATED",
          "POST_ON_ABORTING",
          "POST_ON_ERROR",
          "POST_ON_RESET",
          "POST_ON_EXECUTE",
          "POST_ON_STATE_UPDATE",
          "POST_ON_RATE_CHANGED",
          "POST_COMPONENT_ACTION_LISTENER_NUM"
        };
      if (type < POST_COMPONENT_ACTION_LISTENER_NUM)
        {
          return typeString[type];
        }
      return "";
    }

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PostComponentActionListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * PostComponentActionListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for PostComponentActionListener.
     *
     * @endif
     */
    virtual void operator()(UniqueId ec_id,
                            ReturnCode_t ret) = 0;
  };

  //============================================================
  /*!
   * @if jp
   * @brief PortActionListener Υ
   *
   * - ADD_PORT:             Port ɲû
   * - REMOVE_PORT:          Port 
   *
   * @else
   * @brief The types of PortActionListener
   * 
   * @endif
   */

  enum PortActionListenerType
    {
      ADD_PORT,
      REMOVE_PORT,
      PORT_ACTION_LISTENER_NUM
    };

  /*!
   * @if jp
   * @class PortActionListener 饹
   * @brief PortActionListener 饹
   *
   * ƥб桼ɤƤФľΥߥ
   * ǥ뤵ꥹʥ饹δ쥯饹
   *
   * - ADD_PORT:
   * - REMOVE_PORT:
   *
   * @else
   * @class PortActionListener class
   * @brief PortActionListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events in rtobject.
   *
   * @endif
   */
  class PortActionListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief PreComponentActionListenerType ʸѴ
     *
     * PreComponentActionListenerType ʸѴ
     *
     * @param type Ѵо PreComponentActionListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert PreComponentActionListenerType into the string.
     *
     * Convert PreComponentActionListenerType into the string.
     *
     * @param type The target PreComponentActionListenerType for transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(PortActionListenerType type)
    {
      static const char* typeString[] =
        {
          "ADD_PORT",
          "REMOVE_PORT",
          "PORT_ACTION_LISTENER_NUM"
        };
      if (type < PORT_ACTION_LISTENER_NUM) { return typeString[type]; }
      return "";
    }

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PortActionListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * PortActionListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for PortActionListener
     *
     * @endif
     */
    virtual void operator()(const ::RTC::PortProfile& pprof) = 0;
  };


  //============================================================
  /*!
   * @if jp
   * @brief ExecutionContextActionListener Υ
   *
   * - ADD_PORT:             ExecutionContext ɲû
   * - REMOVE_PORT:          ExecutionContext 
   *
   * @else
   * @brief The types of ExecutionContextActionListener
   * 
   * @endif
   */

  enum ExecutionContextActionListenerType
    {
      EC_ATTACHED,
      EC_DETACHED,
      EC_ACTION_LISTENER_NUM
    };

  /*!
   * @if jp
   * @class ExecutionContextActionListener 饹
   * @brief ExecutionContextActionListener 饹
   *
   * ƥб桼ɤƤФľΥߥ
   * ǥ뤵ꥹʥ饹δ쥯饹
   *
   * - ADD_PORT:
   * - REMOVE_PORT:
   *
   * @else
   * @class ExecutionContextActionListener class
   * @brief ExecutionContextActionListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events in rtobject.
   *
   * @endif
   */
  class ExecutionContextActionListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief PreComponentActionListenerType ʸѴ
     *
     * PreComponentActionListenerType ʸѴ
     *
     * @param type Ѵо PreComponentActionListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert PreComponentActionListenerType into the string.
     *
     * Convert PreComponentActionListenerType into the string.
     *
     * @param type The target PreComponentActionListenerType for transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(ExecutionContextActionListenerType type)
    {
      static const char* typeString[] =
        {
          "ATTACH_EC",
          "DETACH_ECT",
          "EC_ACTION_LISTENER_NUM"
        };
      if (type < EC_ACTION_LISTENER_NUM) { return typeString[type]; }
      return "";
    }

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ExecutionContextActionListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * ExecutionContextActionListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for ExecutionContextActionListener
     *
     * @endif
     */
    virtual void operator()(UniqueId ec_id) = 0;
  };



  //============================================================
  /*!
   * @if jp
   * @class PreComponentActionListenerHolder 
   * @brief PreComponentActionListener ۥ饹
   *
   * ʣ PreComponentActionListener ݻ륯饹
   *
   * @else
   * @class PreComponentActionListenerHolder
   * @brief PreComponentActionListener holder class
   *
   * This class manages one ore more instances of
   * PreComponentActionListener class.
   *
   * @endif
   */
  class PreComponentActionListenerHolder
  {
    typedef std::pair<PreComponentActionListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    PreComponentActionListenerHolder();
    
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PreComponentActionListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(PreComponentActionListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(PreComponentActionListener* listener);

    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @endif
     */
    void notify(UniqueId ec_id);
      
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };


  /*!
   * @if jp
   * @class PostComponentActionListenerHolder
   * @brief PostComponentActionListener ۥ饹
   *
   * ʣ PostComponentActionListener ݻ륯饹
   *
   * @else
   * @class PostComponentActionListenerHolder
   * @brief PostComponentActionListener holder class
   *
   * This class manages one ore more instances of
   * PostComponentActionListener class.
   *
   * @endif
   */
  class PostComponentActionListenerHolder
  {
    typedef std::pair<PostComponentActionListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    PostComponentActionListenerHolder();
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PostComponentActionListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(PostComponentActionListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(PostComponentActionListener* listener);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @param cdrdata ǡ
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @param cdrdata Data
     * @endif
     */
    void notify(UniqueId ec_id, ReturnCode_t ret);
    
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };


  //============================================================
  /*!
   * @if jp
   * @class PortActionListenerHolder
   * @brief PortActionListener ۥ饹
   *
   * ʣ PortActionListener ݻ륯饹
   *
   * @else
   * @class PortActionListenerHolder
   * @brief PortActionListener holder class
   *
   * This class manages one ore more instances of
   * PortActionListener class.
   *
   * @endif
   */
  class PortActionListenerHolder
  {
    typedef std::pair<PortActionListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    PortActionListenerHolder();
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PortActionListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(PortActionListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(PortActionListener* listener);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @param cdrdata ǡ
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @param cdrdata Data
     * @endif
     */
    void notify(const RTC::PortProfile& pprofile);
    
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };

  /*!
   * @if jp
   * @class ExecutionContextActionListenerHolder
   * @brief ExecutionContextActionListener ۥ饹
   *
   * ʣ ExecutionContextActionListener ݻ륯饹
   *
   * @else
   * @class ExecutionContextActionListenerHolder
   * @brief ExecutionContextActionListener holder class
   *
   * This class manages one ore more instances of
   * ExecutionContextActionListener class.
   *
   * @endif
   */
  class ExecutionContextActionListenerHolder
  {
    typedef std::pair<ExecutionContextActionListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    ExecutionContextActionListenerHolder();
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~ExecutionContextActionListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(ExecutionContextActionListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(ExecutionContextActionListener* listener);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @param cdrdata ǡ
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @param cdrdata Data
     * @endif
     */
    void notify(UniqueId ec_id);
    
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };


  /*!
   * @if jp
   * @class ComponentActionListeners
   * @brief ComponentActionListeners 饹
   *
   *
   * @else
   * @class ComponentActionListeners
   * @brief ComponentActionListeners class
   *
   *
   * @endif
   */
  class ComponentActionListeners
  {
  public:
    /*!
     * @if jp
     * @brief PreComponentActionListenerTypeꥹ
     * PreComponentActionListenerTypeꥹʤǼ
     * @else
     * @brief PreComponentActionListenerType listener array
     * The PreComponentActionListenerType listener is stored. 
     * @endif
     */
    PreComponentActionListenerHolder 
    preaction_[PRE_COMPONENT_ACTION_LISTENER_NUM];
    /*!
     * @if jp
     * @brief PostComponentActionTypeꥹ
     * PostComponentActionTypeꥹʤǼ
     * @else
     * @brief PostComponentActionType listener array
     * The PostComponentActionType listener is stored.
     * @endif
     */
    PostComponentActionListenerHolder 
    postaction_[POST_COMPONENT_ACTION_LISTENER_NUM];
    /*!
     * @if jp
     * @brief PortActionTypeꥹ
     * PortActionTypeꥹʤǼ
     * @else
     * @brief PortActionType listener array
     * The PortActionType listener is stored.
     * @endif
     */
    PortActionListenerHolder
    portaction_[PORT_ACTION_LISTENER_NUM];
    /*!
     * @if jp
     * @brief ExecutionContextActionTypeꥹ
     * ExecutionContextActionTypeꥹʤǼ
     * @else
     * @brief ExecutionContextActionType listener array
     * The ExecutionContextActionType listener is stored.
     * @endif
     */
   ExecutionContextActionListenerHolder
    ecaction_[EC_ACTION_LISTENER_NUM];
  };


}; // namespace RTC

#endif // RTC_COMPONENTACTIONLISTENER_H
