// -*- C++ -*-
/*!
 * @file PortAdmin.cpp
 * @brief RTC's Port administration class
 * @date $Date: 2008-01-14 07:56:11 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2010
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: PortAdmin.cpp 2132 2011-05-24 03:56:21Z n-ando $
 *
 */

#include <functional>
#include <rtm/RTC.h>
#include <rtm/PortAdmin.h>
#include <rtm/CORBA_SeqUtil.h>

namespace RTC
{
  /*!
   * @if jp
   * @brief Portѥե󥯥
   * @else
   * @brief Functor to fing a Port
   * @endif
   */
  struct PortAdmin::find_port_name
  {
    find_port_name(const char* name) : m_name(name) {};
    bool operator()(const PortService_ptr& p)
    {
      try
        {
#ifndef ORB_IS_RTORB
          PortProfile_var prof(p->get_port_profile());
          std::string name(prof->name);
#else // ORB_IS_RTORB
          PortProfile *pp;
          pp = p->get_port_profile();
          std::string name( pp->name);
          delete pp;
#endif // ORB_IS_RTORB
          return m_name == name;
        }
      catch (...)
        {
          return false;
        }
      return false;
    }
    const std::string m_name;
  };

  struct PortAdmin::find_port
  {
    find_port(const PortService_ptr& p) : m_port(p) {};
    bool operator()(const PortService_ptr& p)
    {
      return m_port->_is_equivalent(p);
    }
    const PortService_ptr m_port;
  };
  
  /*!
   * @if jp
   * @brief Portѥե󥯥
   * @else
   * @brief Functor to delete the Port
   * @endif
   */
  struct PortAdmin::del_port
  {
    PortAdmin* m_pa;
    del_port(PortAdmin* pa) : m_pa(pa) {};
    void operator()(PortBase* p)
    {
      m_pa->removePort(*p);
    }
  };
  
  /*!
   * @if jp
   * @brief 󥹥ȥ饯
   * @else
   * @brief Constructor
   * @endif
   */
  PortAdmin::PortAdmin(CORBA::ORB_ptr orb, PortableServer::POA_ptr poa)
    : m_pORB(CORBA::ORB::_duplicate(orb)),
      m_pPOA(PortableServer::POA::_duplicate(poa)),
      rtclog("portadmin")
  {
  }
  
  /*!
   * @if jp
   * @brief PortServiceList μ
   * @else
   * @brief Get PortServiceList
   * @endif
   */
  PortServiceList* PortAdmin::getPortServiceList() const
  {
    PortServiceList_var ports;
    ports = new PortServiceList(m_portRefs);
    return ports._retn();
  }
  
  /*!
   * @if jp
   * @brief PortProfileList μ
   * @else
   * @brief Get PortProfileList
   * @endif
   */
  PortProfileList PortAdmin::getPortProfileList() const
  {

#ifndef ORB_IS_RTORB
    PortProfileList port_profs;
    //    port_prof_collect p(port_profs);
    port_prof_collect2 p(port_profs);
    //    m_portServants.for_each(p);
    ::CORBA_SeqUtil::for_each(m_portRefs, p);
#else // ORB_IS_RTORB
    CORBA::ULong len = m_portRefs.length();

    PortProfileList port_profs = PortProfileList(len);
    PortProfile *pp;

    for (CORBA::ULong i (0); i < len; ++i)
      {
        try
          {
            pp = m_portRefs[i]->get_port_profile();
            port_profs[i] = *(pp);
            delete pp;
          }
        catch (...)
          {
            ;
          }
      }
#endif // ORB_IS_RTORB
    return port_profs;
  }
  
  /*!
   * @if jp
   * @brief Port Υ֥ȻȤμ
   * @else
   * @brief Get the reference to the Port's object
   * @endif
   */
  PortService_ptr PortAdmin::getPortRef(const char* port_name) const
  {
    CORBA::Long index;
    index = CORBA_SeqUtil::find(m_portRefs, find_port_name(port_name));
    if (index >= 0) 
      {//throw NotFound(port_name);
	return m_portRefs[index];
      }
    return RTC::PortService::_nil();
  }
  
  /*!
   * @if jp
   * @brief Port ΥХȤΥݥ󥿤μ
   * @else
   * @brief Get pointer to the Port's servant
   * @endif
   */
  PortBase* PortAdmin::getPort(const char* port_name) const
  {
    return m_portServants.find(port_name);
  }
  
  /*!
   * @if jp
   * @brief Port Ͽ
   * @else
   * @brief Regsiter the Port
   * @endif
   */
  bool PortAdmin::addPort(PortBase& port)
  {
    // Check for duplicate
    if (CORBA_SeqUtil::find(m_portRefs, find_port_name(port.getName())) != -1)
      {
        return false;
      }

    // Store Port's ref to PortServiceList
    CORBA_SeqUtil::push_back(m_portRefs,
                             RTC::PortService::_duplicate(port.getPortRef()));
    
    // Store Port servant
    return m_portServants.registerObject(&port);
  }

  /*!
   * @if jp
   * @brief Port Ͽ
   * @else
   * @brief Regsiter the Port
   * @endif
   */
  bool PortAdmin::addPort(PortService_ptr port)
  {
    // Check for duplicate
    try
      {
        PortProfile_var prof(port->get_port_profile());
        std::string name(prof->name);
        
// Why RtORB delete _var object explicitly?
#ifdef ORB_IS_RTORB
        delete prof._retn();
#endif
        if (CORBA_SeqUtil::find(m_portRefs, find_port_name(name.c_str())) != -1)
          {
            return false;
          }
      }
    catch (...)
      {
        return false;
      }
    CORBA_SeqUtil::push_back(m_portRefs, RTC::PortService::_duplicate(port));
    return true;
  }

  void PortAdmin::registerPort(PortBase& port)
  {
    if (!addPort(port))
      {
	RTC_ERROR(("registerPort(PortBase&) failed."));
      }
  }
  
  void PortAdmin::registerPort(PortService_ptr port)
  {
    if (!addPort(port))
      {
	RTC_ERROR(("registerPort(PortService_ptr) failed."));
      }
  }
  
  /*!
   * @if jp
   * @brief Port Ͽ
   * @else
   * @brief Unregister the Port's registration
   * @endif
   */
  bool PortAdmin::removePort(PortBase& port)
  {
    try
      {
	port.disconnect_all();
	// port.shutdown();
	
	const char* tmp(port.getProfile().name);
	CORBA_SeqUtil::erase_if(m_portRefs, find_port_name(tmp));
	
        PortableServer::ObjectId_var oid = m_pPOA->servant_to_id(&port);
	m_pPOA->deactivate_object(oid);
	port.setPortRef(RTC::PortService::_nil());
	
	return m_portServants.unregisterObject(tmp) == NULL ? false : true;
      }
    catch (...)
      {
	return false;
      }
  }

  bool PortAdmin::removePort(PortService_ptr port)
  {
    try
      {
	CORBA_SeqUtil::erase_if(m_portRefs, find_port(port));
	return true;
      }
    catch (...)
      {
	return false;
      }
  }

  void PortAdmin::deletePort(PortBase& port)
  {
    if (!removePort(port))
      {
	RTC_ERROR(("deletePort(PortBase&) failed."));
      }
  }
  void PortAdmin::deletePort(PortService_ptr port)
  {
    if (!removePort(port))
      {
	RTC_ERROR(("deletePort(PortService_ptr) failed."));
      }
  }

  
  /*!
   * @if jp
   * @brief ̾λˤPort Ͽ
   * @else
   * @brief Unregister the Port's registration by its name
   * @endif
   */
  void PortAdmin::deletePortByName(const char* port_name)
  {
    if (!port_name) return;
    PortBase& p(*m_portServants.find(port_name));
    removePort(p);
  }

  /*!
   * @if jp
   * @brief Ƥ Port Υ󥿡ե activates 
   * @else
   * @brief Activate all Port interfaces
   * @endif
   */
  void PortAdmin::activatePorts()
  {
    std::vector<PortBase*> ports;
    ports = m_portServants.getObjects();
    for (int i(0), len(ports.size()); i < len; ++i)
      {
        ports[i]->activateInterfaces();
      }
  }

  /*!
   * @if jp
   * @brief Ƥ Port Υ󥿡ե deactivates 
   * @else
   * @brief Deactivate all Port interfaces
   * @endif
   */
  void PortAdmin::deactivatePorts()
  {
    std::vector<PortBase*> ports;
    ports = m_portServants.getObjects();
    for (int i(0), len(ports.size()); i < len; ++i)
      {
        ports[i]->deactivateInterfaces();
      }
  }

  /*!
   * @if jp
   * @brief Ƥ Port deactivateϿ
   * @else
   * @brief Deactivate all Ports and unregister them
   * @endif
   */
  void PortAdmin::finalizePorts()
  {
    deactivatePorts();
    std::vector<PortBase*> ports;
    ports = m_portServants.getObjects();
    for_each(ports.begin(), ports.end(), del_port(this));
  }
};
