
#include <iostream>
#include <vector>
#include <fstream>
#include <boost/program_options.hpp>
#include <boost/program_options/errors.hpp>
#include <boost/timer/timer.hpp>

#include <omp.h>
#include <signal.h>

#include <ros/ros.h>
#include <ros/package.h>

#include "hough_feature_clustering.h"

struct ResultComp
{
    bool operator()(ResultData r1, ResultData r2)
    {
        return r1.confidence > r2.confidence;
    }
} result_comp;


// global variables to write results to disk
std::string output_path;
std::map<std::string, std::vector<ObjectResult> > rgb_results_map;
std::map<std::string, std::vector<ObjectResult> > shape_results_map;
std::map<std::string, std::vector<ObjectResult> > combined_results_map;
std::ostringstream processing_time;


void write_results_to_disk(std::map<std::string, std::vector<ObjectResult> > results_map, std::string result_type)
{
    // ----- write results to files -----


    std::vector<std::string> subfolders;
    std::string command = "mkdir -p ";

    if(result_type == "rgb")
    {
        // create folder for output
        output_path = ros::package::getPath("or_nodes") + "/output/rgb/";

        command.append(output_path);
        int sys = std::system(command.c_str());
        std::cout << "executing command: '" << command << "'" << std::endl;
        sleep(1);

        // subfolders for different color hist strategies
        subfolders.push_back("color_instances_best/");
        subfolders.push_back("color_instances_average/");
        subfolders.push_back("color_classes_best/");
        subfolders.push_back("color_classes_average/");
        subfolders.push_back("hough_clustering/");
    }
    else if(result_type == "depth")
    {
        // create folder for output
        output_path = ros::package::getPath("or_nodes") + "/output/depth/";

        command.append(output_path);
        int sys = std::system(command.c_str());
        std::cout << "executing command: '" << command << "'" << std::endl;
        sleep(1);

        // subfolders for different color hist strategies
        subfolders.push_back("knn-1/");
        subfolders.push_back("knn-2/");
        subfolders.push_back("knn-3/");
        subfolders.push_back("knn-4/");
    }
    else if(result_type == "combined")
    {
        // create folder for output
        output_path = ros::package::getPath("or_nodes") + "/output/combined/";

        command.append(output_path);
        int sys = std::system(command.c_str());
        std::cout << "executing command: '" << command << "'" << std::endl;
        sleep(1);

        // subfolders for different color hist strategies
        subfolders.push_back("class_from_best_knn-1-2_color_class_best/");
        subfolders.push_back("class_from_best_knn-4_color_class_best/");
        subfolders.push_back("class_from_all_knn-1-2_color_class_best/");
        subfolders.push_back("class_from_all_knn-4_color_class_best/");
        subfolders.push_back("color_instance_best_or_knn1_best/");
        subfolders.push_back("color_instance_best_or_knn4_best/");
        subfolders.push_back("color_class_best_or_knn2_best/");
        subfolders.push_back("color_class_best_or_knn4_best/");
        subfolders.push_back("color_instance_best_or_knn1_random/");
        subfolders.push_back("color_class_best_or_knn2_random/");
        subfolders.push_back("color_class_best_or_knn4_random/");
    }


    // create separate results map for each strategy / subfolder
    for(unsigned res = 0; res < subfolders.size(); res++)
    {
        // prepare summary
        std::ofstream summary_file;
        int global_num_correct_classes = 0; // counts all correct class recognitions
        int global_num_correct_instances = 0; // counts all correct instance recognitions
        int global_num_2nd_correct_classes = 0; // counts how often the correct class was 2nd in hypotheses list
        int global_num_2nd_correct_instances = 0; // counts how often the correct class was 2nd in hypotheses list
        int global_num_objects = 0; // counts all tested images
        int global_correct_instance_in_list = 0; // counts how often the correct instance was in the hypothesis list
        int global_correct_classes_in_list = 0;// counts how often the correct class was in the hypothesis list
        int global_no_hypotheses = 0; // counts objects that had 0 hypotheses
        int global_total_hypotheses = 0; // counts all hypotheses from all objects to determine the average length of hypothesis list


        std::string local_output_path = output_path + subfolders.at(res);
        command = "mkdir -p ";
        command.append(local_output_path);
        int sys = std::system(command.c_str());
        sleep(1);

        std::cout << "executing command: '" << command << "'" << std::endl;

        std::map<std::string, ObjectResult> local_results_map;

        // split combined results_map into separate result maps
        for(std::map<std::string, std::vector<ObjectResult> >::iterator it = results_map.begin(); it != results_map.end(); it++)
        {
            local_results_map.insert(std::make_pair<std::string, ObjectResult>(it->first, it->second.at(res)));
        }


        // summary file
        std::string summary_file_name = local_output_path + "zzz_summary.txt";
        summary_file.open(summary_file_name.c_str(), std::ios::out);

        // object file
        std::ofstream object_file;
        std::string file_name;
        // get first name for first output file
        std::map<std::string, ObjectResult>::iterator it = local_results_map.begin();
        if(it != local_results_map.end())
        {
            file_name = it->second.gt_instance_name;
        }
        object_file.open((local_output_path + file_name).c_str(), std::ios::out);

        // init variables
        int num_objects = 0;
        int correct_classes = 0;
        int correct_instances = 0;

        for(std::map<std::string, ObjectResult>::iterator it = local_results_map.begin(); it != local_results_map.end(); it++)
        {
            // as soon as we arrive at the next object, open new output file for it
            if(it->second.gt_instance_name != file_name)
            {
                // finish previous file
                object_file << std::endl << std::endl;
                object_file << "Summary for " << file_name << ": ";
                object_file << "  correct instances: " << correct_instances << " of " << num_objects << " (" << ((float)correct_instances/(float)num_objects) << ")";
                object_file << "  correct classes: " << correct_classes << " of " << num_objects << " (" << ((float)correct_classes/(float)num_objects) << ")";
                object_file.close();

                // write to summary file
                summary_file << file_name << ": " << "  correct instances: " << correct_instances << " of " << num_objects << " (" << ((float)correct_instances/(float)num_objects) <<
                                "), correct classes: " << correct_classes << " of " << num_objects << " ( " << ((float)correct_classes/(float)num_objects) << ")" << std::endl;
                // re-init variables
                num_objects = 0;
                correct_classes = 0;
                correct_instances = 0;

                // open next file
                file_name = it->second.gt_instance_name;
                object_file.open((local_output_path + file_name).c_str(), std::ios::out);
            }

            // write object info to file
            object_file << it->first << " - ground_truth (instance, class):   "
                        << it->second.gt_instance_name << ", " << it->second.gt_class_name << std::endl;
            object_file << "results (confidence / instance / class)" << std::endl;

            // sort results
            std::vector<ResultData> temp_res = it->second.results;
            std::sort(temp_res.begin(), temp_res.end(), result_comp);

            // update counters
            num_objects++;
            global_num_objects++;
            if(temp_res.size() > 0)
            {
                // count correct recognitions
                if(it->second.gt_class_name == temp_res.at(0).class_name)
                {
                    correct_classes++;
                    global_num_correct_classes++;
                }
                if(it->second.gt_instance_name == temp_res.at(0).instance_name)
                {
                    correct_instances++;
                    global_num_correct_instances++;
                }
                // count correct recognitions in 2nd place of hypotheses list
                if(temp_res.size() > 1)
                {
                    // only count classes if correct class is on 2nd place, but not on 1st place
                    if(it->second.gt_class_name == temp_res.at(1).class_name && it->second.gt_class_name != temp_res.at(0).class_name)
                    {
                        global_num_2nd_correct_classes++;
                    }
                    if(it->second.gt_instance_name == temp_res.at(1).instance_name)
                    {
                        global_num_2nd_correct_instances++;
                    }
                }
                // count how often the correct instance was in the hypotheses list
                bool class_counted = false;
                for(unsigned i = 0; i < temp_res.size(); i++)
                {
                    if(it->second.gt_instance_name == temp_res.at(i).instance_name)
                    {
                        global_correct_instance_in_list++;
                    }
                    if(it->second.gt_class_name == temp_res.at(i).class_name && !class_counted)
                    {
                        class_counted = true;
                        global_correct_classes_in_list++;
                    }
                }
                // count number of hypotheses
                global_total_hypotheses += temp_res.size();
            }
            else
            {
                // count images without hypotheses
                global_no_hypotheses++;
            }

            // write results to file
            for(unsigned i = 0; i < temp_res.size(); i++)
            {
                ResultData res = temp_res.at(i);
                object_file << "    " << res.confidence << ", " << res.instance_name << ", " << res.class_name << std::endl;
            }
            object_file << std::endl << std::endl;
        }

        // finish last file
        object_file << std::endl << std::endl;
        object_file << "Summary for " << file_name << ": ";
        object_file << "  correct instances: " << correct_instances << " of " << num_objects << " (" << ((float)correct_instances/(float)num_objects) << ")";
        object_file << "  correct classes: " << correct_classes << " of " << num_objects << " (" << ((float)correct_classes/(float)num_objects) << ")";
        object_file.close();

        // write last info to summary file
        summary_file << file_name << ": " << "  correct instances: " << correct_instances << " of " << num_objects << " (" << ((float)correct_instances/(float)num_objects) <<
                        "), correct classes: " << correct_classes << " of " << num_objects << " ( " << ((float)correct_classes/(float)num_objects) << ")" << std::endl;
        // write summary of summary
        summary_file << std::endl << std::endl;
        summary_file << "Summary for all objects:" << std::endl;
        summary_file << "  correct instances: " << global_num_correct_instances << " of " << global_num_objects << " (" << ((float)global_num_correct_instances/(float)global_num_objects) << ")" << std::endl;
        summary_file << "  correct classes: " << global_num_correct_classes << " of " << global_num_objects << " (" << ((float)global_num_correct_classes/(float)global_num_objects) << ")" << std::endl;
        summary_file << std::endl;
        summary_file << "  correct instances are 2nd in list: " << global_num_2nd_correct_instances << " of " << global_num_objects << " (" << ((float)global_num_2nd_correct_instances/(float)global_num_objects) << ")" << std::endl;
        summary_file << "  correct classes are 2nd in list: " << global_num_2nd_correct_classes << " of " << global_num_objects << " (" << ((float)global_num_2nd_correct_classes/(float)global_num_objects) << ")" << std::endl;
        summary_file << std::endl;
        summary_file << "  correct instance in list: " << global_correct_instance_in_list << " of " << global_num_objects << " (" << ((float)global_correct_instance_in_list/(float)global_num_objects) << ")" << std::endl;
        summary_file << "  correct classes in list: " << global_correct_classes_in_list << " of " << global_num_objects << " (" << ((float)global_correct_classes_in_list/(float)global_num_objects) << ")" << std::endl;
        summary_file << std::endl;
        summary_file << "  files with 0 hypotheses: " << global_no_hypotheses << " of " << global_num_objects << " (" << ((float)global_no_hypotheses/(float)global_num_objects) << ")" << std::endl;
        summary_file << "  average number of hypotheses per object: " << ((float)global_total_hypotheses/(float)global_num_objects) << std::endl;
        summary_file << std::endl;
        summary_file << "Processing time: " << processing_time.str();
        // close summary file
        summary_file.close();

    }
}


void my_handler(int s)
{
    std::cout << "Caught SIGINT" << std::endl;
    // write results to disk
    write_results_to_disk(rgb_results_map, "rgb");
    write_results_to_disk(shape_results_map, "depth");
    write_results_to_disk(combined_results_map, "combined");
    exit(1);
}


int main(int argc, char **argv)
{
    signal (SIGINT, my_handler);

    ros::init(argc, argv, "hfc_eval_tool");
    ros::NodeHandle nh;

    boost::program_options::options_description generic("Generic options");
    boost::program_options::options_description training("Training");
    boost::program_options::options_description detection("Detection");

    generic.add_options()
            ("help,h", "Display this help message");
    // TODO ("output,o", boost::program_options::value<std::string>(), "The output folder (created automatically) for ism files after training or the detection log after detection");

    training.add_options()
            ("train,t", boost::program_options::value<std::string>(), "Train object models")
            ("images,i", boost::program_options::value<std::vector<std::string> >()->multitoken()->composing(), "Specifiy a list of training images")
            ("names,n", boost::program_options::value<std::vector<std::string> >()->multitoken()->composing(), "Specifiy a list of object instance names for the given training images")
            ("classes,c", boost::program_options::value<std::vector<std::string> >()->multitoken()->composing(), "Specifiy a list of class ids for the given training images (several object instances belong to the same class)");
    // TODO add a mapping from class numbers to class names as string

    //TODO
    detection.add_options()
            ("recognize,r", boost::program_options::value<std::string>(), "Recognize objects in images")
            ("test_images,a", boost::program_options::value<std::vector<std::string> >()->multitoken()->composing(), "Specify a list of test images")
            ("groundtruth_names,g", boost::program_options::value<std::vector<std::string> >()->multitoken()->composing(), "Specifiy a list of ground truth (instance) object names for the given images")
            ("groundtruth_classes,d", boost::program_options::value<std::vector<std::string> >()->multitoken()->composing(), "Specifiy a list of ground truth class ids for the given images)");


    boost::program_options::options_description desc;
    desc.add(generic).add(training).add(detection);

    // parse command line arguments
    boost::program_options::variables_map variables;
    try
    {
        boost::program_options::store(boost::program_options::command_line_parser(argc, argv).options(desc).run(), variables);
        boost::program_options::notify(variables);
    }
    catch (std::exception& e)
    {
        std::cerr << e.what() << std::endl;
    }

    // show help
    if (variables.count("help") || variables.size() == 0)
    {
        std::cout << desc << std::endl;
        return 1;
    }
    else
    {
        try {

            // -------- train objects ---------
            if (variables.count("train"))
            {
                std::cout << "starting the training process" << std::endl;
                std::string config_file = variables["train"].as<std::string>();

                // TODO read config file
                bool has_background = false; // TODO read this from config file

                // create hough clustering object
                HoughFeatureClustering hfc(&nh, 0, true);
                ros::Duration(3).sleep(); // wait for topic to be advertised

                // collect information for training
                std::map<std::string, ObjectModel> object_model_map;

                // add the training images
                if (variables.count("images") && variables.count("names") && variables.count("classes"))
                {
                    std::vector<std::string> images = variables["images"].as<std::vector<std::string> >();
                    std::vector<std::string> names = variables["names"].as<std::vector<std::string> >();
                    std::vector<std::string> classes = variables["classes"].as<std::vector<std::string> >();

                    if (images.size() == classes.size() && images.size() == names.size())
                    {
                        for (int i = 0; i < (int)images.size(); i++)
                        {
                            std::string image_file_name = images[i];
                            std::string class_name = classes[i];
                            std::string instance_name = names[i];

                            // object already exists
                            if(object_model_map.find(instance_name) != object_model_map.end())
                            {
                                ObjectModel om = object_model_map[instance_name];
                                om.image_file_names.push_back(image_file_name);
                                object_model_map[instance_name] = om;
                            }
                            else // add new object to map
                            {
                                ObjectModel om;
                                om.class_name = class_name;
                                om.instance_name = instance_name;
                                om.image_file_names.push_back(image_file_name);
                                object_model_map.insert(std::make_pair<std::string, ObjectModel>(instance_name,om));
                                // TODO use this with cpp 11:
                                //object_model_map.insert({instance_name,om});
                            }
                        }
                    }
                    else
                    {
                        ROS_ERROR_STREAM("number of images does not match the number of classes or the number of instance names");
                        return 1;
                    }
                }
                else
                {
                    ROS_ERROR_STREAM("Can not proceed: images, names or classes missing!");
                }

                // train
                bool check = true;
                // TODO use auto type
                for(std::map<std::string, ObjectModel>::iterator it = object_model_map.begin(); it != object_model_map.end(); it++)
                {
                    ObjectModel om = it->second;
//                                    if(check) // TODO remove this
//                                    {
//                                        if(om.instance_name != "food_can_2")
//                                        {
//                                            ROS_INFO_STREAM("skipping object " << om.instance_name);
//                                            continue;
//                                        }
//                                        else
//                                        {
//                                            check = false;
//                                        }
//                                    }
                    ROS_INFO_STREAM("Training object " << om.instance_name);
                    if(om.image_file_names.size() >= 2)
                    {
                        // NOTE: learnt file is saved inside the output folder of the object_recognition package
                        //hfc.train(om.class_name, om.instance_name, om.image_file_names, has_background);
                        // TODO temp: should be part of normal object learning
                        //hfc.train_color_histogram(om.class_name, om.instance_name, om.image_file_names, has_background);
                        hfc.train_shape(om.class_name, om.instance_name, om.image_file_names);
                        // hfc.train_shape_with_color(om.class_name, om.instance_name, om.image_file_names);
                    }
                    else
                    {
                        ROS_ERROR_STREAM("Skipping object \"" << om.instance_name << "\" of class \"" << om.class_name << "\" - not enough images.");
                        ROS_ERROR_STREAM("    There have to be at least 2 images: one background image and one object image!");
                    }

                    while(!hfc.isReady() && ros::ok())
                    {
                        ROS_INFO_STREAM("... not ready, yet ...");
                        //ros::spinOnce();
                        hfc.call_available();
                        ros::Duration(1).sleep();
                    }
                    ROS_INFO_STREAM("... ready for next object ...");
                }
                // TODO temp: should be part of normal object learning
                //hfc.save_color_hist_file();
            }






            // ---------- recognize objects -----------

            if (variables.count("recognize"))
            {
                std::cout << "starting the recognition process" << std::endl;
                std::string config_file = variables["recognize"].as<std::string>();

                // TODO read config_file
                // read all objects inside the objectProperties folder in the object_recognition package
                bool load_all_objects = true; // TODO read this from config file
                // if(!load_all_objects) ---> TODO  load only objects from config file
                std::vector<std::string> objects_to_load;
                bool one_object_per_image = true; // TODO read this from config file
                output_path = ""; //TODO read this from config file

                // set number of threads
                int m_num_threads = 1;

                std::vector<HoughFeatureClustering*> hfc_vector;
                for(unsigned i = 0; i < m_num_threads; i++)
                {
                    // create hough clustering objects
                    HoughFeatureClustering* hfc = new HoughFeatureClustering(&nh, i);
                    hfc_vector.push_back(hfc);
                }
                ros::Duration(3).sleep(); // wait for topics to be advertised

                // collect information for testing
                std::map<std::string, ObjectModel> object_model_map;

                if (variables.count("test_images") && variables.count("groundtruth_names") && variables.count("groundtruth_classes"))
                {
                    std::vector<std::string> images = variables["test_images"].as<std::vector<std::string> >();
                    std::vector<std::string> names = variables["groundtruth_names"].as<std::vector<std::string> >();
                    std::vector<std::string> classes = variables["groundtruth_classes"].as<std::vector<std::string> >();

                    if (images.size() == classes.size() && images.size() == names.size())
                    {
                        // create object information from command line parameters
                        for (int i = 0; i < (int)images.size(); i++)
                        {
                            std::string image_file_name = images[i];
                            std::string class_name = classes[i];
                            std::string instance_name = names[i];

//            // TODO only for testing partial data --- REMOVE
//            if(class_name != "apple" && class_name != "ball" && class_name != "banana" && class_name != "bell_pepper"
//                    && class_name != "binder" && class_name != "food_bag") continue;

//        // TODO REMOVE  // partial evaluation
//        if(i % 3 != 0) continue;


                            // object already exists
                            if(object_model_map.find(instance_name) != object_model_map.end())
                            {
                                ObjectModel om = object_model_map[instance_name];
                                om.image_file_names.push_back(image_file_name);
                                object_model_map[instance_name] = om;
                            }
                            else // add new object to map
                            {
                                ObjectModel om;
                                om.class_name = class_name;
                                om.instance_name = instance_name;
                                om.image_file_names.push_back(image_file_name);
                                object_model_map.insert(std::make_pair<std::string, ObjectModel>(instance_name,om));
                                // TODO use this with cpp 11:
                                //object_model_map.insert({instance_name,om});
                            }
                        }

                        //                        ROS_WARN_STREAM("object model map size: " << object_model_map.size());
                        //                        sleep(10);

                        // load object data for recognition
                        // collect all files
                        if(load_all_objects)
                        {
                            objects_to_load.clear();
                            // add all objects in default object directory
                            std::string path = ros::package::getPath("or_nodes");
                            path += "/objectProperties/";
                            std::string ls_file_name = "all_files.txt";
                            std::string in_file_name = path + ls_file_name;
                            std::string command = "ls " + path + " > " + in_file_name;
                            int sys = std::system(command.c_str());
                            sleep(1);

                            // process file listing
                            std::ifstream file;
                            std::string line;
                            file.open(in_file_name.c_str());
                            while(getline(file,line))
                            {
                                if(line == ls_file_name) continue; // skip the filename itself
                                objects_to_load.push_back(line);
                            }
                            file.close();

                            // delete file that was just created
                            std::string rm_command = "rm " + in_file_name;
                            sys = std::system(rm_command.c_str());
                        }
                        // TODO temp: should be part of normal object loading
                        #pragma omp parallel num_threads(m_num_threads)
                        {
                            int tid = omp_get_thread_num();
                            hfc_vector.at(tid)->load_color_hist_file();
                        }


                        // load each file in each thread
                        #pragma omp parallel num_threads(m_num_threads)
                        for(unsigned i = 0; i < objects_to_load.size(); i++)
                        {
                            int tid = omp_get_thread_num();
                            ROS_INFO_STREAM("Thread " << tid << ": Loading object " << (i+1) << " of " << objects_to_load.size() << ": " << objects_to_load.at(i));
                            hfc_vector.at(tid)->load_object(objects_to_load.at(i));
                            // NOTE: works for pure shape, as well as shape+color objects
                            hfc_vector.at(tid)->load_shape_object(objects_to_load.at(i));

                            //while(!hfc->isReady() && ros::ok())
                            while(!hfc_vector.at(tid)->isReady() && ros::ok())
                            {
                                //ROS_INFO_STREAM("Thread " << tid << " not ready, yet ...");
                                //ros::spinOnce();
                                hfc_vector.at(tid)->call_available();
                                ros::Duration(0.1).sleep();
                            }
                            //if(!ros::ok()) return 1;
                        }

                        // recognize images
                        int counter = 0;

                        boost::timer::cpu_timer timer;

                        // prepare to split threads
                        std::vector<ObjectModel> object_model_vector;
                        for(std::map<std::string, ObjectModel>::iterator it = object_model_map.begin(); it != object_model_map.end(); it++)
                        {
                            object_model_vector.push_back(it->second);
                        }

                        for(unsigned it = 0; it < object_model_vector.size(); it++)
                            //for(std::map<std::string, ObjectModel>::iterator it = object_model_map.begin(); it != object_model_map.end(); it++)
                        {
                            ObjectModel om = object_model_vector.at(it);
                            //ObjectModel om = it->second;
                            counter++;
                            // split threads for parallel processing
                            #pragma omp parallel for num_threads(m_num_threads)
                            for(unsigned obj = 0; obj < om.image_file_names.size(); obj++)
                            {
                                int tid = omp_get_thread_num();
                                ROS_WARN_STREAM("Object: " << counter << ", Image " << (obj+1)  << " of " << om.image_file_names.size());
                                // recognize
                                std::string file_name = om.image_file_names.at(obj);
                                hfc_vector.at(tid)->recognize(file_name, one_object_per_image);
                                hfc_vector.at(tid)->recognize_shape(file_name, one_object_per_image);
                                //hfc_vector.at(tid)->recognize_shape_with_color(file_name, one_object_per_image);

                                // wait for recognition
                                //while(!hfc->isReady() && ros::ok())
                                while(!hfc_vector.at(tid)->isReady() && ros::ok())
                                {
                                    //ROS_INFO_STREAM("Thread " << tid << " not ready, yet ...");
                                    //ros::spinOnce();
                                    hfc_vector.at(tid)->call_available();
                                    ros::Duration(0.1).sleep();
                                }

                                // save results
                                #pragma omp critical
                                {
                                    // add new object to rgb map
                                    {
                                        std::vector<std::vector<ResultData> > res_data = hfc_vector.at(tid)->get_last_results();
                                        std::vector<ObjectResult> obj_results;
                                        for(unsigned res = 0; res < res_data.size(); res++)
                                        {
                                            ObjectResult obr;
                                            obr.gt_class_name = om.class_name;
                                            obr.gt_instance_name = om.instance_name;
                                            obr.results.insert(obr.results.begin(), res_data.at(res).begin(),res_data.at(res).end()); // empty vector: insert in the beginning
                                            obj_results.push_back(obr);
                                        }
                                        rgb_results_map.insert(std::make_pair<std::string, std::vector<ObjectResult> >(file_name, obj_results));
                                    }

                                    // add new object to depth map
                                    {
                                        std::vector<std::vector<ResultData> > res_data = hfc_vector.at(tid)->get_last_shape_results();
                                        std::vector<ObjectResult> obj_results;
                                        for(unsigned res = 0; res < res_data.size(); res++)
                                        {
                                            ObjectResult obr;
                                            obr.gt_class_name = om.class_name;
                                            obr.gt_instance_name = om.instance_name;
                                            obr.results.insert(obr.results.begin(), res_data.at(res).begin(),res_data.at(res).end()); // empty vector: insert in the beginning
                                            obj_results.push_back(obr);
                                        }
                                        shape_results_map.insert(std::make_pair<std::string, std::vector<ObjectResult> >(file_name, obj_results));
                                    }

                                    // add new object to combined map
                                    {
                                        std::vector<std::vector<ResultData> > res_data = hfc_vector.at(tid)->get_last_combined_results();
                                        std::vector<ObjectResult> obj_results;
                                        for(unsigned res = 0; res < res_data.size(); res++)
                                        {
                                            ObjectResult obr;
                                            obr.gt_class_name = om.class_name;
                                            obr.gt_instance_name = om.instance_name;
                                            obr.results.insert(obr.results.begin(), res_data.at(res).begin(),res_data.at(res).end()); // empty vector: insert in the beginning
                                            obj_results.push_back(obr);
                                        }
                                        combined_results_map.insert(std::make_pair<std::string, std::vector<ObjectResult> >(file_name, obj_results));
                                    }

                                    #pragma omp flush(rgb_results_map, shape_results_map, combined_results_map)
                                }
                            }

                            // --------------------------------------
                            // if(counter == 5) break; // use this to limit number of objects to be recognized
                            // --------------------------------------

                            //if(!ros::ok()) return 1;
                        }

                        processing_time << "Total processing time: " << timer.format(4, "%w") << " seconds.";

                        // write results to disk
                        write_results_to_disk(rgb_results_map, "rgb");
                        write_results_to_disk(shape_results_map, "depth");
                        write_results_to_disk(combined_results_map, "combined");
                    }
                    else
                    {
                        ROS_ERROR_STREAM("number of images does not match the number of groundtruth classes or the number of groundtruth instance names");
                        return 1;
                    }
                }
                else
                {
                    ROS_ERROR_STREAM("Can not proceed: images, names or classes missing!");
                }
            }

        }
        catch (...)
        {
            std::cerr << "an exception occurred" << std::endl;
            return 1;
        }
    }

    return 0;
}
