#ifndef HOUGH_FEATURE_CLUSTERING
#define HOUGH_FEATURE_CLUSTERING

#include <string>
#include <vector>

#include <ros/ros.h>
#include <ros/callback_queue.h>
#include <std_msgs/Empty.h>
#include <sensor_msgs/Image.h>
#include <cv_bridge/cv_bridge.h>

#include "or_msgs/OrLearnCommand.h"
#include "or_msgs/OrCommand.h"
#include "or_msgs/OrMatchResult.h"
#include "or_msgs/OrObjectNames.h"

// TODO move this later to normal object learning and don't handle it separately
#include <boost/archive/binary_iarchive.hpp>
#include <boost/archive/binary_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>
#include <boost/archive/text_oarchive.hpp>

struct ObjectModel
{
    std::string class_name;
    std::string instance_name;
    std::vector<std::string> image_file_names;
};

struct ResultData
{
    std::string instance_name;
    std::string class_name;
    float confidence;
};

struct ObjectResult
{
    // ground truth
    std::string gt_instance_name;
    std::string gt_class_name;
    // recognition results
    std::vector<ResultData> results;
};



struct Histogram
{
    std::vector<std::pair<std::string, float> > data;

    // compute histogram intersection
    float compute_similarity(Histogram hist)
    {
        if(data.size() != hist.data.size()) return 0;

        float sum = 0;
        for(unsigned i = 0; i < data.size(); i++)
        {
            sum += std::min(data.at(i).second, hist.data.at(i).second);
        }
        return sum;
    }

    // SERIALIZATION
    template<class Archive>
    void save(Archive & ar, const unsigned int version) const
    {
        unsigned size = data.size();
        ar & size;

        for(unsigned i = 0; i < data.size(); i++)
        {
            ar & data.at(i).first;
            ar & data.at(i).second;
        }
    }

    template<class Archive>
    void load(Archive & ar, const unsigned int version_b)
    {
        unsigned size;
        ar & size;

        for(unsigned i = 0; i < size; i++)
        {
            std::string first;
            float second;

            ar & first;
            ar & second;

            data.push_back(std::make_pair<std::string, float>(first, second));
        }
    }

    BOOST_SERIALIZATION_SPLIT_MEMBER()
};



class HoughFeatureClustering
{

public:
    HoughFeatureClustering(ros::NodeHandle* nh, int thread_id = 0, bool training_mode = false);
    virtual ~HoughFeatureClustering();

    // train object
    void train(std::string class_name, std::string instance_name,
               std::vector<std::string> &image_file_names, bool has_background);

    void train_shape(std::string class_name, std::string instance_name,
                                             std::vector<std::string> &image_file_names);

    void train_shape_with_color(std::string class_name, std::string instance_name,
                                             std::vector<std::string> &image_file_names);

    void train_color_histogram(std::string class_name, std::string instance_name,
                             std::vector<std::string> &image_file_names, bool has_background);

    void save_color_hist_file();

    void load_color_hist_file();

    // load objects for recognition
    void load_object(std::string file_name);

    void load_shape_object(std::string file_name);

    void unload_object(std::string file_name);

    // recognize object
    void recognize(std::string image_file_name, bool one_object_per_image);

    void recognize_shape(std::string image_file_name, bool one_object_per_image);

    void recognize_shape_with_color(std::string image_file_name, bool one_object_per_image);


    void call_available();

    // get results of recognition
    std::vector<std::vector<ResultData> > get_last_results()
    {
        return m_last_results;
    }

    std::vector<std::vector<ResultData> > get_last_shape_results()
    {
        return m_last_shape_results;
    }

    std::vector<std::vector<ResultData> > get_last_combined_results();

    std::vector<std::string> remove_dublicates(std::vector<std::string> &in_list);

    void object_saved_callback(const std_msgs::Empty::ConstPtr &msg);
    void recognition_result_callback(const or_msgs::OrMatchResult::ConstPtr &msg);
    void object_loaded_callback(const or_msgs::OrObjectNames::ConstPtr &msg);

    void shape_object_saved_callback(const std_msgs::Empty::ConstPtr &msg);
    void shape_object_loaded_callback(const std_msgs::Empty::ConstPtr &msg);
    void shape_object_results(const or_msgs::OrMatchResult::ConstPtr &msg);

    bool isReady()
    {
        return m_ready && m_ready_shape;
    }

private:

    sensor_msgs::Image imgToMsg(const cv::Mat &img, std::string encoding = "bgr8");
    void send_learn_command_message(int command, std::string value);
    void send_command_message(int command, std::string value);

    // strategy: use only instances or use classes
    // return:
    //  first->  best instance for "best-only" evaluation
    //  second-> best instance for "average-best" evaluation
    std::pair<std::string, std::string> get_instance_of_best_matching_histogram(std::string strategy);

    int m_thread_id;

    float m_sleep_time;
    bool m_ready;
    bool m_ready_shape;
    bool m_one_object_per_image;
    std::string m_path;
    std::string m_path_shape;

    // vector of ResultData for each different color histogram strategy
    std::vector<std::vector<ResultData> > m_last_results;
    std::vector<std::vector<ResultData> > m_last_shape_results;
    std::vector<std::vector<ResultData> > m_last_combined_results;

    std::map<std::string, Histogram> m_color_histograms;
    Histogram m_current_hist;

    std::map<std::string, std::set<std::string> > m_class_instance_map; // maps a class name to a set of instance names

    ros::NodeHandle* m_nh;
    ros::CallbackQueue* m_queue;

    ros::Publisher m_learn_command_pub;
    ros::Publisher m_command_pub;
    ros::Publisher m_shape_instance_name_pub;
    ros::Publisher m_shape_class_name_pub;
    ros::Publisher m_shape_file_input_pub;
    ros::Publisher m_shape_with_color_file_input_pub;
    ros::Publisher m_shape_save_file_pub;
    ros::Publisher m_shape_load_object_pub;
    ros::Publisher m_shape_recognize_pub;
    ros::Publisher m_shape_with_color_recognize_pub;
    ros::Publisher m_shape_with_surf_file_input_pub;

    ros::Subscriber m_notify_object_saved_sub;
    ros::Subscriber m_or_result_sub;
    ros::Subscriber m_or_object_loaded_sub;
    ros::Subscriber m_shape_object_saved_sub;
    ros::Subscriber m_shape_object_loaded_sub;
    ros::Subscriber m_shape_object_results_sub;

    ros::ServiceClient m_histogram_service;
};

BOOST_CLASS_VERSION(Histogram, 12);

#endif // HOUGH_FEATURE_CLUSTERING
