// Software License Agreement (BSD License)
//
//   Stream Manipulator 3d - https://github.com/3DVision-Stack/stream-manipulator-3D
//   Copyright (c) 2016, Federico Spinelli (fspinelli@gmail.com)
//   All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
// * Redistributions of source code must retain the above copyright notice, this
//   list of conditions and the following disclaimer.
//
// * Redistributions in binary form must reproduce the above copyright notice,
//   this list of conditions and the following disclaimer in the documentation
//   and/or other materials provided with the distribution.
//
// * Neither the name of the copyright holder(s) nor the names of its
//   contributors may be used to endorse or promote products derived from
//   this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#include <rqt_stream_manipulator_3d/filters/frustum.h>
#include <pluginlib/class_list_macros.h>
#include <QColorDialog>

namespace rqt_sm3d
{
namespace filters
{
    void
    Frustum::init(const std::string &name)
    {
        name_ = name;
        page_= new QWidget();
        ui_.setupUi(page_);
        config = shm.segment.find<Config>((name_+"Config").c_str()).first;
        while(!config)
        {
            boost::this_thread::sleep(boost::posix_time::milliseconds(50));
            config = shm.segment.find<Config>((name_+"Config").c_str()).first;
            /* QCoreApplication::processEvents(QEventLoop::AllEvents, 100); */
        }
        {//Lock configuration mutex to read
            ShmHandler::Lock  lock(config->mtx);
            button_->setChecked(!config->disabled);
            if (!config->disabled)
                ui_.Title->setStyleSheet("background-color: green");
            ui_.pub_markerB->setChecked(config->pub_marker);
            ui_.organizedB->setChecked(config->organized);
            ui_.negativeB->setChecked(config->negative);
            ui_.fdist->setValue(config->f_dist);
            ui_.ndist->setValue(config->n_dist);
            ui_.hfov->setValue(config->h_fov);
            ui_.vfov->setValue(config->v_fov);
            QColor mcolor;
            mcolor.setRgbF(config->color_r, config->color_g, config->color_b);
            QPixmap pixm(32,32);
            pixm.fill(mcolor);
            QIcon icon(pixm);
            ui_.ColorSelectB->setIcon(icon);
            ui_.ColorSelectB->setIconSize(QSize(32,32));
        }
        ui_.Title->setText(ui_.Title->text().prepend(name_.c_str()));
        connect(button_, SIGNAL(clicked(bool)), this, SLOT(onEnableDisable(bool)));
        connect(ui_.fdist, SIGNAL(valueChanged(double)), this, SLOT(onFdistChanged(double)));
        connect(ui_.hfov, SIGNAL(valueChanged(double)), this, SLOT(onHfovChanged(double)));
        connect(ui_.ndist, SIGNAL(valueChanged(double)), this, SLOT(onNdistChanged(double)));
        connect(ui_.vfov, SIGNAL(valueChanged(double)), this, SLOT(onVfovChanged(double)));
        connect(ui_.pub_markerB, SIGNAL(clicked(bool)), this, SLOT(onPubMarks(bool)));
        connect(ui_.organizedB, SIGNAL(clicked(bool)), this, SLOT(onOrganized(bool)));
        connect(ui_.negativeB, SIGNAL(clicked(bool)), this, SLOT(onNegative(bool)));
        connect(ui_.ColorSelectB, SIGNAL(clicked()), this, SLOT(onColorSelect()));
    }

    void
    Frustum::onEnableDisable(bool checked)
    {
        if (checked)
            ui_.Title->setStyleSheet("background-color: green");
        else
            ui_.Title->setStyleSheet("background-color: red");
        ShmHandler::Lock lock(config->mtx);
        config->disabled=!checked;
    }

    void
    Frustum::onNegative(bool checked)
    {
        ShmHandler::Lock lock(config->mtx);
        config->negative=checked;
    }

    void
    Frustum::onOrganized(bool checked)
    {
        ShmHandler::Lock lock(config->mtx);
        config->organized=checked;
    }

    void
    Frustum::onPubMarks(bool checked)
    {
        ShmHandler::Lock  lock(config->mtx);
        config->pub_marker=checked;
    }

    void
    Frustum::onHfovChanged(double val)
    {
        ShmHandler::Lock  lock(config->mtx);
        config->h_fov = val;
        config->mark_changed = true;
    }
    void
    Frustum::onVfovChanged(double val)
    {
        ShmHandler::Lock  lock(config->mtx);
        config->v_fov = val;
        config->mark_changed = true;
    }
    void
    Frustum::onFdistChanged(double val)
    {
        ShmHandler::Lock  lock(config->mtx);
        config->f_dist = val;
        config->mark_changed = true;
    }
    void
    Frustum::onNdistChanged(double val)
    {
        ShmHandler::Lock  lock(config->mtx);
        config->n_dist = val;
        config->mark_changed = true;
    }
    void
    Frustum::onColorSelect()
    {
        QColor color = QColorDialog::getColor(Qt::white, page_);
        if (!color.isValid())
            return;
        QPixmap pixm(32,32);
        pixm.fill(color);
        QIcon icon(pixm);
        ui_.ColorSelectB->setIcon(icon);
        ui_.ColorSelectB->setIconSize(QSize(32,32));
        ShmHandler::Lock  lock(config->mtx);
        config->color_r = color.redF();
        config->color_g = color.greenF();
        config->color_b = color.blueF();
    }
}//ns
}//ns filters

PLUGINLIB_EXPORT_CLASS(rqt_sm3d::filters::Frustum, rqt_sm3d::Plugin);
