/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "mainwindow.h"
#include "ui_mainwindow.h"
#include "preferencesdialog.h"
#include "spcom.h"
#include "displaywidget.h"

#include <unistd.h>
#include <numeric>

#include <QFileDialog>
#include <QInputDialog>
#include <QLabel>
#include <QMessageBox>
#include <QFontDatabase>

#ifdef _WIN32
#include <windows.h>
#endif

using namespace std::chrono;
using namespace std;
using namespace cv;

MainWindow::MainWindow(QWidget *parent, QApplication& app): QMainWindow(parent),
    writeDirSelected(false), fpsLabel(nullptr), sizeLabel(nullptr), appSettings("nerian.com", "spcom"),
    fpsTimer(this), closeAfterSend(false), resizeWindow(false) {

    QObject::connect(&app, &QApplication::lastWindowClosed, this,
        &MainWindow::writeApplicationSettings);

    fpsTimer.setInterval(200);
    for(int i=0; i<6; i++) {
        fpsCounters.push_back(std::make_pair(0, steady_clock::now()));
    }
}

MainWindow::~MainWindow() {
    if(spcom != nullptr) {
        spcom->terminate();
    }
}

void MainWindow::writeApplicationSettings() {
    // Write settings before exit
    if(!fullscreen) {
        appSettings.setValue("geometry", saveGeometry());
        appSettings.setValue("state", saveState(UI_VERSION));
    }

    appSettings.setValue("remote_host", settings.remoteHost.c_str());
    appSettings.setValue("local_host", settings.localHost.c_str());
    appSettings.setValue("tcp", settings.tcp);
    appSettings.setValue("local_port", settings.localPort);
    appSettings.setValue("remote_port", settings.remotePort);
    appSettings.setValue("write_raw", settings.writeRaw16Bit);
    appSettings.setValue("write_point_cloud", settings.writePointCloud);
    appSettings.setValue("point_cloud_max_dist", settings.pointCloudMaxDist);
    appSettings.setValue("color_scheme", settings.colorScheme);
    appSettings.setValue("max_frame_rate", settings.maxFrameRate);
    appSettings.setValue("read_dir", settings.readDir.c_str());
    appSettings.setValue("write_dir", settings.writeDir.c_str());
    appSettings.setValue("zoom", settings.zoomPercent);
}

bool MainWindow::init(int argc, char** argv) {
    if(!parseOptions(argc, argv)) {
        return false;
    }

    if(!settings.nonGraphical) {
        ui.reset(new Ui::MainWindow);
        ui->setupUi(this);
    }

    QObject::connect(this, &MainWindow::asyncDisplayException, this, [this](const QString& msg){
        displayException(msg.toUtf8().constData());
        spcom->terminate();
        emit enableButtons(true, false);
    });

    QObject::connect(this, &MainWindow::updateStatusBar, this, [this](int imgWidth, int imgHeight){
        char str[20];
        snprintf(str, sizeof(str), "%d x %d pixels", imgWidth, imgHeight);
        sizeLabel->setText(str);
    });

    QObject::connect(this, &MainWindow::repaintDisplayWidget, this, [this](){
        ui->displayWidget->repaint();
    });

    QObject::connect(this, &MainWindow::updateFpsLabel, this, [this](const QString& text){
        fpsLabel->setText(text);
    });

    // Initialize window
    if(!settings.nonGraphical) {
        initGui();
        if(fullscreen) {
            makeFullscreen();
        } else {
            // Restore window position
            restoreGeometry(appSettings.value("geometry").toByteArray());
            restoreState(appSettings.value("state").toByteArray(), UI_VERSION);
        }
        show();
    }

    reinitSpcom();

    if(writeImages) {
        // Automatically start sequence grabbing if a write directory is provided
        spcom->setCaptureSequence(true);
        if(!settings.nonGraphical) {
            ui->actionCapture_sequence->setChecked(true);
        }
    }

    // FPS display
    fpsTimer.start();
    QObject::connect(&fpsTimer, &QTimer::timeout, this, &MainWindow::displayFrameRate);

    return true;
}

void MainWindow::reinitSpcom() {
    emit enableButtons(false, false);

    // Make sure all sockets are closed before creating a new object
    if(spcom != nullptr) {
        spcom->terminate();
    }

    spcom.reset(new SpCom(settings));
    spcom->setFrameDisplayCallback([this](const cv::Mat_<cv::Vec3b>& left, const cv::Mat_<cv::Vec3b>& right, bool resize){
        displayFrame(left, right, resize);
    });
    spcom->setExceptionCallback([this](const std::exception& ex){
        emit asyncDisplayException(ex.what());
    });
    spcom->setSendCompleteCallback([this]() {
        if(!settings.nonGraphical) {
            ui->actionSend_images->setChecked(false);
        }
        settings.readImages = false;
        if(closeAfterSend) {
            close();
        }
    });
    spcom->setConnectedCallback([this]() {
        if(!settings.nonGraphical) {
            emit enableButtons(true, true);
        }
    });

    if(!settings.nonGraphical) {
        ui->displayWidget->setSpCom(spcom);
    }
    spcom->connect();
}

void MainWindow::initGui() {
    initToolBar();
    initStatusBar();

    // Menu items
    QObject::connect(ui->actionChoose_capture_directory, &QAction::triggered,
        this, [this]{chooseWriteDirectory(true);});

    // Zooming
    zoomLabel = new QLabel("100%", this);
    zoomLabel->setFont(QFontDatabase::systemFont(QFontDatabase::FixedFont));
    ui->toolBar->insertWidget(ui->actionZoom_out, zoomLabel);
    QObject::connect(ui->actionZoom_in, &QAction::triggered,
        this, [this]{zoom(1);});
    QObject::connect(ui->actionZoom_out, &QAction::triggered,
        this, [this]{zoom(-1);});

    // Button enable / disable
    QObject::connect(this, &MainWindow::enableButtons, this, [this](bool spcomReady, bool connected){
        ui->actionPreferences->setEnabled(spcomReady);

        ui->actionSend_images->setEnabled(connected);
        ui->actionCapture_single_frame->setEnabled(connected);
        ui->actionCapture_sequence->setEnabled(connected);
        ui->actionNo_color_coding->setEnabled(connected);
        ui->actionRed_blue_color_coding->setEnabled(connected);
        ui->actionRainbow_color_coding->setEnabled(connected);
        ui->actionCapture_16_bit->setEnabled(connected);
        ui->actionWrite_3D_point_clouds->setEnabled(connected);
        ui->actionChoose_capture_directory->setEnabled(connected);
    });

    // Update GUI status
    colorCodingChange(settings.colorScheme);
    ui->actionCapture_16_bit->setChecked(settings.writeRaw16Bit);
    ui->actionWrite_3D_point_clouds->setChecked(settings.writePointCloud);
    ui->actionSend_images->setChecked(settings.readImages);
    ui->displayWidget->setImagesToDisplay(settings.displayImage < 0 || settings.displayImage == 0,
    settings.displayImage < 0 || settings.displayImage == 1);
    zoom(0);
}

void MainWindow::makeFullscreen() {
    ui->statusBar->hide();
    ui->menuBar->hide();
    delete ui->toolBar;
    ui->toolBar = nullptr;
    setWindowFlags(Qt::Window | Qt::FramelessWindowHint);
    showFullScreen();
}

void MainWindow::initStatusBar() {
    // Setup status bar
    sizeLabel = new QLabel(this);
    fpsLabel = new QLabel(this);
    ui->statusBar->addPermanentWidget(sizeLabel);
    ui->statusBar->addPermanentWidget(fpsLabel);
    sizeLabel->setText("Waiting for data...");
}

void MainWindow::initToolBar() {
    QObject::connect(ui->actionNo_color_coding, &QAction::triggered,
        this, [this]{colorCodingChange(Settings::COLOR_SCHEME_NONE);});
    QObject::connect(ui->actionRed_blue_color_coding, &QAction::triggered,
        this, [this]{colorCodingChange(Settings::COLOR_SCHEME_RED_BLUE);});
    QObject::connect(ui->actionRainbow_color_coding, &QAction::triggered,
        this, [this]{colorCodingChange(Settings::COLOR_SCHEME_RAINBOW);});

    QObject::connect(ui->actionCapture_16_bit, &QAction::triggered, this, [this]{
        if(spcom != nullptr) {
            settings.writeRaw16Bit = ui->actionCapture_16_bit->isChecked();
            if(spcom != nullptr) {
                spcom->updateSettings(settings);
            }
        }
    });
    QObject::connect(ui->actionWrite_3D_point_clouds, &QAction::triggered,
        this, [this]{
            settings.writePointCloud = false;
            if(ui->actionWrite_3D_point_clouds->isChecked()) {
                settings.pointCloudMaxDist = QInputDialog::getDouble(this, "Point cloud maximum distance",
                    "Maximum distance (meters):", settings.pointCloudMaxDist, 0.01, 1000, 1, &settings.writePointCloud);
            }
            ui->actionWrite_3D_point_clouds->setChecked(settings.writePointCloud);

            if(spcom != nullptr) {
                spcom->updateSettings(settings);
            }
        });

    QObject::connect(ui->actionCapture_single_frame, &QAction::triggered,
        this, [this]{
            if(chooseWriteDirectory(false) && spcom != nullptr) {
                spcom->captureSingleFrame();
            }
        });

    QObject::connect(ui->actionCapture_sequence, &QAction::triggered,
        this, [this]{
            if(chooseWriteDirectory(false) && spcom != nullptr) {
                spcom->setCaptureSequence(ui->actionCapture_sequence->isChecked());
            } else {
                ui->actionCapture_sequence->setChecked(false);
            }
        });

    QObject::connect(ui->actionQuit, &QAction::triggered,
        this, [this]{close();});

    QObject::connect(ui->actionPreferences, &QAction::triggered,
        this, &MainWindow::preferencesDialog);
    QObject::connect(ui->actionSend_images, &QAction::triggered,
        this, &MainWindow::transmitInputFolder);
}

void MainWindow::displayFrame(const cv::Mat_<cv::Vec3b>& leftFrame, const cv::Mat_<cv::Vec3b>& rightFrame, bool resize) {
    fpsCounters.back().first++;

    if(!settings.nonGraphical) {
        ui->displayWidget->setDisplayFrame(leftFrame, rightFrame, resize);
        ui->displayWidget->update();

        if(resize) {
            emit updateStatusBar(leftFrame.cols, leftFrame.rows);
        }
    }
}

bool MainWindow::parseOptions(int argc, char** argv) {
    bool failed = false;
    writeImages = false;
    fullscreen = false;
    settings.readImages = false;
    settings.readDir = appSettings.value("read_dir", "").toString().toUtf8().constData();
    settings.writeDir = appSettings.value("write_dir", "").toString().toUtf8().constData();
    settings.maxFrameRate = appSettings.value("max_frame_rate", 30).toDouble();
    settings.nonGraphical = false;
    settings.remoteHost = appSettings.value("remote_host", "192.168.10.10").toString().toUtf8().constData();
    settings.localHost = appSettings.value("local_host", "0.0.0.0").toString().toUtf8().constData();
    settings.tcp = appSettings.value("tcp", false).toBool();;
    settings.localPort = appSettings.value("local_port", 7681).toInt();
    settings.remotePort = appSettings.value("remote_port", 7681).toInt();
    settings.writeRaw16Bit = appSettings.value("write_raw", false).toBool();
    settings.displayImage = -1;
    settings.disableReception = false;
    settings.printTimestamps = false;
    settings.writePointCloud = appSettings.value("write_point_cloud", false).toBool();
    settings.pointCloudMaxDist = appSettings.value("point_cloud_max_dist", 10).toDouble();
    settings.colorScheme = static_cast<Settings::ColorScheme>(appSettings.value("color_scheme", 0).toInt());
    settings.zoomPercent = appSettings.value("zoom", 100).toInt();

    int c;
    while ((c = getopt(argc, argv, "-:c:Cf:w:s:nt:r:i:h:p:H:P:dT3:z:F")) != -1) {
        switch(c) {
            case 'c': settings.colorScheme = static_cast<Settings::ColorScheme>(atoi(optarg)); break;
            case 'f': settings.maxFrameRate = atof(optarg); break;
            case 'w':
                settings.writeDir = optarg;
                writeImages = true;
                break;
            case 's':
                settings.readDir = optarg;
                settings.readImages = true;
                closeAfterSend = true;
                break;
            case 'n': settings.nonGraphical = true; break;
            case 'p': settings.remotePort = atoi(optarg); break;
            case 'h': settings.remoteHost = optarg; break;
            case 'P': settings.localPort = atoi(optarg); break;
            case 'H': settings.localHost = optarg; break;
            case 't': settings.tcp = (string(optarg) == "on"); break;
            case 'r': settings.writeRaw16Bit = (string(optarg) == "on"); break;
            case 'i': settings.displayImage = atoi(optarg); break;
            case 'd': settings.disableReception = true; break;
            case 'T': settings.printTimestamps = true; break;
            case '3': {
                double dist = atof(optarg);
                if(dist > 0) {
                    settings.pointCloudMaxDist = dist;
                    settings.writePointCloud = true;
                } else {
                    settings.writePointCloud = false;
                }
                break;
            }
            case 'z': settings.zoomPercent = atoi(optarg); break;
            case 'F': fullscreen = true; break;
            case '-':  // --help
            default:
                settings.nonGraphical = true;
                failed = true;
                cout << endl;
        }
    }

    if(argc - optind != 0 || failed) {
        cerr << "Usage: " << endl
             << argv[0] << " [OPTIONS]" << endl << endl
             << "Options: " << endl
             << "--help     Show this information" << endl
             << "-c VAL     Select color coding scheme (0 = no color, 1 = red / blue," << endl
             << "           2 = rainbow)" << endl
             << "-s DIR     Send the images from the given directory" << endl
             << "-f FPS     Limit send frame rate to FPS" << endl
             << "-d         Disable image reception" << endl
             << "-n         Non-graphical" << endl
             << "-w DIR     Immediately write all images to DIR" << endl
             << "-3 VAL     Write a 3D point cloud with distances up to VAL (0 = off)" << endl
             << "-h HOST    Use the given remote hostname for communication" << endl
             << "-p PORT    Use the given remote port number for communication" << endl
             << "-H HOST    Use the given local hostname for communication" << endl
             << "-P PORT    Use the given local port number for communication" << endl
             << "-t on/off  Activate / deactivate TCP transfers" << endl
             << "-r on/off  Activate / deactivate output of raw 16-bit disparity maps" << endl
             << "-i 0/1     Only displays the left (0) or right (1) image" << endl
             << "-T         Print frame timestamps" << endl
             << "-z VAL     Set zoom factor to VAL percent" << endl
             << "-F         Run in fullscreen mode" << endl;
        return false;
    }

    return true;
}

void MainWindow::colorCodingChange(Settings::ColorScheme newScheme) {
    // First uncheck
    switch(newScheme) {
        case Settings::COLOR_SCHEME_NONE:
            ui->actionNo_color_coding->setChecked(true);
            ui->actionRed_blue_color_coding->setChecked(false);
            ui->actionRainbow_color_coding->setChecked(false);
            break;
        case Settings::COLOR_SCHEME_RAINBOW:
            ui->actionRainbow_color_coding->setChecked(true);
            ui->actionNo_color_coding->setChecked(false);
            ui->actionRed_blue_color_coding->setChecked(false);
            break;
        case Settings::COLOR_SCHEME_RED_BLUE:
            ui->actionRed_blue_color_coding->setChecked(true);
            ui->actionNo_color_coding->setChecked(false);
            ui->actionRainbow_color_coding->setChecked(false);
            break;
    }

    settings.colorScheme = newScheme;

    if(spcom != nullptr) {
        spcom->updateSettings(settings);
    }
}

bool MainWindow::chooseWriteDirectory(bool forceChoice) {
    if(!forceChoice && writeDirSelected) {
        return true;
    }

    QString newDir = QFileDialog::getExistingDirectory(this,
        "Choose capture directory", settings.writeDir.c_str(),
         QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);

    if(newDir != "") {
        settings.writeDir = newDir.toUtf8().constData();
        writeDirSelected = true;
    }

    if(spcom != nullptr) {
        spcom->updateSettings(settings);
        spcom->resetCaptureIndex();
    }

    return newDir != "";
}

void MainWindow::preferencesDialog() {
    PreferencesDialog diag(this, settings);
    if(diag.exec() == QDialog::Accepted) {
        settings = diag.getSettings();
        reinitSpcom();
    }
}

void MainWindow::displayException(const std::string& msg) {
    cerr << "Exception occurred: " << msg << endl;
    QMessageBox msgBox(QMessageBox::Critical, "SpCom Exception!", msg.c_str());
    msgBox.exec();
}

void MainWindow::displayFrameRate() {
    // Compute frame rate
    int framesCount = 0;
    for(auto counter: fpsCounters) {
        framesCount += counter.first;
    }

    int elapsedTime = duration_cast<microseconds>(
        steady_clock::now() - fpsCounters[0].second).count();
    double fps = framesCount / (elapsedTime*1.0e-6);

    // Drop one counter
    fpsCounters.pop_front();
    fpsCounters.push_back(std::make_pair(0, steady_clock::now()));

    // Update label
    char fpsStr[6];
    snprintf(fpsStr, sizeof(fpsStr), "%.2lf", fps);

    if(fpsLabel != nullptr) {
        emit updateFpsLabel(QString(fpsStr) +  " fps");
    }

    // Print console status messages at a lower update rate
    static int printCounter = 0;
    if((printCounter++)>5 && spcom != nullptr) {
        cout << "Fps: " << fpsStr << "; output queue: " << spcom->getWriteQueueSize() << endl;
        printCounter = 0;
    }
}

void MainWindow::transmitInputFolder() {
    if(!settings.readImages) {
        QString newReadDir= QFileDialog::getExistingDirectory(this,
            "Choose input directory", settings.readDir.c_str(),
            QFileDialog::ShowDirsOnly | QFileDialog::DontResolveSymlinks);
        if(newReadDir != "") {
            settings.readDir = newReadDir.toUtf8().constData();
            settings.readImages = true;
        }
    } else {
        settings.readImages = false;
    }

    if(settings.readImages) {
        bool ok = false;
        settings.maxFrameRate = QInputDialog::getDouble(this, "Choose frame rate",
            "Send frame rate:", settings.maxFrameRate, 0.1, 100, 1, &ok);
        if(!ok) {
            settings.readImages = false;
        }
    }

    ui->actionSend_images->setChecked(settings.readImages);
    if(spcom != nullptr) {
        spcom->updateSettings(settings);
    }
}

void MainWindow::zoom(int direction) {
    if(direction != 0) {
        settings.zoomPercent =
            (settings.zoomPercent / 25)*25 + 25*direction;
    }

    if(settings.zoomPercent > 200) {
        settings.zoomPercent = 200;
    } else if(settings.zoomPercent < 25) {
        settings.zoomPercent = 25;
    }

    char labelText[5];
    snprintf(labelText, sizeof(labelText), "%3d%%", settings.zoomPercent);

    zoomLabel->setText(labelText);
    ui->displayWidget->setZoom(settings.zoomPercent);

    ui->actionZoom_in->setEnabled(settings.zoomPercent != 200);
    ui->actionZoom_out->setEnabled(settings.zoomPercent != 25);
}

int main(int argc, char** argv) {
    bool graphical = true;

    try {
        QApplication app(argc, argv);
        MainWindow win(nullptr, app);

        if(!win.init(argc, argv)) {
            return 1;
        }

        graphical = win.isGraphical();

#ifdef _WIN32
        if(graphical) {
            FreeConsole();
        }
#endif

        return app.exec();
    } catch(const std::exception& ex) {
        if(graphical) {
            QApplication app(argc, argv);
            MainWindow::displayException(ex.what());
        }
        return -1;
    }
}
