/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "filequeue.h"
#include <algorithm>
#include <cstring>
#include <dirent.h>

using namespace cv;
using namespace std;

template <class T, typename U>
BaseFileQueue<T, U>::BaseFileQueue(const char* directory, int maxQueueSize,
    bool multiThreaded)
    :ImageQueue<T>(false, maxQueueSize, multiThreaded), threadInitialized(false),
    fileIndex(0) {

    // Crate file list
    DIR *dir;
    if((dir = opendir(directory)) != nullptr) {
        dirent *ent;
         while ((ent = readdir (dir)) != nullptr) {
            imageFiles.push_back(string(directory) + "/" + ent->d_name);
        }
        closedir (dir);
    }

    // Sort files
    sort(imageFiles.begin(), imageFiles.end());
}

template <class T, typename U>
BaseFileQueue<T, U>::~BaseFileQueue() {
}

template <class T, typename U>
const char* BaseFileQueue<T, U>::getNextFileName() {
    if(fileIndex >= imageFiles.size())
        return nullptr;
    else return imageFiles[fileIndex++].c_str();
}

template <class T, typename U>
Mat_<U> BaseFileQueue<T, U>::readNextFile() {
    const char* fileName;
    while((fileName = getNextFileName()) != nullptr) {
        Mat frame;
        try {
            frame = imread(fileName, IMREAD_ANYDEPTH);
        } catch (...) {
            // Leave frame uninitialized in case of errors
        }
        if(frame.data == nullptr) {
            cerr << "Skipping unreadable file: " << fileName << endl;
        } else if(frame.depth() == CV_16U && typeid(U) == typeid(unsigned char)) {
            // Convert 16 to 8 bit
            cv::Mat_<unsigned char> frame8Bit;
            frame.convertTo(frame8Bit, CV_8U, 1.0/256.0);
            return frame8Bit;
        } else if((frame.depth() != CV_8U && typeid(U) == typeid(unsigned char)) ||
                  (frame.depth() != CV_16U && typeid(U) == typeid(unsigned short))) {
            cerr << "Skipping file with invalid bit depth: " << fileName << endl;
        } else {
            return frame;
        }

    }
    return Mat_<U>();
}

template <typename T>
MonoFileQueue<T>::~MonoFileQueue() {
    BaseFileQueue<typename MonoFrame<T>::Type, T >::stopThreads();
}

template <typename T>
void MonoFileQueue<T>::queueFrame() {
    typename MonoFrame<T>::Ptr monoFrame(new Mat_<T>(this->readNextFile()));

    if(monoFrame->data != nullptr)
        BaseFileQueue<typename MonoFrame<T>::Type, T > ::push(monoFrame);
    else this->closeQueue();
}

template <typename T>
StereoFileQueue<T>::~StereoFileQueue() {
    BaseFileQueue<typename StereoFrame<T>::Type, T >::stopThreads();
}

template <typename T>
void StereoFileQueue<T>::queueFrame() {
    typename StereoFrame<T>::Ptr stereoFrame(new typename StereoFrame<T>::Type);
    stereoFrame->first = this->readNextFile();
    stereoFrame->second = this->readNextFile();

    if(stereoFrame->first.data != nullptr && stereoFrame->second.data != nullptr) {
        BaseFileQueue<typename StereoFrame<T>::Type, T >::push(stereoFrame);
    } else {
        this->closeQueue();
    }
}

// Explicit template instantiation
template class MonoFileQueue<unsigned char>;
template class MonoFileQueue<unsigned short>;
template class StereoFileQueue<unsigned char>;
template class StereoFileQueue<unsigned short>;

