/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef MAINWINDOW_H
#define MAINWINDOW_H

#include <opencv2/opencv.hpp>
#include <memory>
#include <chrono>
#include <deque>

#include <QMainWindow>
#include <QSettings>
#include <QTimer>
#include <QLabel>

#include "settings.h"

// Predeclarations
namespace Ui {
class MainWindow;
}
class QLabel;
class QComboBox;
class NVCom;

/*
 * The main application window, displaying the received frames.
 */
class MainWindow: public QMainWindow {
    Q_OBJECT

public:
    MainWindow(QWidget *parent, QApplication& app);
    ~MainWindow();

    bool init(int argc, char** argv);
    bool isGraphical() const {return !settings.nonGraphical;}
    static void displayException(const std::string& msg);

signals:
    void asyncDisplayException(const QString& msg);
    void updateStatusBar(int dropped, int imgWidth, int imgHeight, int bits0, int bits1, int bits2);
    void updateFpsLabel(const QString& text);
    void enableButtons(bool nvcomReady, bool connected);
    void repaintDisplayWidget();

private:
    static constexpr int UI_VERSION = 1;

    std::unique_ptr<Ui::MainWindow> ui;
    Settings settings;
    std::shared_ptr<NVCom> nvcom;
    bool writeImages;
    bool writeDirSelected;
    bool fullscreen;

    QLabel* fpsLabel;
    QLabel* sizeLabel;
    QLabel* droppedLabel;
    QLabel* zoomLabel;
    QSettings appSettings;
    QTimer fpsTimer;
    QComboBox* colorCombo;

    bool closeAfterSend;
    bool resizeWindow;
    int lastDropped;

    std::deque<std::pair<unsigned int, std::chrono::steady_clock::time_point> > fpsCounters;

    bool parseOptions(int argc, char** argv);
    void displayFrame(int origW, int origH, const std::vector<cv::Mat_<cv::Vec3b>>& frames, int numActiveImages, bool resize);
    void colorCodingChange(Settings::ColorScheme newScheme);
    bool chooseWriteDirectory(bool forceChoice);

    void initGui();
    void initToolBar();
    void initStatusBar();

    void showConnectionDialog();
    void displayFrameRate();
    void reinitNVCom();
    void writeApplicationSettings();
    void transmitInputFolder();
    void zoom(int direction);
    void makeFullscreen();
    void openPointCloudDialog();
};

#endif
