/*
 * Software License Agreement (BSD License)
 *
 *  Copyright (c) 2020, Locus Robotics
 *  All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above
 *     copyright notice, this list of conditions and the following
 *     disclaimer in the documentation and/or other materials provided
 *     with the distribution.
 *   * Neither the name of the copyright holder nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 *  FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 *  COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 *  BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 *  LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 *  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
 *  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 *  POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef ROBOT_NAV_RVIZ_PLUGINS_VALIDATE_FLOATS_H
#define ROBOT_NAV_RVIZ_PLUGINS_VALIDATE_FLOATS_H

#include <geometry_msgs/Pose2D.h>
#include <nav_grid/nav_grid_info.h>
#include <nav_2d_msgs/Path2D.h>
#include <nav_2d_msgs/Point2D.h>
#include <nav_2d_msgs/Polygon2D.h>
#include <nav_2d_msgs/ComplexPolygon2D.h>
#include <rviz/validate_floats.h>
#include <vector>

namespace robot_nav_rviz_plugins
{
inline bool validateFloats(const nav_grid::NavGridInfo& info)
{
  return rviz::validateFloats(info.resolution)
      && rviz::validateFloats(info.origin_x)
      && rviz::validateFloats(info.origin_y);
}

inline bool validateFloats(const geometry_msgs::Pose2D& pose)
{
  return rviz::validateFloats(pose.x)
      && rviz::validateFloats(pose.y)
      && rviz::validateFloats(pose.theta);
}

inline bool validateFloats(const nav_2d_msgs::Point2D& point)
{
  return rviz::validateFloats(point.x) && rviz::validateFloats(point.y);
}

template <typename T>
inline bool validateFloats(const std::vector<T>& vec)
{
  for (const auto& element : vec)
  {
    if (!validateFloats(element)) return false;
  }
  return true;
}

inline bool validateFloats(const nav_2d_msgs::Path2D& msg)
{
  return validateFloats(msg.poses);
}

inline bool validateFloats(const nav_2d_msgs::Polygon2D& msg)
{
  return validateFloats(msg.points);
}

inline bool validateFloats(const nav_2d_msgs::ComplexPolygon2D& msg)
{
  if (!validateFloats(msg.outer)) return false;
  for (const auto& inner : msg.inner)
  {
    if (!validateFloats(inner)) return false;
  }
  return true;
}

}  // namespace robot_nav_rviz_plugins

#endif  // ROBOT_NAV_RVIZ_PLUGINS_VALIDATE_FLOATS_H
