/**
 * @authors     Mike Karamousadakis
 * @affiliation BOTA SYS A.G.
 * @brief       Tests Configuration
 */

#include <ros/ros.h>
#include <ros/package.h>
#include <gtest/gtest.h>

#include <rokubimini/configuration/Configuration.hpp>

namespace rokubimini
{
namespace configuration
{
class ConfigurationTest : public ::testing::Test
{
protected:
  Configuration* configuration;
  const std::string fullYaml = ros::package::getPath("rokubimini") + "/test/config_full.yaml";
  const std::string emptyYaml = ros::package::getPath("rokubimini") + "/test/config_empty.yaml";
  ConfigurationTest()
  {
    configuration = new Configuration();
  }

  ~ConfigurationTest() override
  {
    delete configuration;
    // You can do clean-up work that doesn't throw exceptions here.
  }

  // If the constructor and destructor are not enough for setting up
  // and cleaning up each test, you can define the following methods:

  void SetUp() override
  {
    // Code here will be called immediately after the constructor (right
    // before each test).
  }

  void TearDown() override
  {
    // Code here will be called immediately after each test (right
    // before the destructor).
  }
};

TEST_F(ConfigurationTest, fromFileLoadsFull)
{
  configuration->fromFile(fullYaml);
  EXPECT_EQ(configuration->getSetReadingToNanOnDisconnect(), false);
  EXPECT_EQ(configuration->getForceTorqueFilter().getSincFilterSize(), 512);
  EXPECT_EQ(configuration->getForceTorqueFilter().getChopEnable(), false);
  EXPECT_EQ(configuration->getForceTorqueFilter().getSkipEnable(), true);
  EXPECT_EQ(configuration->getForceTorqueFilter().getFastEnable(), false);

  EXPECT_DOUBLE_EQ(configuration->getForceTorqueOffset()(0, 0), 0);
  EXPECT_DOUBLE_EQ(configuration->getForceTorqueOffset()(1, 0), 0);
  EXPECT_DOUBLE_EQ(configuration->getForceTorqueOffset()(2, 0), 0);
  EXPECT_DOUBLE_EQ(configuration->getForceTorqueOffset()(3, 0), 0);
  EXPECT_DOUBLE_EQ(configuration->getForceTorqueOffset()(4, 0), 0);
  EXPECT_DOUBLE_EQ(configuration->getForceTorqueOffset()(5, 0), 0);

  EXPECT_EQ(configuration->getUseCustomCalibration(), false);
  EXPECT_EQ(configuration->getSensorConfiguration().getCalibrationMatrixActive(), true);
  EXPECT_EQ(configuration->getSensorConfiguration().getTemperatureCompensationActive(), false);
  EXPECT_EQ(configuration->getSensorConfiguration().getImuActive(), 1);
  EXPECT_EQ(configuration->getSensorConfiguration().getCoordinateSystemConfigurationActive(), false);
  EXPECT_EQ(configuration->getSensorConfiguration().getInertiaCompensationActive(), 0);
  EXPECT_EQ(configuration->getSensorConfiguration().getOrientationEstimationActive(), 0);

  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(0, 0), -0.0120389);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(0, 1), 0.0206676);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(0, 2), 0.0011164);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(0, 3), 0.0010681);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(0, 4), 0.0006295);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(0, 5), 0.0002029);

  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(1, 0), -0.0109678);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(1, 1), -0.0207696);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(1, 2), -0.0004745);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(1, 3), -0.0010578);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(1, 4), 0.0005841);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(1, 5), 0.0001810);

  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(2, 0), 0.0217535);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(2, 1), -0.0005980);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(2, 2), -0.0002701);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(2, 3), -0.0000395);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(2, 4), -0.0011409);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(2, 5), 0.0001944);

  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(3, 0), 0.0155252);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(3, 1), 0.0100866);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(3, 2), -0.0148265);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(3, 3), 0.0003659);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(3, 4), -0.0005838);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(3, 5), 0.0000055);

  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(4, 0), -0.0169726);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(4, 1), 0.0114251);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(4, 2), -0.0146247);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(4, 3), 0.0004339);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(4, 4), 0.0006094);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(4, 5), 0.0000011);

  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(5, 0), -0.0001013);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(5, 1), -0.0197085);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(5, 2), -0.0146952);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(5, 3), -0.0007251);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(5, 4), -0.0000145);
  EXPECT_DOUBLE_EQ(configuration->getSensorCalibration().getCalibrationMatrix()(5, 5), -0.0000077);

  EXPECT_EQ(configuration->getImuAccelerationFilter(), (unsigned int)1);
  EXPECT_EQ(configuration->getImuAngularRateFilter(), (unsigned int)3);
  EXPECT_EQ(configuration->getImuAccelerationRange(), 3);
  EXPECT_EQ(configuration->getImuAngularRateRange(), 3);
}

TEST_F(ConfigurationTest, fromFileLoadsEmpty)
{
  configuration->fromFile(emptyYaml);

  // EXPECT_EQ(hasMaxCommandAge_, false);
  // EXPECT_EQ(hasAutoStageLastCommand_, false);
  // EXPECT_EQ(hasSetReadingToNanOnDisconnect_, false);
  // EXPECT_EQ(hasForceTorqueFilter_, false);
  // EXPECT_EQ(hasSensorConfiguration_, false);
  // EXPECT_EQ(hasUseCustomCalibration_, false);
  // EXPECT_EQ(hasSensorCalibration_, false);
  // EXPECT_EQ(hasImuAccelerationFilter_, false);
  // EXPECT_EQ(hasImuAngularRateFilter_, false);
  // EXPECT_EQ(hasImuAccelerationRange_, false);
  // EXPECT_EQ(hasImuAngularRateRange_, false);
  // EXPECT_EQ(hasForceTorqueOffset_, false);
}

TEST_F(ConfigurationTest, assignmentOperatorWorksCorrectly)
{
  configuration->fromFile(fullYaml);

  auto new_configuration = Configuration();

  new_configuration = *configuration;

  EXPECT_EQ(new_configuration.getSetReadingToNanOnDisconnect(), false);
  EXPECT_EQ(new_configuration.getForceTorqueFilter().getSincFilterSize(), 512);
  EXPECT_EQ(new_configuration.getForceTorqueFilter().getChopEnable(), false);
  EXPECT_EQ(new_configuration.getForceTorqueFilter().getSkipEnable(), true);
  EXPECT_EQ(new_configuration.getForceTorqueFilter().getFastEnable(), false);

  EXPECT_DOUBLE_EQ(new_configuration.getForceTorqueOffset()(0, 0), 0);
  EXPECT_DOUBLE_EQ(new_configuration.getForceTorqueOffset()(1, 0), 0);
  EXPECT_DOUBLE_EQ(new_configuration.getForceTorqueOffset()(2, 0), 0);
  EXPECT_DOUBLE_EQ(new_configuration.getForceTorqueOffset()(3, 0), 0);
  EXPECT_DOUBLE_EQ(new_configuration.getForceTorqueOffset()(4, 0), 0);
  EXPECT_DOUBLE_EQ(new_configuration.getForceTorqueOffset()(5, 0), 0);

  EXPECT_EQ(new_configuration.getUseCustomCalibration(), false);
  EXPECT_EQ(new_configuration.getSensorConfiguration().getCalibrationMatrixActive(), true);
  EXPECT_EQ(new_configuration.getSensorConfiguration().getTemperatureCompensationActive(), false);
  EXPECT_EQ(new_configuration.getSensorConfiguration().getImuActive(), 1);
  EXPECT_EQ(new_configuration.getSensorConfiguration().getCoordinateSystemConfigurationActive(), false);
  EXPECT_EQ(new_configuration.getSensorConfiguration().getInertiaCompensationActive(), 0);
  EXPECT_EQ(new_configuration.getSensorConfiguration().getOrientationEstimationActive(), 0);

  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(0, 0), -0.0120389);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(0, 1), 0.0206676);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(0, 2), 0.0011164);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(0, 3), 0.0010681);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(0, 4), 0.0006295);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(0, 5), 0.0002029);

  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(1, 0), -0.0109678);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(1, 1), -0.0207696);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(1, 2), -0.0004745);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(1, 3), -0.0010578);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(1, 4), 0.0005841);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(1, 5), 0.0001810);

  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(2, 0), 0.0217535);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(2, 1), -0.0005980);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(2, 2), -0.0002701);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(2, 3), -0.0000395);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(2, 4), -0.0011409);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(2, 5), 0.0001944);

  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(3, 0), 0.0155252);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(3, 1), 0.0100866);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(3, 2), -0.0148265);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(3, 3), 0.0003659);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(3, 4), -0.0005838);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(3, 5), 0.0000055);

  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(4, 0), -0.0169726);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(4, 1), 0.0114251);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(4, 2), -0.0146247);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(4, 3), 0.0004339);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(4, 4), 0.0006094);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(4, 5), 0.0000011);

  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(5, 0), -0.0001013);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(5, 1), -0.0197085);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(5, 2), -0.0146952);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(5, 3), -0.0007251);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(5, 4), -0.0000145);
  EXPECT_DOUBLE_EQ(new_configuration.getSensorCalibration().getCalibrationMatrix()(5, 5), -0.0000077);

  EXPECT_EQ(new_configuration.getImuAccelerationFilter(), (unsigned int)1);
  EXPECT_EQ(new_configuration.getImuAngularRateFilter(), (unsigned int)3);
  EXPECT_EQ(new_configuration.getImuAccelerationRange(), 3);
  EXPECT_EQ(new_configuration.getImuAngularRateRange(), 3);
}

TEST_F(ConfigurationTest, forceTorqueFilterWorksCorrectly)
{
  configuration->setForceTorqueFilter(ForceTorqueFilter(64, true, true, false));
  ForceTorqueFilter filter = configuration->getForceTorqueFilter();
  EXPECT_DOUBLE_EQ(filter.getSincFilterSize(), 64);
  EXPECT_EQ(filter.getChopEnable(), true);
  EXPECT_EQ(filter.getSkipEnable(), true);
  EXPECT_EQ(filter.getFastEnable(), false);
}

TEST_F(ConfigurationTest, forceTorqueOffsetWorksCorrectly)
{
  configuration->setForceTorqueOffset(Eigen::Matrix<double, 6, 1>::Ones());
  Eigen::Matrix<double, 6, 1> matrix = configuration->getForceTorqueOffset();
  EXPECT_DOUBLE_EQ(matrix(0, 0), 1);
  EXPECT_DOUBLE_EQ(matrix(1, 0), 1);
  EXPECT_DOUBLE_EQ(matrix(2, 0), 1);
  EXPECT_DOUBLE_EQ(matrix(3, 0), 1);
  EXPECT_DOUBLE_EQ(matrix(4, 0), 1);
  EXPECT_DOUBLE_EQ(matrix(5, 0), 1);
}

TEST_F(ConfigurationTest, useCustomCalibrationWorksCorrectly)
{
  configuration->setUseCustomCalibration(true);
  auto use_custom_calibration = configuration->getUseCustomCalibration();
  EXPECT_EQ(use_custom_calibration, true);
}

TEST_F(ConfigurationTest, setReadingToNanOnDisconnectWorksCorrectly)
{
  configuration->setSetReadingToNanOnDisconnect(true);
  auto set_reading_to_nan_on_disconnect = configuration->getSetReadingToNanOnDisconnect();
  EXPECT_EQ(set_reading_to_nan_on_disconnect, true);
}

TEST_F(ConfigurationTest, sensorConfigurationWorksCorrectly)
{
  configuration->setSensorConfiguration(SensorConfiguration(true, true, false, false, false, true));
  auto sensor_configuration = configuration->getSensorConfiguration();
  EXPECT_EQ(sensor_configuration.getCalibrationMatrixActive(), true);
  EXPECT_EQ(sensor_configuration.getTemperatureCompensationActive(), true);
  EXPECT_EQ(sensor_configuration.getImuActive(), false);
  EXPECT_EQ(sensor_configuration.getCoordinateSystemConfigurationActive(), false);
  EXPECT_EQ(sensor_configuration.getInertiaCompensationActive(), false);
  EXPECT_EQ(sensor_configuration.getOrientationEstimationActive(), true);
}

TEST_F(ConfigurationTest, sensorCalibrationWorksCorrectly)
{
  Eigen::Matrix<double, 6, 6> calibration_matrix = Eigen::Matrix<double, 6, 6>::Identity();
  auto sensor_calibration = rokubimini::calibration::SensorCalibration();
  sensor_calibration.setCalibrationMatrix(calibration_matrix);
  configuration->setSensorCalibration(sensor_calibration);
  auto calibration_expected = configuration->getSensorCalibration();
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(0, 0), 1);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(0, 1), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(0, 2), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(0, 3), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(0, 4), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(0, 5), 0);

  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(1, 0), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(1, 1), 1);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(1, 2), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(1, 3), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(1, 4), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(1, 5), 0);

  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(2, 0), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(2, 1), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(2, 2), 1);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(2, 3), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(2, 4), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(2, 5), 0);

  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(3, 0), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(3, 1), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(3, 2), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(3, 3), 1);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(3, 4), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(3, 5), 0);

  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(4, 0), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(4, 1), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(4, 2), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(4, 3), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(4, 4), 1);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(4, 5), 0);

  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(5, 0), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(5, 1), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(5, 2), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(5, 3), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(5, 4), 0);
  EXPECT_DOUBLE_EQ(calibration_expected.getCalibrationMatrix()(5, 5), 1);
}

TEST_F(ConfigurationTest, imuAccelerationFilterWorksCorrectly)
{
  configuration->setImuAccelerationFilter((unsigned int)128);
  auto imu_acceleration_filter = configuration->getImuAccelerationFilter();
  EXPECT_EQ(imu_acceleration_filter, (unsigned int)128);
}

TEST_F(ConfigurationTest, imuAngularRateFilterWorksCorrectly)
{
  configuration->setImuAngularRateFilter((unsigned int)128);
  auto imu_angular_rate_filter = configuration->getImuAngularRateFilter();
  EXPECT_EQ(imu_angular_rate_filter, (unsigned int)128);
}

TEST_F(ConfigurationTest, imuAccelerationRangeWorksCorrectly)
{
  configuration->setImuAccelerationRange((uint8_t)15);
  auto imu_acceleration_range = configuration->getImuAccelerationRange();
  EXPECT_EQ(imu_acceleration_range, 15);
}

TEST_F(ConfigurationTest, imuAngularRateRangeWorksCorrectly)
{
  configuration->setImuAngularRateRange((uint8_t)15);
  auto imu_angular_rate_range = configuration->getImuAngularRateRange();
  EXPECT_EQ(imu_angular_rate_range, 15);
}

}  // namespace configuration
}  // namespace rokubimini
