#include <ros/ros.h>

#include <geometry_msgs/Pose.h>
#include <geometry_msgs/PoseArray.h>
#include <visualization_msgs/Marker.h>
#include <visualization_msgs/MarkerArray.h>

#include <std_srvs/SetBool.h>
#include <geometry_msgs/Twist.h>

#include <std_srvs/Trigger.h>
#include <std_msgs/Float64.h>

#include <move_base_msgs/MoveBaseAction.h>
#include <actionlib/server/simple_action_server.h>

typedef actionlib::SimpleActionServer<move_base_msgs::MoveBaseAction> MoveBaseActionServer;

namespace rsm {

/**
 * @class AdditionsServiceProvider
 * @brief Class that establishes communication between the different states and the RSM's
 * 		  periphery including the GUI for the states and classes defined in rsm_additions package. It offers
 * 		  services and publishes topics based on the variables that need to be saved during state transitions.
 */
class AdditionsServiceProvider {

public:
	AdditionsServiceProvider();
	~AdditionsServiceProvider();
	void publishTopics();

private:
	ros::NodeHandle _nh;

	ros::ServiceServer _set_navigation_to_reverse_service;
	ros::Subscriber _reverse_mode_cmd_vel_subscriber;
	ros::Publisher _reverse_mode_cmd_vel_publisher;

	ros::Subscriber frontiers_marker_array_subscriber;
	ros::Publisher exploration_goals_publisher;

	ros::ServiceServer _reset_kinect_position_serivce;
	ros::Publisher _kinetic_joint_controller;

	/**
	 * SimpleActionServer for making Explore Lite run and lead it to believe it talks to Move Base
	 */
	MoveBaseActionServer* as;
	/**
	 *Topic name for the autonomy cmd vel topic to be recorded
	 */
	std::string _autonomy_cmd_vel_topic;

	/**
	 * List of all extracted frontier centers
	 */
	geometry_msgs::PoseArray _exploration_goals;
	/**
	 * Is the Navigation stack used as Plugin for navigation
	 */
	bool _navigation_plugin_used;

	/**
	 * Callback for receiving autonomy cmd vel messages and save the ones not equals zero in the cirular buffer
	 * @param msg Cmd vel autonomy message
	 */
	void cmdVelCallback(const geometry_msgs::Twist::ConstPtr& msg);

	bool setNavigationToReverse(std_srvs::SetBool::Request &req,
			std_srvs::SetBool::Response &res);
	/**
	 * Receives cmd vel and negate linear movement so robot goes backwards
	 * @param cmd_vel cmd vel generated for backward movement
	 */
	void reverseModeCmdVelCallback(
			const geometry_msgs::Twist::ConstPtr& cmd_vel);

	/**
	 * Callback for requests to the SimpleActionServer that just responds with "Goal reached"
	 * @param frontier_goal
	 */
	void navigationGoalCallback(
			const move_base_msgs::MoveBaseGoalConstPtr& frontier_goal);
	/**
	 * Callback for frontier visualization generated by Explore Lite. Extracts the centers of these
	 * frontiers and saves them in a list
	 * @param frontiers Marker list containing frontiers as points and nearest frontier points for
	 * each frontier as sphere
	 */
	void frontierCallback(
			const visualization_msgs::MarkerArray::ConstPtr& frontiers);
	/**
	 * Publish list of extracted frontier centers for further calculation
	 */
	void publishExplorationGoals();

	bool resetKinectPosition(std_srvs::Trigger::Request &req,
			std_srvs::Trigger::Response &res);
};

}
