#!/usr/bin/env python
import sys

import roslib

import gencpp
import genmsg

from  roslib import packages,msgs
import os

from cStringIO import StringIO

import argparse

NAME='create_boost_header'

def write_boost_includes(s, spec):
    """
    Writes the message-specific includes

    @param s: The stream to write to
    @type s: stream
    @param spec: The message spec to iterate over
    @type spec: roslib.msgs.MsgSpec
    @param serializer: The serializer type for which to include headers
    @type serializer: str
    """
    for field in spec.parsed_fields():
        if (not field.is_builtin):
            (pkg, name) = genmsg.names.package_resource_name(field.base_type)
            pkg = (pkg or spec.package) # convert '' to this package
            s.write('#include <%s/boost/%s.h>\n'%(pkg,  name))

    s.write('\n')


def write_boost_serialization(s, spec, cpp_name_prefix, file):
    """
    Writes the boost::serialize function for a message

    @param s: Stream to write to
    @type s: stream
    @param spec: The message spec
    @type spec: roslib.msgs.MsgSpec
    @param cpp_name_prefix: The C++ prefix to prepend to a message to refer to it (e.g. "std_msgs::")
    @type cpp_name_prefix: str
    """
    (cpp_msg_unqualified, cpp_msg_with_alloc, _) = gencpp.cpp_message_declarations(cpp_name_prefix, spec.short_name)

    s.write("/* Auto-generated by create_boost_header.py for file %s */\n"%(file))
    s.write('#ifndef %s_BOOST_SERIALIZATION_%s_H\n'%(spec.package.upper(), spec.short_name.upper()))
    s.write('#define %s_BOOST_SERIALIZATION_%s_H\n\n'%(spec.package.upper(), spec.short_name.upper()))
    s.write('#include <boost/serialization/serialization.hpp>\n')
    s.write('#include <boost/serialization/nvp.hpp>\n')
    s.write('#include <boost/serialization/vector.hpp>\n')
    s.write('#include <boost/serialization/string.hpp>\n')
    s.write('#include <%s/%s.h>\n'%(spec.package,spec.short_name))
    write_boost_includes(s, spec)
    s.write('namespace boost\n{\n')
    s.write('namespace serialization\n{\n\n')

    s.write('template<class Archive, class ContainerAllocator>\n')

    s.write('void serialize(Archive& a, %s & m, unsigned int)\n{\n'%(cpp_msg_with_alloc))

    for field in spec.parsed_fields():
        s.write('    a & make_nvp("%s",m.%s);\n'%(field.name,field.name))
    s.write('}\n\n')

    s.write('} // namespace serialization\n')
    s.write('} // namespace boost\n\n')
    s.write('#endif // %s_BOOST_SERIALIZATION_%s_H\n'%(spec.package.upper(), spec.short_name.upper()))



def generate_boost_serialization(package, msg_path, msg_type, boost_header_path):
    """
    Generate a boost::serialization header

    @param msg_path: The path to the .msg file
    @type msg_path: str
    """
    mc = genmsg.msg_loader.MsgContext()

    spec = genmsg.msg_loader.load_msg_from_file(mc, msg_path, msg_type)
    cpp_prefix = '%s::'%(package)

    s = StringIO()
    write_boost_serialization(s, spec, cpp_prefix, msg_path)

    (output_dir,filename) = os.path.split(boost_header_path)
    try:
        os.makedirs(output_dir)
    except OSError, e:
        pass

    f = open(boost_header_path, 'w')
    print >> f, s.getvalue()

    s.close()


def create_boost_headers(argv, stdout, stderr):
    parser = argparse.ArgumentParser(description='Generate boost serialization header for ROS message.')
    parser.add_argument('pkg',metavar='PKG',type=str, nargs=1,help='The package name.')
    parser.add_argument('msg_type',metavar='MSG_TYPE',type=str, nargs=1,help='The message type.')
    parser.add_argument('msg_file_path',metavar='MSG_FILE_PATH',type=str, nargs=1,help='The full path to a .msg file.')
    parser.add_argument('boost_file_path',metavar='BOOST_HEADER_PATH',type=str, nargs=1,help='The full path to the generated boost header file.')

    args = parser.parse_args()

    generate_boost_serialization(args.pkg[0], args.msg_file_path[0], args.msg_type[0], args.boost_file_path[0])

if __name__ == "__main__":
    try:
        create_boost_headers(sys.argv, sys.stdout, sys.stderr)
    except Exception, e:
        sys.stderr.write("Failed to generate boost headers: " + str(e))
        raise
        #sys.exit(1)
