#!/usr/bin/env python

import sys
import argparse
import os

import rospy
import rospkg
import rostest

from rosgraph_msgs.msg import Log

from smacha.util import Tester
from smacha.util import bcolors

# Necessary SRVs
from smacha_ros.srv import Generate

# YAML parser
from ruamel import yaml

# Used to get the path to the SMACHA scripts
import rospkg


# Script directory
rospack = rospkg.RosPack()
ROOT_DIR = rospack.get_path('smacha_ros')

CONF_FILE = 'test_generate.yml'
DEBUG_LEVEL = 1

class TestClass(Tester):
    """Tester class for general unit testing of various SMACHA tool
    functionalities.
    """

    _logmsg = ''
    _logerr = False

    _script_dirs = []
    _conf_dict = None

    def __init__(self, *args, **kwargs):
        # Set Tester member variables
        self.set_debug_level(DEBUG_LEVEL)

        # Store the base path
        self._base_path = os.path.dirname(os.path.abspath(__file__))

        # Call the parent constructor
        super(TestClass, self).__init__(*args, **kwargs)

        # Read the config file
        self._read_conf(CONF_FILE)

        # Create the service proxy
        rospy.wait_for_service('/smacha/generate')
        self._generator = rospy.ServiceProxy('/smacha/generate', Generate)

        rospy.Subscriber('/rosout', Log, self._std_out_cb)

    def _std_out_cb(self, data):
        
        # Check if there are any errors in the execution 
        # of the state machine that might not throw an
        # exception during the exec().
        msg = data.msg
        level = data.level
        if level >= 8:
            self._logmsg = msg
            self._logerr = True

    def _read_conf(self, conf_file):
        # Read the configuration file before parsing arguments,
        try:
            conf_file_loc = os.path.join(self._base_path, conf_file)
            f = open(conf_file_loc)
            self._conf_dict = yaml.load(f)
        except Exception as e:
            print('Failed to read the configuration file. See error:\n{}'.format(e))
            exit()

        try:
            [self._script_dirs.append(d) for d in self._conf_dict['SCRIPT_DIRS']]
        except Exception as e:
            rospy.logerr("Failed to read SCRIPT_DIRS from conf file\n{}".format(e))
            exit()


    def test_code_generation_and_execution(self):
        """Generate Python code from scripts listed in the conf file
        and run them."""
        
        script_files = []
        script_loc = []
        
        generated_code = ''
        script_file = ''
        smacha_script = ''
        script_names = []
        # Get a list of available files
        for d in self._script_dirs:
            file_list = os.listdir(os.path.join(self._base_path, d))
            script_files.extend(file_list)
            script_loc.extend([os.path.join(self._base_path, d)]*len(file_list))
        
        rospy.logwarn('script_files: {0}'.format(script_files))
        rospy.logwarn('script_loc: {0}'.format(script_loc))
        for test_case in self._conf_dict['TEST_GENERATE']:
            with self.subTest(test_case=test_case):
                success = False
                try:
                    matching_files = (s for s in script_files if test_case.values()[0]['script'] in s).next()

                    if len(matching_files) == 0:
                        raise Exception("Script not found in directories")
                    try:
                        script_file = script_loc[script_files.index(matching_files)] + '/' + matching_files
                    except:
                        rospy.logerr('script_loc:\n{}'.format(script_loc))
                        rospy.logerr('matching_files:\n{}'.format(matching_files))
                        rospy.logerr('script_files.index(matching_files):\n{}'.format(script_files.index(matching_files)))
                        raise Exception("Failed to compile string")
                    # Wait for service
                    self._generator.wait_for_service()

                    # Read the script
                    script_file_handle = open(script_file, "r")
                    smacha_script = script_file_handle.read()
                    script_file_handle.close()
                    # Pass the script into the generator service and write the result into a variable
                    generated_code = self._generator(smacha_script).code
                    # We need to take the "init_node" from the generated code as this program here already initializes a node
                    generated_code = generated_code.replace(
                    "rospy.init_node",
                    "# rospy.init_node"
                    )

                    # Execute the generated python code
                    exec(generated_code, globals())
                    if self._logerr:
                        raise Exception("Error in state machine execution !!")
                    success = True
                except Exception as e:
                    rospy.logerr(
                        'Failed due to exception when generating [{0}]!!'.format(script_file))
                    rospy.logerr('Exception: {0}'.format(e))
                    rospy.logerr('YAML\n===========\n{0}'.format(smacha_script))
                
                self.assertTrue(success) 


if __name__=="__main__":
    
    rospy.init_node('test_smacha_service_generate',log_level=rospy.DEBUG)
    rostest.rosrun('smacha_ros', 'test_smacha_service_generate', TestClass)
