/*
 * Copyright (C) 2018 Sertac Karaman
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>
 *
 */

#ifndef _SMP_COLLISION_CHECKER_STANDARD_HPP_
#define _SMP_COLLISION_CHECKER_STANDARD_HPP_

#include <smp/components/collision_checkers/standard.h>

#include <smp/common/region.hpp>
#include <smp/components/collision_checkers/base.hpp>

template <class typeparams, int NUM_DIMENSIONS>
smp::collision_checker_standard<typeparams,
                                NUM_DIMENSIONS>::collision_checker_standard() {

  num_discretization_steps = 20;
  discretization_length = 0.1;
  discretization_method = 2;
}

template <class typeparams, int NUM_DIMENSIONS>
smp::collision_checker_standard<typeparams,
                                NUM_DIMENSIONS>::~collision_checker_standard() {

  for (typename std::list<region_t *>::iterator iter = list_obstacles.begin();
       iter != list_obstacles.end(); iter++) {

    region_t *region_curr = *iter;

    delete region_curr;
  }
}

template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<
    typeparams, NUM_DIMENSIONS>::cc_update_insert_vertex(vertex_t *vertex_in) {

  return 1;
}

template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<
    typeparams, NUM_DIMENSIONS>::cc_update_insert_edge(edge_t *edge_in) {

  return 1;
}

template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<
    typeparams, NUM_DIMENSIONS>::cc_update_delete_vertex(vertex_t *vertex_in) {

  return 1;
}

template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<
    typeparams, NUM_DIMENSIONS>::cc_update_delete_edge(edge_t *edge_in) {

  return 1;
}

// returns a negative number to indicate error
// returns 0 if there is a collision
// returns 1 if no collision
template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<
    typeparams, NUM_DIMENSIONS>::check_collision_state(state_t *state_in) {

  if (list_obstacles.size() == 0)
    return 1;

  for (typename std::list<region_t *>::iterator iter = list_obstacles.begin();
       iter != list_obstacles.end(); iter++) {
    region_t *region_curr = *iter;

    bool collision = true;

    for (int i = 0; i < NUM_DIMENSIONS; i++) {

      if (fabs((*state_in)[i] - region_curr->center[i]) >=
          region_curr->size[i] / 2.0)
        collision = false;
    }

    if (collision) {
      return 0;
    }
  }

  return 1;
}

// returns a negative number to indicate error
// returns 0 if there is a collision
// returns 1 if no collision
template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<typeparams, NUM_DIMENSIONS>::
    check_collision_trajectory(trajectory_t *trajectory_in) {

  if (list_obstacles.size() == 0)
    return 1;

  if (trajectory_in->list_states.size() == 0)
    return 1;

  typename std::list<state_t *>::iterator iter = trajectory_in->list_states.begin();

  state_t *state_prev = *iter;

  if (this->check_collision_state(state_prev) == 0)
    return 0;

  iter++;

  for (; iter != trajectory_in->list_states.end(); iter++) {

    state_t *state_curr = *iter;

    if (discretization_method != 0) {
      // Compute the increments
      double dist_total = 0.0;
      double increments[NUM_DIMENSIONS];
      for (int i = 0; i < NUM_DIMENSIONS; i++) {
        double increment_curr = (*state_curr)[i] - (*state_prev)[i];
        dist_total += increment_curr * increment_curr;
        increments[i] = increment_curr;
      }
      dist_total = sqrt(dist_total);

      // Compute the number of increments
      int num_increments;
      if (discretization_method == 1) {
        num_increments = num_discretization_steps;
      } else if (discretization_method == 2) {
        num_increments = (int)floor(dist_total / discretization_length);
      }

      // Execute the remaining only if the discretization is required.
      if (num_increments > 0) {

        for (int i = 0; i < NUM_DIMENSIONS; i++) // Normalize the increments.
          increments[i] = increments[i] / ((double)(num_increments + 1));

        for (typename std::list<region_t *>::iterator iter = list_obstacles.begin();
             iter != list_obstacles.end(); iter++) {

          region_t *region_curr = *iter;

          for (int idx_state = 1; idx_state <= num_increments; idx_state++) {
            bool collision = true;

            for (int i = 0; i < NUM_DIMENSIONS; i++) {
              if (fabs((*state_prev)[i] + increments[i] * idx_state -
                       region_curr->center[i]) >= region_curr->size[i] / 2.0) {
                collision = false;
              }
            }
            if (collision == true) {
              return 0;
            }
          }
        }
      }
    }

    if (check_collision_state(state_curr) == 0) {
      return 0;
    }

    state_prev = state_curr;
  }

  return 1;
}

template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<typeparams, NUM_DIMENSIONS>::
    set_discretization_steps(int num_discretization_steps_in) {

  if (num_discretization_steps <= 0) {
    num_discretization_steps = 0;
    discretization_length = 0;
    discretization_method = 0;
  } else {
    num_discretization_steps = num_discretization_steps_in;
    discretization_method = 1;
  }

  return 1;
}

template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<typeparams, NUM_DIMENSIONS>::
    set_discretization_length(double discretization_length_in) {

  if (discretization_length <= 0.0) {
    num_discretization_steps = 0;
    discretization_length = 0.05;
    discretization_method = 0;
  } else {
    discretization_length = discretization_length_in;
    discretization_method = 2;
  }

  return 1;
}

template <class typeparams, int NUM_DIMENSIONS>
int smp::collision_checker_standard<typeparams, NUM_DIMENSIONS>::add_obstacle(
    region_t &obstacle_in) {

  list_obstacles.push_back(new region_t(obstacle_in));

  return 1;
}

#endif
