/*******************************************************************************
 * Copyright (c) 2017 Nerian Vision Technologies
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

// Activate DLL export for GenTL
#define GCTLIDLL

#include "misc/common.h"
#include "system/library.h"
#include "system/system.h"
#include "interface/interface.h"
#include "device/logicaldevice.h"
#include "device/physicaldevice.h"
#include "stream/datastream.h"

#include <genicam/gentl.h>
#include <iostream>
#include <fstream>

/*
 * This is the adapter layer that maps calls to the GenTL C interface to
 * method calls of the internal objects.
 */

namespace GenTL {

/*
* Debugging macros (sorry for that)
*/

#ifdef ENABLE_DEBUGGING

#ifdef _WIN32
    std::fstream debugStream("C:\\debug\\gentl-debug-" + std::to_string(time(nullptr)) + ".txt", std::ios::out);
#else
    std::ostream& debugStream = std::cout;
#endif

#define DEBUG_VAL(val)      #val << " = " << (std::string(#val) == "iAddress" ? std::hex : std::dec) << val
#define DEBUG               {debugStream << __FUNCTION__ << std::endl;}
#define DEBUG1(a)           {debugStream << __FUNCTION__ << ": " << DEBUG_VAL(a) << std::endl;}
#define DEBUG2(a, b)        {debugStream << __FUNCTION__ << ": " << DEBUG_VAL(a) << "; " << DEBUG_VAL(b) << std::endl;}
#define DEBUG3(a, b, c)     {debugStream << __FUNCTION__ << ": "<< DEBUG_VAL(a) << "; " << DEBUG_VAL(b) \
    << "; " << DEBUG_VAL(c) << std::endl;}
#define DEBUG4(a, b, c, d)  {debugStream << __FUNCTION__ << ": "<< DEBUG_VAL(a) << "; " << DEBUG_VAL(b) \
    << "; " << DEBUG_VAL(c) << "; " << DEBUG_VAL(d) << std::endl;}

#else

#define DEBUG                 {}
#define DEBUG1(a)             {}
#define DEBUG2(a, b)          {}
#define DEBUG3(a, b, c)       {}
#define DEBUG4(a, b, c, d)    {}

#endif


thread_local GenTL::GC_ERROR lastError = GenTL::GC_ERR_SUCCESS;

// Verifies that the given handle is valid
inline bool verifyHandle(void* handle, Handle::HandleType type) {
    return handle != nullptr && reinterpret_cast<Handle*>(handle)->getType() == type;
}

/*
 * Library functions
 */

GC_API GCInitLib(void) {
    DEBUG;
    return lastError = Library::initLib();
}

GC_API GCCloseLib(void) {
    DEBUG;
    return lastError = Library::closeLib();
}

GC_API GCGetInfo(TL_INFO_CMD iInfoCmd, INFO_DATATYPE* piType, void* pBuffer, size_t* piSize) {
    DEBUG1(iInfoCmd);
    return lastError = Library::getInfo(iInfoCmd, piType, pBuffer, piSize);
}

GC_API GCGetLastError(GC_ERROR* piErrorCode, char* sErrorText, size_t* piSize) {
    DEBUG;
    return lastError = Library::getLastError(piErrorCode, sErrorText, piSize, lastError);
}


/*
 * System functions
 */

GC_API TLOpen(TL_HANDLE* phSystem) {
    DEBUG;
    return lastError = System::open(reinterpret_cast<System**>(phSystem));
}

GC_API TLClose(TL_HANDLE hSystem) {
    DEBUG1(hSystem);
    if(!verifyHandle(hSystem, Handle::TYPE_SYSTEM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return System::close(reinterpret_cast<System*>(hSystem));
    }
}

GC_API TLUpdateInterfaceList(TL_HANDLE hSystem, bool8_t* pbChanged, uint64_t iTimeout) {
    DEBUG1(hSystem);
    if(!verifyHandle(hSystem, Handle::TYPE_SYSTEM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<System*>(hSystem)->updateInterfaceList(pbChanged, iTimeout);
    }
}

GC_API TLGetNumInterfaces(TL_HANDLE hSystem, uint32_t* piNumIfaces) {
    DEBUG1(hSystem);
    if(!verifyHandle(hSystem, Handle::TYPE_SYSTEM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<System*>(hSystem)->getNumInterfaces(piNumIfaces);
    }
}

GC_API TLGetInterfaceID(TL_HANDLE hSystem, uint32_t iIndex, char* sIfaceID, size_t* piSize) {
    DEBUG2(hSystem, iIndex);
    if(!verifyHandle(hSystem, Handle::TYPE_SYSTEM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<System*>(hSystem)->getInterfaceID(iIndex, sIfaceID, piSize);
    }
}

GC_API TLGetInterfaceInfo(TL_HANDLE hSystem, const char* sIfaceID, INTERFACE_INFO_CMD iInfoCmd,
        INFO_DATATYPE* piType, void* pBuffer, size_t* piSize) {
    DEBUG3(hSystem, sIfaceID, iInfoCmd);
    if(!verifyHandle(hSystem, Handle::TYPE_SYSTEM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<System*>(hSystem)->getInterfaceInfo(sIfaceID, iInfoCmd, piType, pBuffer, piSize);
    }
}

GC_API TLOpenInterface(TL_HANDLE hSystem, const char* sIfaceID, IF_HANDLE* phIface) {
    DEBUG2(hSystem, sIfaceID);
    if(!verifyHandle(hSystem, Handle::TYPE_SYSTEM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<System*>(hSystem)->openInterface(sIfaceID, phIface);
    }
}

GC_API TLGetInfo(TL_HANDLE hSystem, TL_INFO_CMD iInfoCmd, INFO_DATATYPE* piType,
        void* pBuffer, size_t* piSize) {
    DEBUG2(hSystem, iInfoCmd);
    if(!verifyHandle(hSystem, Handle::TYPE_SYSTEM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<System*>(hSystem)->getInfo(iInfoCmd, piType, pBuffer, piSize);
    }
}


/*
 * Interface functions
 */

GC_API IFClose(IF_HANDLE hIface) {
    DEBUG1(hIface);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->close();
    }
}

GC_API IFUpdateDeviceList(IF_HANDLE hIface, bool8_t* pbChanged, uint64_t iTimeout) {
    DEBUG2(hIface, iTimeout);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->updateDeviceList(pbChanged, iTimeout);
    }
}

GC_API IFGetNumDevices(IF_HANDLE hIface, uint32_t* piNumDevices) {
    DEBUG1(hIface);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->getNumDevices(piNumDevices);
    }
}

GC_API IFGetDeviceID(IF_HANDLE hIface, uint32_t iIndex, char* sDeviceID, size_t* piSize) {
    DEBUG2(hIface, iIndex);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->getDeviceID(iIndex, sDeviceID, piSize);
    }
}

GC_API IFGetDeviceInfo(IF_HANDLE hIface, const char* sDeviceID, DEVICE_INFO_CMD iInfoCmd,
        INFO_DATATYPE* piType, void* pBuffer, size_t* piSize) {
    DEBUG3(hIface, sDeviceID, iInfoCmd);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->getDeviceInfo(sDeviceID, iInfoCmd, piType, pBuffer, piSize);
    }
}

GC_API IFGetParentTL(IF_HANDLE hIface, TL_HANDLE* phSystem) {
    DEBUG2(hIface, phSystem);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->getParentTL(phSystem);
    }
}

GC_API IFGetInfo(IF_HANDLE hIface, INTERFACE_INFO_CMD iInfoCmd, INFO_DATATYPE* piType,
        void* pBuffer, size_t* piSize) {
    DEBUG2(hIface, iInfoCmd);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->getInfo(iInfoCmd, piType, pBuffer, piSize);
    }
}

GC_API IFOpenDevice(IF_HANDLE hIface, const char* sDeviceID, DEVICE_ACCESS_FLAGS iOpenFlags,
        DEV_HANDLE* phDevice) {
    DEBUG3(hIface, sDeviceID, iOpenFlags);
    if(!verifyHandle(hIface, Handle::TYPE_INTERFACE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Interface*>(hIface)->openDevice(sDeviceID, iOpenFlags, phDevice);
    }
}


/*
 * Device functions
 */

GC_API DevClose(DEV_HANDLE hDevice) {
    DEBUG1(hDevice);
    if(!verifyHandle(hDevice, Handle::TYPE_DEVICE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        lastError =  reinterpret_cast<LogicalDevice*>(hDevice)->close();
        reinterpret_cast<LogicalDevice*>(hDevice)->getPhysicalDevice()->getInterface()->getSystem()->freeUnusedDevices();
        return lastError;
    }
}

GC_API DevGetParentIF(DEV_HANDLE hDevice, IF_HANDLE* phIface) {
    DEBUG1(hDevice);
    if(!verifyHandle(hDevice, Handle::TYPE_DEVICE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<LogicalDevice*>(hDevice)->getParentIF(phIface);
    }
}

GC_API DevGetInfo(DEV_HANDLE hDevice, DEVICE_INFO_CMD iInfoCmd, INFO_DATATYPE* piType,
        void* pBuffer, size_t* piSize) {
    DEBUG2(hDevice, iInfoCmd);
    if(!verifyHandle(hDevice, Handle::TYPE_DEVICE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<LogicalDevice*>(hDevice)->getInfo(
            iInfoCmd, piType, pBuffer, piSize);
    }
}

GC_API DevGetNumDataStreams(DEV_HANDLE hDevice, uint32_t* piNumDataStreams) {
    DEBUG1(hDevice);
    if(!verifyHandle(hDevice, Handle::TYPE_DEVICE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<LogicalDevice*>(hDevice)->getNumDataStreams(piNumDataStreams);
    }
}

GC_API DevGetPort(DEV_HANDLE hDevice, PORT_HANDLE* phRemoteDev) {
    DEBUG1(hDevice);
    if(!verifyHandle(hDevice, Handle::TYPE_DEVICE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<LogicalDevice*>(hDevice)->getPort(phRemoteDev);
    }
}

GC_API DevGetDataStreamID(DEV_HANDLE hDevice, uint32_t iIndex, char* sDataStreamID,
        size_t* piSize) {
    DEBUG2(hDevice, iIndex);
    if(!verifyHandle(hDevice, Handle::TYPE_DEVICE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<LogicalDevice*>(hDevice)->getDataStreamID(
            iIndex, sDataStreamID, piSize);
    }
}

GC_API DevOpenDataStream(DEV_HANDLE hDevice, const char* sDataStreamID, DS_HANDLE* phDataStream) {
    DEBUG2(hDevice, sDataStreamID);
    if(!verifyHandle(hDevice, Handle::TYPE_DEVICE)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<LogicalDevice*>(hDevice)->openDataStream(sDataStreamID, phDataStream);
    }
}


/*
 * Data stream functions
 */

GC_API DSAnnounceBuffer(DS_HANDLE hDataStream, void* pBuffer, size_t iSize, void* pPrivate, BUFFER_HANDLE* phBuffer) {
    DEBUG2(hDataStream, iSize);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->announceBuffer(pBuffer, iSize, pPrivate, phBuffer);
    }
}

GC_API DSAllocAndAnnounceBuffer(DS_HANDLE hDataStream, size_t iBufferSize, void* pPrivate, BUFFER_HANDLE* phBuffer) {
    DEBUG2(hDataStream, iBufferSize);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->allocAndAnnounceBuffer(iBufferSize, pPrivate, phBuffer);
    }
}

GC_API DSClose(DS_HANDLE hDataStream) {
    DEBUG1(hDataStream);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->close();
    }
}

GC_API DSRevokeBuffer(DS_HANDLE hDataStream, BUFFER_HANDLE hBuffer, void ** ppBuffer, void ** ppPrivate) {
    DEBUG2(hDataStream, hBuffer);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM) || !verifyHandle(hBuffer, Handle::TYPE_BUFFER)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->revokeBuffer(hBuffer, ppBuffer, ppPrivate);
    }
}

GC_API DSQueueBuffer(DS_HANDLE hDataStream, BUFFER_HANDLE hBuffer) {
    DEBUG2(hDataStream, hBuffer);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM) || !verifyHandle(hBuffer, Handle::TYPE_BUFFER)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->queueBuffer(hBuffer);
    }
}

GC_API DSGetParentDev(DS_HANDLE hDataStream, DEV_HANDLE* phDevice) {
    DEBUG1(hDataStream);
   if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->getParentDev(phDevice);
    }
}

GC_API DSStartAcquisition(DS_HANDLE hDataStream, ACQ_START_FLAGS iStartFlags, uint64_t iNumToAcquire) {
    DEBUG2(hDataStream, iNumToAcquire);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->startAcquisition(iStartFlags, iNumToAcquire);
    }
}

GC_API DSStopAcquisition(DS_HANDLE hDataStream, ACQ_STOP_FLAGS iStopFlags) {
    DEBUG1(hDataStream);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->stopAcquisition(iStopFlags);
    }
}

GC_API DSFlushQueue(DS_HANDLE hDataStream, ACQ_QUEUE_TYPE iOperation) {
    DEBUG2(hDataStream, iOperation);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->flushQueue(iOperation);
    }
}

GC_API DSGetBufferChunkData(DS_HANDLE hDataStream, BUFFER_HANDLE hBuffer, SINGLE_CHUNK_DATA* pChunkData,
        size_t* piNumChunks) {
    DEBUG2(hDataStream, hBuffer);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM) || !verifyHandle(hBuffer, Handle::TYPE_BUFFER)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->getBufferChunkData(hBuffer, pChunkData, piNumChunks);
    }
}

GC_API DSGetBufferID(DS_HANDLE hDataStream, uint32_t iIndex, BUFFER_HANDLE* phBuffer) {
   DEBUG2(hDataStream, iIndex);
   if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->getBufferID(iIndex, phBuffer);
    }
}

GC_API DSGetBufferInfo(DS_HANDLE hDataStream, BUFFER_HANDLE hBuffer, BUFFER_INFO_CMD iInfoCmd, INFO_DATATYPE* piType,
        void* pBuffer, size_t* piSize) {
    DEBUG3(hDataStream, hBuffer, iInfoCmd);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM) || !verifyHandle(hBuffer, Handle::TYPE_BUFFER)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->getBufferInfo(hBuffer, iInfoCmd, piType, pBuffer, piSize);
    }
}

GC_API DSGetInfo(DS_HANDLE hDataStream, STREAM_INFO_CMD iInfoCmd, INFO_DATATYPE* piType,
        void* pBuffer, size_t* piSize) {
    DEBUG2(hDataStream, iInfoCmd);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->getInfo(iInfoCmd, piType, pBuffer, piSize);
    }
}

GC_API DSGetNumBufferParts(DS_HANDLE hDataStream, BUFFER_HANDLE hBuffer, uint32_t *piNumParts ) {
    DEBUG2(hDataStream, hBuffer);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM) || !verifyHandle(hBuffer, Handle::TYPE_BUFFER)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->getNumBufferParts(hBuffer, piNumParts);
    }
}

GC_API DSGetBufferPartInfo(DS_HANDLE hDataStream, BUFFER_HANDLE hBuffer, uint32_t iPartIndex,
        BUFFER_PART_INFO_CMD iInfoCmd, INFO_DATATYPE *piType, void *pBuffer, size_t *piSize) {
    DEBUG4(hDataStream, hBuffer, iPartIndex, iInfoCmd);
    if(!verifyHandle(hDataStream, Handle::TYPE_STREAM) || !verifyHandle(hBuffer, Handle::TYPE_BUFFER)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<DataStream*>(hDataStream)->getBufferPartInfo(hBuffer, iPartIndex,
            iInfoCmd, piType, pBuffer, piSize);
    }
}


/*
 * Port functions
 */

GC_API GCGetPortInfo(PORT_HANDLE hPort, PORT_INFO_CMD iInfoCmd, INFO_DATATYPE* piType,
        void* pBuffer, size_t* piSize) {
    DEBUG2(hPort, iInfoCmd);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->getPortInfo(iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->getPortInfo(iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->getPortInfo(iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->getPortInfo(iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->getPortInfo(iInfoCmd, piType, pBuffer, piSize);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}

GC_API GCGetPortURL(PORT_HANDLE hPort, char* sURL, size_t* piSize) {
    DEBUG1(hPort);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->getPortURL(sURL, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->getPortURL(sURL, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->getPortURL(sURL, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->getPortURL(sURL, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->getPortURL(sURL, piSize);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}

GC_API GCGetNumPortURLs(PORT_HANDLE hPort, uint32_t* piNumURLs) {
    DEBUG1(hPort);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->getNumPortURLs(piNumURLs);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->getNumPortURLs(piNumURLs);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->getNumPortURLs(piNumURLs);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->getNumPortURLs(piNumURLs);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->getNumPortURLs(piNumURLs);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}

GC_API GCGetPortURLInfo(PORT_HANDLE hPort, uint32_t iURLIndex, URL_INFO_CMD iInfoCmd,
        INFO_DATATYPE* piType, void* pBuffer, size_t* piSize) {
    DEBUG3(hPort, iURLIndex, iInfoCmd);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->getPortURLInfo(iURLIndex, iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->getPortURLInfo(iURLIndex, iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->getPortURLInfo(iURLIndex, iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->getPortURLInfo(iURLIndex, iInfoCmd, piType, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->getPortURLInfo(iURLIndex, iInfoCmd, piType, pBuffer, piSize);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}

GC_API GCReadPort(PORT_HANDLE hPort, uint64_t iAddress, void* pBuffer, size_t* piSize) {
    DEBUG2(hPort, iAddress);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->readPort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->readPort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->readPort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->readPort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->readPort(iAddress, pBuffer, piSize);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}

GC_API GCWritePort(PORT_HANDLE hPort, uint64_t iAddress, const void* pBuffer, size_t* piSize) {
    DEBUG2(hPort, iAddress);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->writePort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->writePort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->writePort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->writePort(iAddress, pBuffer, piSize);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->writePort(iAddress, pBuffer, piSize);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}

GC_API GCWritePortStacked(PORT_HANDLE hPort, PORT_REGISTER_STACK_ENTRY* pEntries, size_t* piNumEntries) {
    DEBUG1(hPort);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->writePortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->writePortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->writePortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->writePortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->writePortStacked(pEntries, piNumEntries);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}

GC_API GCReadPortStacked(PORT_HANDLE hPort, PORT_REGISTER_STACK_ENTRY* pEntries, size_t* piNumEntries) {
    DEBUG1(hPort);
    if(verifyHandle(hPort, Handle::TYPE_PORT)) {
        return lastError = reinterpret_cast<Port*>(hPort)->readPortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_SYSTEM)) {
        return lastError = reinterpret_cast<System*>(hPort)->getPort()->readPortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_INTERFACE)) {
        return lastError = reinterpret_cast<Interface*>(hPort)->getPort()->readPortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_DEVICE)) {
        return lastError = reinterpret_cast<LogicalDevice*>(hPort)->getLocalPort()->readPortStacked(pEntries, piNumEntries);
    } else if(verifyHandle(hPort, Handle::TYPE_STREAM)) {
        return lastError = reinterpret_cast<DataStream*>(hPort)->getPort()->readPortStacked(pEntries, piNumEntries);
    } else {
        return lastError = GC_ERR_INVALID_HANDLE;
    }
}


/*
 * Event functions
 */

GC_API EventFlush(EVENT_HANDLE hEvent) {
    DEBUG1(hEvent);
    if(!verifyHandle(hEvent, Handle::TYPE_EVENT)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Event*>(hEvent)->flush();
    }
}

GC_API EventGetData(EVENT_HANDLE hEvent, void* pBuffer, size_t* piSize, uint64_t iTimeout) {
    DEBUG2(hEvent, iTimeout);
    if(!verifyHandle(hEvent, Handle::TYPE_EVENT)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Event*>(hEvent)->getData(pBuffer, piSize, iTimeout);
    }
}

GC_API EventGetDataInfo(EVENT_HANDLE hEvent, const void* pInBuffer, size_t iInSize,
        EVENT_DATA_INFO_CMD iInfoCmd, INFO_DATATYPE* piType, void* pOutBuffer, size_t* piOutSize) {
    DEBUG3(hEvent, iInSize, iInfoCmd);
    if(!verifyHandle(hEvent, Handle::TYPE_EVENT)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Event*>(hEvent)->getDataInfo(pInBuffer, iInSize, iInfoCmd, piType, pOutBuffer, piOutSize);
    }
}

GC_API EventGetInfo(EVENT_HANDLE hEvent, EVENT_INFO_CMD iInfoCmd, INFO_DATATYPE * piType,
        void * pBuffer, size_t * piSize) {
    DEBUG2(hEvent, iInfoCmd);
    if(!verifyHandle(hEvent, Handle::TYPE_EVENT)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Event*>(hEvent)->getInfo(iInfoCmd, piType, pBuffer, piSize);
    }
}

GC_API EventKill(EVENT_HANDLE hEvent) {
    DEBUG1(hEvent);
    if(!verifyHandle(hEvent, Handle::TYPE_EVENT)) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = reinterpret_cast<Event*>(hEvent)->kill();
    }
}

GC_API GCRegisterEvent(EVENTSRC_HANDLE hModule, EVENT_TYPE iEventID, EVENT_HANDLE* phEvent) {
    DEBUG2(hModule, iEventID);
    if(hModule == nullptr) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = Event::registerEvent(hModule, iEventID, phEvent);
    }
}

GC_API GCUnregisterEvent(EVENTSRC_HANDLE hModule, EVENT_TYPE iEventID) {
    DEBUG2(hModule, iEventID);
    if (hModule == nullptr) {
        return lastError = GC_ERR_INVALID_HANDLE;
    } else {
        return lastError = Event::unregisterEvent(hModule, iEventID);
    }
}

}

