/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#include "displaywidget.h"
#include "nvcom.h"

#include <QMouseEvent>

constexpr int DisplayWidget::IMAGE_PADDING;

DisplayWidget::DisplayWidget(QWidget * parent): QWidget(parent),
        resizeWidget(false), displayLeft(true), displayRight(true), zoomPercent(100),
        mouseX(-1), mouseY(-1), mouseOnLeftImage(false), labelFont("Courier New", 12, QFont::Bold) {

    labelFont.setStyleHint(QFont::Monospace);
    labelFont.setWeight(2);
    setMouseTracking(true);
}

void DisplayWidget::setNVCom(std::shared_ptr<NVCom> nvcom) {
    this->nvcom = nvcom;
}

void DisplayWidget::setDisplayFrame(const cv::Mat_<cv::Vec3b>& left,
        const cv::Mat_<cv::Vec3b>& right, bool resize) {
    displayLeftFrame = left;
    displayRightFrame = right;
    if(resize) {
        resizeWidget = true;
    }
}

void DisplayWidget::setZoom(int percent) {
    zoomPercent = percent;
    resizeWidget = true;
}

void DisplayWidget::setImagesToDisplay(bool displayLeft, bool displayRight) {
    this->displayLeft = displayLeft;
    this->displayRight = displayRight;
}

void DisplayWidget::paintEvent(QPaintEvent *) {
    if(nvcom == nullptr) {
        // Should never happen
        return;
    }

    std::unique_lock<std::mutex> lock(nvcom->getDisplayMutex());
    if(displayLeftFrame.data == nullptr) {
        return; // No frame available yet
    }

    const double scale = zoomPercent/100.0;

    // Resize widget if necessary
    if(resizeWidget) {
        int width = 0;
        if(displayLeft && displayRight) {
            width = displayLeftFrame.cols*scale + displayRightFrame.cols*scale + IMAGE_PADDING;
        } else if(displayLeft) {
            width = displayLeftFrame.cols*scale;
        } else if(displayRight) {
            width = displayRightFrame.cols*scale;
        }

        setFixedSize(width, displayLeftFrame.rows*scale);
        resizeWidget = false;
    }

    painter.begin(this);
    painter.scale(scale, scale);

    // Display the received images
    QImage img1(displayLeftFrame.data, displayLeftFrame.cols, displayLeftFrame.rows,
        displayLeftFrame.step[0], QImage::Format_RGB888);

    QImage img2(displayRightFrame.data, displayRightFrame.cols, displayRightFrame.rows,
        displayRightFrame.step[0], QImage::Format_RGB888);

    if(displayLeft && displayRight) {
        painter.drawImage(0, 0, img1);
        painter.drawImage(displayLeftFrame.cols + IMAGE_PADDING/scale, 0, img2);
        drawMouseLabel();
    } else if(displayLeft) {
        painter.drawImage(0, 0, img1);
    } else if(displayRight) {
        painter.drawImage(0, 0, img2);
        drawMouseLabel();
    }

    painter.end();
}

void DisplayWidget::drawMouseLabel() {
    if(mouseX < 0 || mouseY < 0) {
        return; // No position to draw
    }

    cv::Point3f point = nvcom->getDisparityMapPoint(mouseX, mouseY);
    if(point == cv::Point3f(0, 0, 0)) {
        // No point available
        return;
    }

    char labelStr[100];
    snprintf(labelStr, sizeof(labelStr), "x = %7.3lf m\ny = %7.3lf m\nz = %7.3lf m",
        point.x, point.y, point.z);

    const double scale = zoomPercent/100.0;

    int drawX;
    if(mouseOnLeftImage) {
        drawX = mouseX;
    } else {
        drawX = mouseX + displayLeftFrame.cols + IMAGE_PADDING/scale;
    }
    int drawY = mouseY;

    painter.setPen(QPen(QColor::fromRgb(255, 0, 0), 2));
    painter.drawEllipse(QPointF(drawX, mouseY), 6, 6);

    int offsetX = 25, offsetY = 0;
    painter.setPen(Qt::black);
    painter.setFont(labelFont);
    QRectF boundingBox;
    painter.drawText(QRectF(drawX + offsetX, drawY + offsetY, 1000, 1000), Qt::AlignLeft | Qt::TextDontPrint,
        labelStr, &boundingBox);

    if((mouseOnLeftImage && boundingBox.x() + boundingBox.width() > displayLeftFrame.cols) ||
            boundingBox.x() + boundingBox.width() > width()/scale) {
        // Display label to the left if there isn't enough space on the right
        boundingBox.translate(-boundingBox.width() - 2*offsetX, 0);
        drawX -= boundingBox.width();
        offsetX = -offsetX;
    }

    if(boundingBox.y() + boundingBox.height() > height()/scale) {
        // Display label on top if there isn't enough space below
        boundingBox.translate(0, -boundingBox.height() - 2*offsetY);
        drawY -= boundingBox.height();
        offsetY = -offsetY;
    }

    constexpr int border = 10;
    painter.fillRect(boundingBox.x() - border, boundingBox.y() - border,
        boundingBox.width() + 2*border, boundingBox.height() + 2*border, QColor::fromRgbF(1.0, 1.0, 1.0, 0.5));
    painter.drawText(QRectF(drawX + offsetX, drawY + offsetY, 1000, 1000), Qt::AlignLeft,
        labelStr);
}


void DisplayWidget::mouseMoveEvent(QMouseEvent *event) {
    if(event != nullptr) {
        double scale = zoomPercent/100.0;
        int x;
        if(event->x() > displayLeftFrame.cols*scale) {
            x = (event->x() - IMAGE_PADDING) / scale - displayLeftFrame.cols;
            mouseOnLeftImage = false;
        } else {
            x = event->x() / scale;
            mouseOnLeftImage = true;
        }

        int y = event->y() / scale;

        if(x >= 0 && x < displayLeftFrame.cols) {
            mouseX = x;
            mouseY = y;
            repaint();
        }
    }
}
