/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef NVSHARED_IMAGEQUEUE_H
#define NVSHARED_IMAGEQUEUE_H

#include <vector>
#include <string>
#include <utility>
#include <opencv2/opencv.hpp>
#include <thread>
#include <memory>
#include <mutex>
#include <condition_variable>

// Loads a sequence of stereo images through a backround thread
template <class T>
class ImageQueue {
public:
    typedef T FrameType;

    ImageQueue(bool allowFrameSkip, int maxQueueSize, bool multiThreaded);
    virtual ~ImageQueue();

    // Pops a new image from the queue
    std::shared_ptr<const T> pop(double timeout = -1);

    int getQueueSize() {return queueSize;}

protected:
    // Pushes a new frame to the que
    void push(const std::shared_ptr<const T>& frame);
    // Signals that the queue has reached its end
    void closeQueue();
    // Waits until there's a free slot in the queue
    void waitForFreeSlot();

    // Method for loading the next frame. Has to be overloaded.
    virtual void queueFrame() =  0;

    // Stops the thread if it has been created. Should be called
    // in destructor.
    void stopThreads();

private:
    std::thread imageThread; // Queue thread
    bool allowFrameSkip;
    int queueSize;
    std::vector<std::shared_ptr<const T> > queue; // The image queue
    int readIndex, writeIndex; // Current index of the queue
    std::mutex queueMutex;
    std::condition_variable popCond, pushCond;
    volatile bool done; // True if we reached the end
    bool multiThreaded; // If enabled, images are prefetched asynchroneously
    bool threadCreated;

    // Main method for the queue thread
    void threadMain();
};


// Frame types
template <class T>
struct MonoFrame {
    typedef cv::Mat_<T> Type;
    typedef std::shared_ptr<Type> Ptr;
    typedef std::shared_ptr<const Type> ConstPtr;
};

template <class T>
struct StereoFrame {
    typedef std::pair<cv::Mat_<T>, cv::Mat_<T> > Type;
    typedef std::shared_ptr<Type> Ptr;
    typedef std::shared_ptr<const Type> ConstPtr;
};

template <class T>
struct DoubleStereoFrame {
    typedef std::pair<std::pair<cv::Mat_<T>, cv::Mat_<T> >,
        std::pair<cv::Mat_<T>, cv::Mat_<T> > > Type;
    typedef std::shared_ptr<Type> Ptr;
    typedef std::shared_ptr<const Type> ConstPtr;
};

typedef MonoFrame<unsigned char> MonoFrame8U;
typedef MonoFrame<unsigned short> MonoFrame16U;
typedef MonoFrame<float> MonoFrame32F;
typedef StereoFrame<unsigned char> StereoFrame8U;
typedef StereoFrame<unsigned short> StereoFrame16U;
typedef StereoFrame<float> StereoFrame32F;
typedef DoubleStereoFrame<unsigned char> DoubleStereoFrame8U;
typedef DoubleStereoFrame<unsigned short> DoubleStereoFrame16U;
typedef DoubleStereoFrame<float> DoubleStereoFrame32F;


// Typedefs for common queues
typedef ImageQueue<MonoFrame<unsigned char>::Type> MonoImageQueue8U;
typedef ImageQueue<StereoFrame<unsigned char>::Type> StereoImageQueue8U;
typedef ImageQueue<DoubleStereoFrame<unsigned char>::Type> DoubleStereoImageQueue8U;
typedef ImageQueue<MonoFrame<unsigned short>::Type> MonoImageQueue16U;
typedef ImageQueue<StereoFrame<unsigned short>::Type> StereoImageQueue16U;
typedef ImageQueue<DoubleStereoFrame<unsigned short>::Type> DoubleStereoImageQueue16U;
typedef ImageQueue<MonoFrame<float>::Type> MonoImageQueue32F;
typedef ImageQueue<StereoFrame<float>::Type> StereoImageQueue32F;
typedef ImageQueue<DoubleStereoFrame<float>::Type> DoubleStereoImageQueue32F;

#endif
