/*******************************************************************************
 * Copyright (c) 2019 Nerian Vision GmbH
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *******************************************************************************/

#ifndef NVCOM_H
#define NVCOM_H

#include <queue>
#include <memory>
#include <thread>
#include <mutex>
#include <functional>
#include <opencv2/opencv.hpp>
#include <fstream>

#include <visiontransfer/asynctransfer.h>
#include <visiontransfer/reconstruct3d.h>
#include "ratelimit.h"
#include "imagereader.h"
#include "colorcoder.h"
#include "settings.h"

/*
 * All functionality of NVCom except the GUI
 */
class NVCom {
public:
    NVCom(const Settings& newSettings);
    ~NVCom();

    void terminate();
    void connect();

    unsigned int getWriteQueueSize() {return writeQueue.size();}
    void updateSettings(const Settings& newSettings);

    void captureSingleFrame() {captureNextFrame = true;}
    void setCaptureSequence(bool capture) {captureSequence = capture;}
    void resetCaptureIndex() {captureIndex = 0;}

    // Methods for setting callback functions
    void setFrameDisplayCallback(const std::function<void(const cv::Mat_<cv::Vec3b>& left,
            const cv::Mat_<cv::Vec3b>& right, bool resize)>& f) {
        frameDisplayCallback = f;
    }
    void setExceptionCallback(const std::function<void(const std::exception& ex)>& f) {
        exceptionCallback = f;
    }
    void setSendCompleteCallback(const std::function<void()>& f) {
        sendCompleteCallback = f;
    }
    void setConnectedCallback(const std::function<void()>& f) {
        connectedCallback = f;
    }
    void setDisconnectCallback(const std::function<void()>& f) {
        disconnectCallback = f;
    }

    // A mutex that should be used when accessing the display frames
    // outside the callback
    std::mutex& getDisplayMutex() {return displayMutex;}

    void getBitDepths(int& bitsLeft, int& bitsRight);

    // Projects the given point of the disparity map to 3D
    cv::Point3f getDisparityMapPoint(int x, int y);

    int getNumDroppedFrames() {
        return asyncTrans->getNumDroppedFrames();
    }

private:
    Settings settings;

    volatile bool terminateThreads;
    volatile bool captureNextFrame;
    volatile bool captureSequence;
    int captureIndex;
    int seqNum;
    int minDisparity;
    int maxDisparity;

    std::unique_ptr<ImageReader> imageReader;
    std::unique_ptr<RateLimit> frameRateLimit;
    std::unique_ptr<visiontransfer::AsyncTransfer> asyncTrans;
    std::queue<std::pair<std::string, cv::Mat> > writeQueue;
    std::unique_ptr<ColorCoder> redBlueCoder;
    std::unique_ptr<ColorCoder> rainbowCoder;
    visiontransfer::Reconstruct3D recon3d;

    cv::Size2i lastFrameSize;
    int lastLeftFormat, lastRightFormat;
    cv::Mat_<cv::Vec3b> convertedLeftFrame;
    cv::Mat_<cv::Vec3b> convertedRightFrame;
    std::fstream timestampFile;
    visiontransfer::ImagePair lastImagePair;

    // Callback functions
    std::function<void(const cv::Mat_<cv::Vec3b>&, const cv::Mat_<cv::Vec3b>&, bool)> frameDisplayCallback;
    std::function<void(const std::exception& ex)> exceptionCallback;
    std::function<void()> sendCompleteCallback;
    std::function<void()> connectedCallback;
    std::function<void()> disconnectCallback;

    // Variables for main loop thread
    std::thread mainLoopThread;
    std::mutex displayMutex;
    std::mutex imagePairMutex;

    // Variables for writing thread
    std::thread writingThread;
    std::mutex writingMutex;
    std::condition_variable writingCond;

    void mainLoop();
    void writeLoop();

    void framePause();
    void scheduleWrite(const cv::Mat& frame, int index, int camera);
    void captureFrameIfRequested(const visiontransfer::ImagePair& imagePair, const cv::Mat& receivedLeftFrame,
        const cv::Mat& receivedRightFrame);
    bool transmitFrame();
    void receiveFrame(visiontransfer::ImagePair& imagePair, cv::Mat& receivedLeftFrame, cv::Mat& receivedRightFrame);
    void convertFrame(const cv::Mat& src, cv::Mat_<cv::Vec3b>& dst, bool colorCode);
    void colorCodeAndDisplay(const cv::Mat& receivedLeftFrame, const cv::Mat& receivedRightFrame,
        bool imageDisparityPair);
    void joinAllThreads();
};

#endif
