// -*- C++ -*-
/*!
 * @file  InPortProvider.h
 * @brief InPortProvider class
 * @date  $Date: 2007-12-31 03:08:03 $
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2006-2010
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: InPortProvider.h 2146 2011-05-27 09:56:11Z n-ando $
 *
 */

#ifndef RTC_INPORTPROVIDER_H
#define RTC_INPORTPROVIDER_H

#include <string>

#include <coil/Factory.h>

#include <rtm/BufferBase.h>
#include <rtm/NVUtil.h>
#include <rtm/SystemLogger.h>
#include <rtm/DataPortStatus.h>
#include <rtm/InPortConnector.h>

namespace RTC
{
  class ConnectorListeners;
  class ConnectorInfo;

  /*!
   * @if jp
   *
   * @class OutPortProvider
   * @brief OutPortProvider
   *
   * InPort  PROVIDED 󥿡ե뤿ݴ쥯饹
   * InPort Фƿ󥿡եˤϡΥ
   * Ѿʲδؿɬפ롣
   *
   * - init()
   * - setBuffer()
   * - setListener()
   * - setConnector()
   * 
   * ˡ󥹥ȥ饯ǰʲδؿƤӡɬ
   * פ롣
   * 
   * - setInterfaceType()
   * - setDataFlowType()
   * - setSubscriptionType()
   *
   * ΤۤInPortProvider ΥץѥƥȤƳ˸ɬפΤ
   * ͤϡprotected ѿ (SDOPackage::NVList) m_properties Ф
   * åȤ뤳ȡåȤ줿ͤϡ󥿡եΥץե
   * ơޤ³¾Υ󥿡եˤΥ󥿡ե˴
   * ͿݤѤ롣ʲβ۴ؿϡݡȤΥ󥿡
   * եץե³˥ݡȤƤӽФ롣
   * ͽ᥻åȤ줿Υ󥿡եΥץեϤδ
   * ƤӽФˤݡȤ롣
   *
   * - publishInterfaceProfile()
   * - publishInterface()
   *
   * InPort  InPortProvider Υեȥ饹ФѲǽ
   *  InPortProvider 礻󶡲ǽʥ󥿡եפ
   * 롣äơInPortФ PROVIDED 󥿡ե
   * 󶡤 InPortProvider Υ֥饹ϡInPortProviderFactory
   * ˥եȥؿϿɬפ롣
   *
   * RTC::InPortProviderFactory::instance().addFactory() 
   *
   * - 1: ץХ̾, "corba_cdr" ʤ
   * - 2: եȥؿ coil::Creator<B, T>
   * - 3: ؿ coil::Destructor<B, T>
   * 
   * ͿƸƤӽФɬפ롣ʲϡեȥؤϿȡ
   * ؿȤǤ롣
   * 
   * <pre>
   * extern "C"
   * {
   *   void InPortCorbaCdrProviderInit(void)
   *   {
   *     RTC::InPortProviderFactory&
   *                         factory(RTC::InPortProviderFactory::instance());
   *     factory.addFactory("corba_cdr",
   *                        ::coil::Creator<::RTC::InPortProvider,
   *                                        ::RTC::InPortCorbaCdrProvider>,
   *                        ::coil::Destructor<::RTC::InPortProvider,
   *                                           ::RTC::InPortCorbaCdrProvider>);
   *   }
   * };
   * </pre>
   *
   * Τ褦ˡեȥؤϿؿȤơextern "C"
   * ˤꥷܥ򻲾ȲǽˤƤ뤳Ȥǡ
   * InPortProvider ֥ͭȲ (DLL) ưŪɲǽ
   * ץХηưŪɲä뤳ȤǽȤʤ롣
   *
   * @since 0.4.0
   *
   * @else
   *
   * @class InPortProvider
   * @brief InPortProvider
   *
   * The virtual class for InPort's PROVIDED interface
   * implementation.  New interface for InPort have to inherit this
   * class, and have to implement the following functions.
   *
   * - init()
   * - setBuffer()
   * - setListener()
   * - setConnector()
   * 
   * Moreover, calling the following functions in the constructor, and
   * properties have to be set.
   *
   * - setPortType()
   * - setDataType()
   * - setInterfaceType()
   * - setDataFlowType()
   * - setSubscriptionType()
   *
   * InPortProvider's properties that have to be provided to others
   * should be set to protected variable (SDOPackage::NVList)
   * m_properties. Values that are set to the property are published
   * as interface profile information, and it is also published to
   * required interface when connection is established.  The following
   * virtual functions are called when port's profiles are acquired
   * from others or connections are established. The following virtual
   * functions are called when port's profiles are acquired from
   * others or connections are established. Interface profile
   * information that is reviously set is given to Port calling by
   * these functions.
   *
   * - publishInterfaceProfile()
   * - publishInterface()
   *
   * InPort inquires available InPortProviders to the factory class
   * of InPortProvider, and publishes available interfaces to
   * others. Therefore, sub-classes of InPortProvider that provides
   * PROVIDED interface to InPort should register its factory to
   * InPortProviderFactory.
   *
   * RTC::InPortProviderFactory::instance().addFactory() would be
   * called with the following arguments.
   *
   * 1st arg: The name of provider. ex. "corba_cdr"
   * 2nd arg: Factory function. coil::Creator<B, T>
   * 3rd arg: Destruction function. coil::Destructor<B, T>
   *
   * The following example shows how to register factory function.
   * And it is also declared as a initialization function.
   *
   * <pre>
   * extern "C"
   * {
   *   void InPortCorbaCdrProviderInit(void)
   *   {
   *     RTC::InPortProviderFactory&
   *                         factory(RTC::InPortProviderFactory::instance());
   *     factory.addFactory("corba_cdr",
   *                        ::coil::Creator<::RTC::InPortProvider,
   *                                        ::RTC::InPortCorbaCdrProvider>,
   *                        ::coil::Destructor<::RTC::InPortProvider,
   *                                           ::RTC::InPortCorbaCdrProvider>);
   *   }
   * };
   * </pre>
   *
   * It is recommended that the registration process is declared as a
   * initialization function with "extern C" to be accessed from the
   * outside of module.  If the InPortProviders are compiled as a
   * shared object or DLL for dynamic loading, new InPortProvider
   * types can be added dynamically.
   *
   * @since 0.4.0
   *
   * @endif
   */
  class InPortProvider
    : public DataPortStatus
  {
  public:
    DATAPORTSTATUS_ENUM
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *
     * @else
     * @brief Constructor
     *
     * Constructor
     *
     * @endif
     */
    InPortProvider();
    
    /*!
     * @if jp
     * @brief ǥȥ饯
     *
     * ۥǥȥ饯
     *
     * @else
     * @brief Destructor
     *
     * Virtual destructor
     *
     * @endif
     */
    virtual ~InPortProvider(void);

    /*!
     * @if jp
     * @brief 
     *
     * OutPortProvider γƼԤ饹ǤϡͿ줿
     * PropertiesɬפʾƳƼԤ init() 
     * ϡOutPortProviderľ太ӡ³ˤ줾ƤФ
     * ǽ롣äơδؿʣƤФ뤳ȤꤷƵ
     * Ҥ٤Ǥ롣
     * 
     * @param prop 
     *
     * @else
     *
     * @brief Initializing configuration
     *
     * This operation would be called to configure in initialization.
     * In the concrete class, configuration should be performed
     * getting appropriate information from the given Properties data.
     * This function might be called right after instantiation and
     * connection sequence respectivly.  Therefore, this function
     * should be implemented assuming multiple call.
     *
     * @param prop Configuration information
     *
     * @endif
     */
    virtual void init(coil::Properties& prop) = 0;

    /*!
     * @if jp
     * @brief Хåե򥻥åȤ
     *
     * OutPortProviderǡФХåե򥻥åȤ롣
     * Ǥ˥åȤ줿Хåե硢ΥХåեؤ
     * ݥ󥿤Фƾ񤭤롣
     * OutPortProviderϥХåեνͭꤷƤʤΤǡ
     * Хåեκϥ桼ǤǹԤʤФʤʤ
     *
     * @param buffer OutPortProviderǡФХåեؤΥݥ
     *
     * @else
     * @brief Setting outside buffer's pointer
     *
     * A pointer to a buffer from which OutPortProvider retrieve data.
     * If already buffer is set, previous buffer's pointer will be
     * overwritten by the given pointer to a buffer.  Since
     * OutPortProvider does not assume ownership of the buffer
     * pointer, destructor of the buffer should be done by user.
     * 
     * @param buffer A pointer to a data buffer to be used by OutPortProvider
     *
     * @endif
     */
    virtual void setBuffer(BufferBase<cdrMemoryStream>* buffer) = 0;

    /*!
     * @if jp
     * @brief ꥹʤꤹ롣
     *
     * OutPort ϥǡˤƼ磻٥ȤФΥꥹ
     * ֥Ȥ򥳡뤹륳Хå󶡤롣ܺ٤
     * ConnectorListener.h  ConnectorDataListener, ConnectorListener
     * 򻲾ȤΤȡOutPortProvider Υ֥饹ǤϡͿ줿ꥹ
     * ʤŬڤʥߥ󥰤ǸƤӽФ٤Ǥ롣٤ƤΥꥹ
     * ƤӽФɬפϤʤ
     *
     * @param info ³
     * @param listeners ꥹʥ֥
     *
     * @else
     * @brief Set the listener. 
     *
     * OutPort provides callback functionality that calls specific
     * listener objects according to the events in the data publishing
     * process. For details, see documentation of
     * ConnectorDataListener class and ConnectorListener class in
     * ConnectorListener.h. In the sub-classes of OutPortProvider, the
     * given listeners should be called in the proper timing. However,
     * it is not necessary to call all the listeners.
     *
     * @param info Connector information
     * @param listeners Listener objects
     *
     * @endif
     */
    virtual void setListener(ConnectorInfo& info,
                             ConnectorListeners* listeners) = 0;

    /*!
     * @if jp
     * @brief Connectorꤹ롣
     *
     * OutPort ³Ω OutPortConnector ֥Ȥ
     * ֥ȤΥݥ󥿤ȶˤδؿƤӽФͭ
     * OutPort ݻΤ OutPortProvider  OutPortConnector 
     * ƤϤʤ
     *
     * @param connector OutPortConnector
     *
     * @else
     * @brief set Connector
     *
     * OutPort creates OutPortConnector object when it establishes
     * connection between OutPort and InPort, and it calls this
     * function with a pointer to the connector object. Since the
     * OutPort has the ownership of this connector, OutPortProvider
     * should not delete it.
     *
     * @param connector OutPortConnector
     *
     * @endif
     */
    virtual void setConnector(InPortConnector* connector) = 0;

    /*!
     * @if jp
     * @brief InterfaceProfile
     *
     * InterfaceProfile롣
     * ǻꤹץѥƥ NameValue ֥Ȥ
     * dataport.interface_type ͤĴ١ݡȤꤵƤ
     * 󥿡եפȰפΤ߾롣
     *
     * @param properties InterfaceProfileץѥƥ
     *
     * @else
     * @brief Publish InterfaceProfile information
     *
     * Publish interfaceProfile information.
     * Check the dataport.interface_type value of the NameValue object 
     * specified by an argument in property information and get information
     * only when the interface type of the specified port is matched.
     *
     * @param properties Properties to get InterfaceProfile information
     *
     * @endif
     */
    virtual void publishInterfaceProfile(SDOPackage::NVList& properties);
    
    /*!
     * @if jp
     * @brief Interface
     *
     * Interface롣ǻꤹץѥƥ
     * NameValue ֥Ȥdataport.interface_type ͤĴ١ݡ
     * ȤꤵƤʤNameValue ˾ɲä롣ǤƱ쥤
     * 󥿡եϿѤߤξϲԤʤ
     *
     * @param properties Interfaceץѥƥ
     * @return true: ｪλ
     *
     * @else
     * @brief Publish interface information
     *
     * Publish interface information.  Check the
     * dataport.interface_type value of the NameValue object specified
     * by an argument in the property information, and add the
     * information to the NameValue if this port is not specified.
     * This does not do anything if the same interface is already
     * subscribed.
     *
     * @param properties Properties to receive interface information
     * @return true: normal return
     *
     * @endif
     */
    virtual bool publishInterface(SDOPackage::NVList& properties);
    
  protected:
    /*!
     * @if jp
     * @brief 󥿥եפꤹ
     *
     * 󥿥եפꤹ롣
     *
     * @param interface_type оݥ󥿥ե
     *
     * @else
     * @brief Set the interface type
     *
     * Set the interface type.
     *
     * @param interface_type The target interface type for set
     *
     * @endif
     */
    void setInterfaceType(const char* interface_type);
    
    /*!
     * @if jp
     * @brief ǡեפꤹ
     *
     * ǡեפꤹ롣
     *
     * @param dataflow_type оݥǡե
     *
     * @else
     * @brief Set the dataflow type
     *
     * Set the dataflow type.
     *
     * @param dataflow_type The target data flow type for set
     *
     * @endif
     */
    void setDataFlowType(const char* dataflow_type);
    
    /*!
     * @if jp
     * @brief ֥ץ󥿥פꤹ
     *
     * ֥ץ󥿥פꤹ롣
     *
     * @param subs_type оݥ֥ץ󥿥
     *
     * @else
     * @brief Set the subscription type
     *
     * Set the subscription type.
     *
     * @param subs_type The target subscription type for set
     *
     * @endif
     */
    void setSubscriptionType(const char* subs_type);


  protected:
    /*!
     * @if jp
     * @brief ݡȥץեݻץѥƥ
     * @else
     * @brief Properties to hold port profile
     * @endif
     */
    SDOPackage::NVList m_properties;

    /*!
     * @if jp
     * @brief ȥ꡼
     * @else
     * @brief Logger stream
     * @endif
     */
    mutable Logger rtclog;

  private:
    std::string m_interfaceType;
    std::string m_dataflowType;
    std::string m_subscriptionType;

  public:
    // functors
    /*!
     * @if jp
     * @brief 󥿡եץե뤿Υե󥯥
     * @else
     * @brief Functor to publish interface profile
     * @endif
     */
    struct publishInterfaceProfileFunc
    {
      publishInterfaceProfileFunc(SDOPackage::NVList& prop) : m_prop(prop) {}
      void operator()(InPortProvider* provider)
      {
	provider->publishInterfaceProfile(m_prop);
      }
      SDOPackage::NVList& m_prop;
    };

    /*!
     * @if jp
     * @brief 󥿡եץե뤿Υե󥯥
     * @else
     * @brief Functor to publish interface profile
     * @endif
     */
    struct publishInterfaceFunc
    {
      publishInterfaceFunc(SDOPackage::NVList& prop)
        : m_prop(prop), provider_(0) {}
      void operator()(InPortProvider* provider)
      {
	if (provider->publishInterface(m_prop))
          {
            provider_ = provider;
          }
      }
      SDOPackage::NVList& m_prop;
      InPortProvider* provider_;
    };


  };

  /*!
   * @if jp
   * @brief InPortProviderFactory
   * @else
   * @brief InPortProviderFactory type definition
   * @endif
   */
  typedef ::coil::GlobalFactory<InPortProvider> InPortProviderFactory;

#if defined(WIN32) || defined(_WIN32) || defined(__WIN32__) || defined(__NT__)
  EXTERN template class DLL_PLUGIN ::coil::GlobalFactory<InPortProvider>;
#endif
};     // namespace RTC
#endif // RTC_INPORTPROVIDER_H
