// -*- C++ -*-
/*!
 * @file PortConnectListener.h
 * @brief port's internal action listener classes
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2011
 *     Noriaki Ando
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef RTC_PORTCONNECTLISTENER_H
#define RTC_PORTCONNECTLISTENER_H

#include <vector>
#include <utility>
#include <coil/Mutex.h>
#include <coil/Guard.h>
#include <rtm/RTC.h>
#include <rtm/idl/RTCSkel.h>

namespace RTC
{
  //============================================================
  /*!
   * @if jp
   * @brief PortConnectListener Υ
   *
   * - ON_NOTIFY_CONNECT:         notify_connect() ؿƤӽФľ
   * - ON_NOTIFY_DISCONNECT:      notify_disconnect() ƤӽФľ
   * - ON_UNSUBSCRIBE_INTERFACES: notify_disconnect() IFɲ
   *
   * @else
   * @brief The types of ConnectorDataListener
   * 
   * - ON_NOTIFY_CONNECT:         right after entering into notify_connect()
   * - ON_NOTIFY_DISCONNECT:      right after entering into notify_disconnect()
   * - ON_UNSUBSCRIBE_INTERFACES: unsubscribing IF in notify_disconnect()
   *
   * @endif
   */
  enum PortConnectListenerType
    {
      ON_NOTIFY_CONNECT,
      ON_NOTIFY_DISCONNECT,
      ON_UNSUBSCRIBE_INTERFACES,
      PORT_CONNECT_LISTENER_NUM
    };

  /*!
   * @if jp
   * @class PortConnectListener 饹
   * @brief PortConnectListener 饹
   *
   * ƥб桼ɤƤФľΥߥ
   * ǥ뤵ꥹʥ饹δ쥯饹
   *
   * - ON_NOTIFY_CONNECT:         notify_connect() ؿƤӽФľ
   * - ON_NOTIFY_DISCONNECT:      notify_disconnect() ƤӽФľ
   * - ON_UNSUBSCRIBE_INTERFACES: notify_disconnect() IFɲ
   *
   * @else
   * @class PortConnectListener class
   * @brief PortConnectListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events in rtobject.
   *
   * - ON_NOTIFY_CONNECT:         right after entering into notify_connect()
   * - ON_NOTIFY_DISCONNECT:      right after entering into notify_disconnect()
   * - ON_UNSUBSCRIBE_INTERFACES: unsubscribing IF in notify_disconnect()
   *
   * @endif
   */
  class PortConnectListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief PortConnectListenerType ʸѴ
     *
     * PortConnectListenerType ʸѴ
     *
     * @param type Ѵо PortConnectListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert PortConnectListenerType into the string.
     *
     * Convert PortConnectListenerType into the string.
     *
     * @param type The target PortConnectListenerType for transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(PortConnectListenerType type);

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PortConnectListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * PortConnectListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for PortConnectListener.
     *
     * @endif
     */
    virtual void operator()(const char* portname,
                            RTC::ConnectorProfile& profile) = 0;
  };


  //============================================================
  /*!
   * @if jp
   * @brief PortConnectRetListenerType Υ
   *
   * - ON_CONNECT_NEXTPORT:     notify_connect() ΥɸƤӽФľ
   * - ON_SUBSCRIBE_INTERFACES: notify_connect() Υ󥿡եľ
   * - ON_CONNECTED:            nofity_connect() ³λ˸ƤӽФ
   * - ON_DISCONNECT_NEXT:      notify_disconnect() ˥ɸƤӽФľ
   * - ON_DISCONNECTED:         notify_disconnect() ꥿
   *
   * @else
   * @brief The types of PortConnectRetListenerType
   * 
   * - ON_CONNECT_NEXTPORT:     after cascade-call in notify_connect()
   * - ON_SUBSCRIBE_INTERFACES: after IF subscribing in notify_connect()
   * - ON_CONNECTED:            completed nofity_connect() connection process
   * - ON_DISCONNECT_NEXT:      after cascade-call in notify_disconnect()
   * - ON_DISCONNECTED:         completed notify_disconnect() disconnection
   *
   * @endif
   */
  enum PortConnectRetListenerType
    {
      ON_PUBLISH_INTERFACES,
      ON_CONNECT_NEXTPORT,
      ON_SUBSCRIBE_INTERFACES,
      ON_CONNECTED,
      ON_DISCONNECT_NEXT,
      ON_DISCONNECTED,
      PORT_CONNECT_RET_LISTENER_NUM
    };

   /*!
   * @if jp
   * @class PortConnectRetListener 饹
   * @brief PortConnectRetListener 饹
   *
   * ƥб桼ɤƤФľΥߥ
   * ǥ뤵ꥹʥ饹δ쥯饹
   *
   * - ON_PUBLISH_INTERFACES:   notify_connect() Υ󥿡եľ
   * - ON_CONNECT_NEXTPORT:     notify_connect() ΥɸƤӽФľ
   * - ON_SUBSCRIBE_INTERFACES: notify_connect() Υ󥿡եľ
   * - ON_CONNECTED:            nofity_connect() ³λ˸ƤӽФ
   * - ON_DISCONNECT_NEXT:      notify_disconnect() ˥ɸƤӽФľ
   * - ON_DISCONNECTED:         notify_disconnect() ꥿
   *
   * @else
   * @class PortConnectRetListener class
   * @brief PortConnectRetListener class
   *
   * This class is abstract base class for listener classes that
   * provides callbacks for various events in rtobject.
   *
   * - ON_CONNECT_NEXTPORT:     after cascade-call in notify_connect()
   * - ON_SUBSCRIBE_INTERFACES: after IF subscribing in notify_connect()
   * - ON_CONNECTED:            completed nofity_connect() connection process
   * - ON_DISCONNECT_NEXT:      after cascade-call in notify_disconnect()
   * - ON_DISCONNECTED:         completed notify_disconnect() disconnection
   *
   * @endif
   */
  class PortConnectRetListener
  {
  public:
    /*!
     * @if jp
     *
     * @brief PortConnectRetListenerType ʸѴ
     *
     * PortConnectRetListenerType ʸѴ
     *
     * @param type Ѵо PortConnectRetListenerType
     *
     * @return ʸѴ
     *
     * @else
     *
     * @brief Convert PortConnectRetListenerType into string.
     *
     * Convert PortConnectRetListenerType into string.
     *
     * @param type The target PortConnectRetListenerType for transformation
     *
     * @return Trnasformation result of string representation
     *
     * @endif
     */
    static const char* toString(PortConnectRetListenerType type);

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PortConnectRetListener();

    /*!
     * @if jp
     *
     * @brief ۥХåؿ
     *
     * PortConnectRetListener ΥХåؿ
     *
     * @else
     *
     * @brief Virtual Callback function
     *
     * This is a the Callback function for PortConnectRetListener.
     *
     * @endif
     */
    virtual void operator()(const char* portname,
                            RTC::ConnectorProfile& profile,
                            ReturnCode_t ret) = 0;
  };

  //============================================================
  /*!
   * @if jp
   * @class PortConnectListenerHolder 
   * @brief PortConnectListener ۥ饹
   *
   * ʣ PortConnectListener ݻ륯饹
   *
   * @else
   * @class PortConnectListenerHolder
   * @brief PortConnectListener holder class
   *
   * This class manages one ore more instances of
   * PortConnectListener class.
   *
   * @endif
   */
  class PortConnectListenerHolder
  {
    typedef std::pair<PortConnectListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    PortConnectListenerHolder();
    
    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PortConnectListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(PortConnectListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(PortConnectListener* listener);

    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @endif
     */
    void notify(const char* portname, RTC::ConnectorProfile& profile);
      
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };


  /*!
   * @if jp
   * @class PortConnectRetListenerHolder
   * @brief PortConnectRetListener ۥ饹
   *
   * ʣ PortConnectRetListener ݻ륯饹
   *
   * @else
   * @class PortConnectRetListenerHolder
   * @brief PortConnectRetListener holder class
   *
   * This class manages one ore more instances of
   * PortConnectRetListener class.
   *
   * @endif
   */
  class PortConnectRetListenerHolder
  {
    typedef std::pair<PortConnectRetListener*, bool> Entry;
    typedef coil::Guard<coil::Mutex> Guard;
  public:
    /*!
     * @if jp
     * @brief 󥹥ȥ饯
     * @else
     * @brief Constructor
     * @endif
     */
    PortConnectRetListenerHolder();

    /*!
     * @if jp
     * @brief ǥȥ饯
     * @else
     * @brief Destructor
     * @endif
     */
    virtual ~PortConnectRetListenerHolder();
    
    /*!
     * @if jp
     *
     * @brief ꥹʡɲ
     *
     * ꥹʡɲä롣
     *
     * @param listener ɲäꥹ
     * @param autoclean true:ǥȥ饯Ǻ,
     *                  false:ǥȥ饯Ǻʤ
     * @else
     *
     * @brief Add the listener.
     *
     * This method adds the listener. 
     *
     * @param listener Added listener
     * @param autoclean true:The listener is deleted at the destructor.,
     *                  false:The listener is not deleted at the destructor. 
     * @endif
     */
    void addListener(PortConnectRetListener* listener, bool autoclean);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡκ
     *
     * ꥹʤ롣
     *
     * @param listener ꥹ
     * @else
     *
     * @brief Remove the listener. 
     *
     * This method removes the listener. 
     *
     * @param listener Removed listener
     * @endif
     */
    void removeListener(PortConnectRetListener* listener);
    
    /*!
     * @if jp
     *
     * @brief ꥹʡΤ
     *
     * ϿƤꥹʤΥХå᥽åɤƤӽФ
     *
     * @param info ConnectorInfo
     * @param cdrdata ǡ
     * @else
     *
     * @brief Notify listeners. 
     *
     * This calls the Callback method of the registered listener. 
     *
     * @param info ConnectorInfo
     * @param cdrdata Data
     * @endif
     */
    void notify(const char* portname, RTC::ConnectorProfile& profile,
                ReturnCode_t ret);
    
  private:
    std::vector<Entry> m_listeners;
    coil::Mutex m_mutex;
  };

  /*!
   * @if jp
   * @class PortConnectListeners
   * @brief PortConnectListeners 饹
   *
   *
   * @else
   * @class PortConnectListeners
   * @brief PortConnectListeners class
   *
   *
   * @endif
   */
  class PortConnectListeners
  {
  public:
    /*!
     * @if jp
     * @brief PortConnectListenerType ꥹ
     * PortConnectListenerType ꥹʤǼ
     * @else
     * @brief PortConnectListenerType listener array
     * The PortConnectListenerType listener is stored. 
     * @endif
     */
    PortConnectListenerHolder 
    portconnect_[PORT_CONNECT_LISTENER_NUM];
    /*!
     * @if jp
     * @brief PortConnectRetTypeꥹ
     * PortConnectRetTypeꥹʤǼ
     * @else
     * @brief PortConnectRetType listener array
     * The PortConnectRetType listener is stored.
     * @endif
     */
    PortConnectRetListenerHolder 
    portconnret_[PORT_CONNECT_RET_LISTENER_NUM];
  };


}; // namespace RTC

#endif // RTC_PORTCONNECTLISTENER_H
