// -*- C++ -*-
/*!
 * @file SdoServiceAdmin.h
 * @brief SDO service administration class
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2011
 *     Noriaki Ando
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id: SdoConfiguration.cpp 1971 2010-06-03 08:46:40Z n-ando $
 *
 */

#ifndef RTC_SDOSERVICEADMIN_H
#define RTC_SDOSERVICEADMIN_H

#include <coil/Mutex.h>
#include <coil/Factory.h>

#include <rtm/idl/SDOPackageStub.h>
#include <rtm/SystemLogger.h>

namespace RTC
{
  class RTObject_impl;
  class SdoServiceProviderBase;
  class SdoServiceConsumerBase;

  /*!
   * @if jp
   *
   * @class SDO service administration class
   * @brief SDO service 饹
   *
   * Υ饹ϡSDO Service 뤿Υ饹Ǥ롣SDO
   * Service  OMG SDO Specification ˤƤ롢SDO
   * εǽΤ󶡤ޤ׵᤹륵ӥΰĤǤ롣ܺ٤ϻͤˤ
   * Ƥʤܥ饹ǤϰʲΤ褦˿񤦥ӥ
   * ΤȤ뤿Υ饹ܥ饹Ǥ롣
   *
   * SDO Service ˤƤϡSDO/RTC˽ͭ졢Υӥ
   * Τ SDO Service Provider¾SDO/RTC䥢ץꥱ
   * 륵ӥ֥ȤλȤꡢεǽѤ
   * ΤSDO Service Consumer ȸƤ֡
   *
   * SDO Service Provider ¾Υץꥱ󤫤ƤФ졢SDO/RTC
   * εǽ˥뤿Ѥ롣¾SDO/RTCޤϥץꥱ
   * ϡ
   *
   * - SDO::get_service_profiles ()
   * - SDO::get_service_profile (in UniqueIdentifier id)
   * - SDO::get_sdo_service (in UniqueIdentifier id) 
   *
   * Τ줫Υڥ졼ˤꡢServiceProfile ޤ SDO
   * Service λȤǽѤ뤿Υڥ졼Ƥӽ
   * ¾SDO/RTCޤϥץꥱǤλȤ˴ǤդΥ
   * ߥ󥰤ǹԤ졢ӥ¦ǤϡɤɤȤƤ
   * Τ뤳ȤϤǤʤǡSDO/RTC¦⡢ǤդΥߥ󥰤ǥ
   * ӥ󶡤ߤ뤳ȤǤ뤿ᡢӥ¦Ǥϡˤ
   * ӥѤǤȤϸ¤ʤΤȤƥӥڥ졼
   * ƤӽФɬפ롣
   *
   * SDO Service Consumer SDO/RTCʳSDO/RTCޤϥץ
   * 󤬥ӥμΤSDO/RTC˥֥ȻȤ
   * ޤץեͿ뤳ȤǡSDO/RTC¦饵ӥڥ졼
   * 󤬸ƤФ쳰SDO/RTCޤϥץꥱ󶡤뵡ǽ
   * Ǥ롣ޤ֥Ūʥ֥ȤͿ뤳ȤǡSDO/RTC¦
   * ΥХå¸뤿ˤѤ뤳ȤǤ롣󥷥塼
   * ޤϡץХȤϰۤʤꡢSDO Configuration󥿡ե
   * ɲáԤ롣Ϣ륪ڥ졼ϰʲΤȤǤ롣
   *
   * - Configuration::add_service_profile (in ServiceProfile sProfile)
   * - Configuration::remove_service_profile (in UniqueIdentifier id)
   *
   * SDO/RTCޤϥץꥱϡȤSDO Servcie
   * Provider λȤIDinterface typeץѥƥȤȤ
   * ServcieProfile ˥åȤǡadd_service_profile() ΰ
   * Ϳ뤳ȤǡSDO/RTC˥ӥͿ롣κݡIDUUID
   * ʤɰդIDǤʤФʤʤޤݤˤIDˤоݤ
   * ServiceProfileõ뤿ᡢӥ¦ǤϺޤID
   * ݻƤʤФʤʤ
   *
   *
   * @since 1.1.0
   *
   * @else
   *
   * @class SDO service administration class
   * @brief SDO service administration class
   *
   * This class is the administration class for SDO Services. The SDO
   * Service, which is defined in the OMG SDO Specification, is a kind
   * of service for certain functionalities which is provided and/or
   * consumed by SDO. The specification does not define details.
   * However, in this implementation, the following behaviors of SDO
   * services are assumed and this class manages these SDO services.
   *
   * In this context, the SDO Services that are owned by SDO/RTC is
   * called SDO Service Provider, and the SDO Services that receive
   * references to provided services by other SDOs/RTCs or
   * applications is called SDO Serivce Consumer.
   *
   * SDO Service Provider is called from other applications, and it is
   * used to access to internal functionality of an SDO/RTC.  Other
   * SDO/RTC or applications would get a ServiceProfiles or a
   * reference to the SDO serivce through the following operations,
   * and then they call operations of the service.
   *
   * - SDO::get_service_profiles ()
   * - SDO::get_service_profile (in UniqueIdentifier id)
   * - SDO::get_sdo_service (in UniqueIdentifier id) 
   *
   * Since references of services in other SDOs/RTCs or applications
   * could be released anytime, service providers cannot know where
   * and how many consumers refer them. On the other hand, since
   * SDO/RTC which provides services can stop and delete them anytime,
   * consumers have to call operations by assuming that the reference
   * cannot be accessed always.
   *
   * SDO Service Consumer, which is a reference to the service entity
   * in the other SDOs/RTCs or other applications, is given with
   * ServiceProfile, and SDO/RTC would call operation to access some
   * functionality to the service object. And giving certain observer
   * object, it will work as callback from SDO/RTC. SDO service
   * consumer, which is defferent from SDO service provider, is added
   * and deleted through SDO Configuration interface shown as follows.
   *
   * - Configuration::add_service_profile (in ServiceProfile sProfile)
   * - Configuration::remove_service_profile (in UniqueIdentifier id)
   *
   * To set a SDO service to the target SDO/RTC, other SDOs/RTCs and
   * applications have to give their service references with
   * ServiceProfile including ID, interface type and properties to the
   * add_service_profile() operation.  The ID has to be unique such as
   * UUID.  Since ID is used when the service is removed from the
   * target SDO/RTC, SDOs/RTCs and applications of service provider
   * side have to keep the ID until removing the service.
   *
   * @since 1.1.0
   *
   * @endif
   */
  class SdoServiceAdmin
  {
  public:
    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     * 
     * 󥹥ȥ饯
     *
     * @param 
     * 
     * @else
     *
     * @brief Constructor
     * 
     * Constructor
     *
     * @param 
     *
     * @endif
     */
    SdoServiceAdmin(::RTC::RTObject_impl& rtobj);

    /*!
     * @if jp
     *
     * @brief ۥǥȥ饯
     * 
     * ۥǥȥ饯
     * 
     * @else
     *
     * @brief Virtual destractor
     *
     * Virtual destractor.
     *
     * @endif
     */
    virtual ~SdoServiceAdmin();
    
    /*!
     * @if jp
     *
     * @brief SDO Service Provider  ServiceProfileList 
     * 
     * @else
     *
     * @brief Get ServiceProfileList of SDO Service Provider
     *
     * @endif
     */
    SDOPackage::ServiceProfileList* getServiceProviderProfiles();

    /*!
     * @if jp
     *
     * @brief SDO Service Provider  ServiceProfile 
     *
     * id ǻꤵ줿IFR IDSDO Service Provider 
     * ServiceProfile 롣id  NULL ݥ󥿤ξ硢ꤵ줿
     * id ˳ServiceProfile ¸ߤʤ硢InvalidParameter
     * 㳰Ф롣
     *
     * @param id SDO Service provider  IFR ID
     * @return ꤵ줿 id  ServiceProfile
     * 
     * @else
     *
     * @brief Get ServiceProfile of an SDO Service Provider
     *
     * This operation returnes ServiceProfile of an SDO Service
     * Provider which has the specified id. If the specified id is
     * NULL pointer or the specified id does not exist in the
     * ServiceProfile list, InvalidParameter exception will be thrown.
     *
     * @param id IFR ID of an SDO Service provider
     * @return ServiceProfile which has the specified id
     *
     * @endif
     */
    SDOPackage::ServiceProfile* getServiceProviderProfile(const char* id);

    /*!
     * @if jp
     *
     * @brief SDO Service Provider  Service 
     *
     * id ǻꤵ줿IFR IDSDO Service Υ֥ȥե
     *  롣id  NULL ݥ󥿤ξ硢ꤵ줿 id ˳
     * ServiceProfile ¸ߤʤ硢InvalidParameter 㳰Ф
     * 롣
     *
     * @param id SDO Service provider  IFR ID
     * @return ꤵ줿 id  SDO Service Υ֥ȥե
     * 
     * @else
     *
     * @brief Get ServiceProfile of an SDO Service
     *
     * This operation returnes an object reference of an SDO Service
     * Provider which has the specified id. If the specified id is
     * NULL pointer or the specified id does not exist in the
     * ServiceProfile list, InvalidParameter exception will be thrown.
     *
     * @param id IFR ID of an SDO Service provider
     * @return an SDO Service reference which has the specified id
     *
     * @endif
     */
    SDOPackage::SDOService_ptr getServiceProvider(const char* id);

    /*!
     * @if jp
     * @brief SDO service provider 򥻥åȤ
     * @else
     * @brief Set a SDO service provider
     * @endif
     */
    bool addSdoServiceProvider(const SDOPackage::ServiceProfile& prof,
                               SdoServiceProviderBase* provider);

    /*!
     * @if jp
     * @brief SDO service provider 
     * @else
     * @brief Remove a SDO service provider
     * @endif
     */
    bool removeSdoServiceProvider(const char* id);

    /*!
     * @if jp
     *
     * @brief Service Consumer ɲä
     * 
     * @else
     *
     * @brief Add Service Consumer
     *
     * @endif
     */
    bool addSdoServiceConsumer(const SDOPackage::ServiceProfile& sProfile);
    
    /*!
     * @if jp
     *
     * @brief Service Consumer 
     * 
     * @else
     *
     * @brief Remove Service Consumer
     *
     * @endif
     */
    bool removeSdoServiceConsumer(const char* id);
    
protected:
    /*!
     * @if jp
     *
     * @brief Ĥ줿ӥɤĴ٤
     * 
     * @else
     *
     * @brief If it is enabled service type
     *
     * @endif
     */
    bool isEnabledConsumerType(const SDOPackage::ServiceProfile& sProfile);

    /*!
     * @if jp
     *
     * @brief ¸ߤ륵ӥɤĴ٤
     * 
     * @else
     *
     * @brief If it is existing service type
     *
     * @endif
     */
    bool isExistingConsumerType(const SDOPackage::ServiceProfile& sProfile);

    const std::string getUUID() const;
    
    std::string ifrToKey(std::string& ifr);


  private:
    RTC::RTObject_impl& m_rtobj;
    coil::vstring m_consumerTypes;
    bool m_allConsumerEnabled;
    
    /*!
     * @if jp
     * @brief Lock դ SDO ServiceProfileList
     * @else
     * @brief SDO ServiceProfileList with mutex lock
     * @endif
     */
    std::vector<SdoServiceProviderBase*> m_providers;
    coil::Mutex m_provider_mutex;
    
    /*!
     * @if jp
     * @brief Lock դ SDO ServiceProfileList
     * @else
     * @brief SDO ServiceProfileList with mutex lock
     * @endif
     */
    std::vector<SdoServiceConsumerBase*> m_consumers;
    coil::Mutex m_consumer_mutex;

    /*!
     * @if jp
     * @brief logger
     * @else
     * @brief logger
     * @endif
     */
    ::RTC::Logger rtclog;
  };


};

#endif // RTC_SDOSERVICEADMIN_H
