#ifndef URG_SENSOR_H
#define URG_SENSOR_H

/*!
  \file
  \~japanese
  \brief URG ZT

  URG p̊{IȊ֐񋟂܂B


  \~english
  \brief URG sensor

  URG p̊{IȊ֐񋟂܂B

  \~
  \author Satofumi KAMIMURA

  $Id: urg_sensor.h,v 540bc11f70c8 2011/05/08 23:04:49 satofumi $
*/

#ifdef __cplusplus
extern "C" {
#endif

#include "urg_c/urg_connection.h"
#include "urg_c/urg_time.h"


    /*!
      \~japanese
      \brief v^Cv
    */
    typedef enum {
        URG_DISTANCE,           /*!< \~japanese  */
        URG_DISTANCE_INTENSITY, /*!< \~japanese  + x */
        URG_MULTIECHO,          /*!< \~japanese }`GR[̋ */
        URG_MULTIECHO_INTENSITY, /*!< \~japanese }`GR[( + x) */
        URG_STOP,                /*!< \~japanese v̒~ */
        URG_UNKNOWN,             /*!< \~japanese s */
    } urg_measurement_type_t;

    /*!
      \~japanese
      \brief  byte ŕ\邩̎w
    */
    typedef enum {
        URG_COMMUNICATION_3_BYTE, /*!< \~japanese  3 byte ŕ\ */
        URG_COMMUNICATION_2_BYTE, /*!< \~japanese  2 byte ŕ\ */
    } urg_range_data_byte_t;


    enum {
        URG_SCAN_INFINITY = 0,  /*!< \~japanese ̃f[^擾 */
        URG_MAX_ECHO = 3, /*!< \~japanese }`GR[̍őGR[ */
    };


    /*! \~japanese G[nh \~english error handler */
    typedef urg_measurement_type_t
    (*urg_error_handler)(const char *status, void *urg);


    /*!
      \~japanese
      \brief URG ZTǗ

      \~english
      \brief URG sensor
    */
    typedef struct
    {
        int is_active;
        int last_errno;
        urg_connection_t connection;

        int first_data_index;
        int last_data_index;
        int front_data_index;
        int area_resolution;
        long scan_usec;
        int min_distance;
        int max_distance;
        int scanning_first_step;
        int scanning_last_step;
        int scanning_skip_step;
        int scanning_skip_scan;
        urg_range_data_byte_t range_data_byte;

        int timeout;
        int specified_scan_times;
        int scanning_remain_times;
        int is_laser_on;

        int received_first_index;
        int received_last_index;
        int received_skip_step;
        urg_range_data_byte_t received_range_data_byte;
        int is_sending;

        urg_error_handler error_handler;

        char return_buffer[80];
    } urg_t;


    /*!
      \~japanese
      \brief ڑ

      w肵foCXɐڑAvł悤ɂB

      \param[in,out] urg URG ZTǗ
      \param[in] connection_type ʐM^Cv
      \param[in] device_or_address ڑfoCX
      \param[in] baudrate_or_port ڑ{[[g [bps] / TCP/IP |[g

      \retval 0 
      \retval <0 G[

      connection_type ɂ́Aȉ̍ڂwł܂B

      - #URG_SERIAL
      - VAAUSB ڑ

      - #URG_ETHERNET
      - C[T[lbgڑ

      Example
      \code
      urg_t urg;

      if (urg_open(&urg, URG_SERIAL, "/dev/ttyACM0", 115200) < 0) {
      return 1;
      }

      ...

      urg_close(&urg); \endcode

      \attention URG C Cȗ̊֐ĂяoOɁÅ֐ĂяoKv܂B

      \~
      \see urg_close()
    */
    extern int urg_open(urg_t *urg, urg_connection_type_t connection_type,
                        const char *device_or_address,
                        long baudrate_or_port);


    /*!
      \~japanese
      \brief ؒf

      [UAURG Ƃ̐ڑؒf܂B

      \param[in,out] urg URG ZTǗ

      \~
      \see urg_open()
    */
    extern void urg_close(urg_t *urg);


    /*!
      \brief ^CAEgԂ̐ݒ

      \param[in,out] urg URG ZTǗ
      \param[in] msec ^CAEg鎞 [msec]

      \attention urg_open() Ăяo timeout ̐ݒl̓ftHglɏ邽߁Å֐ urg_open() ɌĂяoƁB
    */
    extern void urg_set_timeout_msec(urg_t *urg, int msec);


    /*! \~japanese ^CX^v[h̊Jn */
    extern int urg_start_time_stamp_mode(urg_t *urg);


    /*!
      \~japanese
      \brief ^CX^v̎擾

      \param[in,out] urg URG ZTǗ

      \retval >=0 ^CX^v [msec]
      \retval <0 G[

      Example
      \code
      urg_start_time_stamp_mode(&urg);

      before_ticks = get_pc_msec_function();
      time_stamp = urg_time_stamp(&urg);
      after_ticks = get_pc_msec_function();

      // ^CX^vɂĂ̌vZ
      ...

      urg_stop_time_stamp_mode(&urg); \endcode

      ڂ \ref sync_time_stamp.c QƂĉB
    */
    extern long urg_time_stamp(urg_t *urg);


    /*! \~japanese ^CX^v[h̏I */
    extern int urg_stop_time_stamp_mode(urg_t *urg);


    /*!
      \~japanese
      \brief f[^̎擾Jn

      f[^̎擾Jn܂Bۂ̃f[^ urg_get_distance(), urg_get_distance_intensity(), urg_get_multiecho(), urg_get_multiecho_intensity() Ŏ擾ł܂B

      \param[in,out] urg URG ZTǗ
      \param[in] type f[^E^Cv
      \param[in] scan_times f[^̎擾
      \param[in] skip_scan f[^̎擾Ԋu

      \retval 0 
      \retval <0 G[

      type ɂ͎擾f[^̎ނw肵܂B

      - #URG_DISTANCE ... f[^
      - #URG_DISTANCE_INTENSITY ... f[^Ƌxf[^
      - #URG_MULTIECHO ... }`GR[ł̋f[^
      - #URG_MULTIECHO_INTENSITY ... }`GR[ł(f[^Ƌxf[^)

      scan_times ͉̃f[^擾邩 0 ȏ̐Ŏw肵܂BA0 ܂ #URG_SCAN_INFINITY w肵ꍇ́Ãf[^擾܂B\n
      Jnv𒆒fɂ urg_stop_measurement() g܂B

      skip_scan ̓~[̉]̂AP̃XLɉXLȂw肵܂Bskip_scan Ɏwł͈͂ [0, 9] łB

      \image html skip_scan_image.png ɂP񂾂v邩

      Ƃ΁A~[̂P] 100 [msec] ̃ZT skip_scan  1 w肵ꍇAf[^̎擾Ԋu 200 [msec] ɂȂ܂B

      Example
      \code
      enum { CAPTURE_TIMES = 10 };
      urg_start_measurement(&urg, URG_DISTANCE, CAPTURE_TIMES, 0);

      for (i = 0; i < CAPTURE_TIMES; ++i) {
      int n = urg_get_distance(&urg, data, &time_stamp);

      // Mf[^̗p
      ...
      } \endcode

      \~
      \see urg_get_distance(), urg_get_distance_intensity(), urg_get_multiecho(), urg_get_multiecho_intensity(), urg_stop_measurement()
    */
    extern int urg_start_measurement(urg_t *urg, urg_measurement_type_t type,
                                     int scan_times, int skip_scan);


    /*!
      \~japanese
      \brief f[^̎擾

      ZT狗f[^擾܂BO urg_start_measurement()  #URG_DISTANCE wŌĂяoĂKv܂B

      \param[in,out] urg URG ZTǗ
      \param[out] data f[^ [mm]
      \param[out] time_stamp ^CX^v [msec]

      \retval >=0 Mf[^
      \retval <0 G[

      data ɂ́AZT擾f[^i[܂Bdata ̓f[^i[̃TCYmۂĂKv܂Bdata Ɋi[f[^ urg_max_data_size() Ŏ擾ł܂B

      time_stamp ɂ́AZT̃^CX^vi[܂Btime_stamp 擾Ȃꍇ NULL w肵ĉB

      Example
      \code
      long *data = (long*)malloc(urg_max_data_size(&urg) * sizeof(data[0]));

      ...

      // f[^̂ݎ擾
      urg_start_measurement(&urg, URG_DISTANCE, 1, 0);
      int n = urg_get_distance(&urg, data, NULL);

      ...

      // f[^ƃ^CX^v擾
      long time_stamp;
      urg_start_measurement(&urg, URG_DISTANCE, 1, 0);
      n = urg_get_distance(&urg, data, &time_stamp); \endcode

      \~
      \see urg_start_measurement(), urg_max_data_size()
    */
    extern int urg_get_distance(urg_t *urg, long data[], long *time_stamp, unsigned long long *system_time_stamp);


    /*!
      \~japanese
      \brief Ƌxf[^̎擾

      urg_get_distance() ɉAxf[^̎擾ł֐łBO urg_start_measurement()  #URG_DISTANCE_INTENSITY wŌĂяoĂKv܂B

      \param[in,out] urg URG ZTǗ
      \param[out] data f[^ [mm]
      \param[out] intensity xf[^
      \param[out] time_stamp ^CX^v [msec]

      \retval >=0 Mf[^
      \retval <0 G[

      xf[^Ƃ́AvZɎgg`̔ˋxłAZT̃V[YɓقȂ܂B xf[^gƂŁÂ̔˗̑܂ȔZW𐄑ł܂B

      data, time_stamp ɂĂ urg_get_distance() ƓłB

      intensity ɂ́AZT擾xf[^i[܂Bintensity ̓f[^i[̃TCYmۂĂKv܂Bintensity Ɋi[f[^ urg_max_data_size() Ŏ擾ł܂B

      \~
      Example
      \code
      int data_size = urg_max_data_size(&urg);
      long *data = malloc(data_size * sizeof(long));
      long *intensity = malloc(data_size * sizeof(unsigned short));

      ...

      urg_start_measurement(&urg, URG_DISTANCE_INTENSITY, 1, 0);
      int n = urg_get_distance_intensity(&urg, data, intesnity, NULLL); \endcode

      \~
      \see urg_start_measurement(), urg_max_data_size()
    */
    extern int urg_get_distance_intensity(urg_t *urg, long data[],
                                          unsigned short intensity[],
                                          long *time_stamp, unsigned long long *system_time_stamp);


    /*!
      \~japanese
      \brief f[^̎擾 (}`GR[)

      }`GR[ł̋f[^擾֐łBO urg_start_measurement()  #URG_MULTIECHO wŌĂяoĂKv܂B

      \param[in,out] urg URG ZTǗ
      \param[out] data_multi f[^ [mm]
      \param[out] time_stamp ^CX^v [msec]

      \retval >=0 Mf[^
      \retval <0 G[

      }`GR[Ƃ͕̋f[^łB }`GR[́AP̃[Uɂĕ̋f[^ꂽƂɓ܂B

      \image html multiecho_image.png }`GR[̃C[W}

      time_stamp ɂĂ urg_get_distance() ƓłB

      data_multi ɂ́AZT擾f[^P step ő #URG_MAX_ECHO (3 )i[܂B}`GR[݂Ȃڂ̃f[^l -1 i[Ă܂B

      \verbatim
      data_multi[0] ... step n ̋f[^ (1 )
      data_multi[1] ... step n ̋f[^ (2 )
      data_multi[2] ... step n ̋f[^ (3 )
      data_multi[3] ... step (n + 1)  f[^ (1 )
      data_multi[4] ... step (n + 1)  f[^ (2 )
      data_multi[5] ... step (n + 1)  f[^ (3 )
      ... \endverbatim

      i[́Ae step ɂ urg_get_distance() ̂ƂƓ̃f[^ (3n + 0) ̈ʒuɊi[AȊÕf[^ (3n + 1), (3n + 2) ̈ʒuɍ~Ɋi[܂B\n
      ܂ data_multi[3n + 1] >= data_multi[3n + 2] ɂȂ邱Ƃ͕ۏ؂܂ data_multi[3n + 0]  data_multi[3n + 1] ̊֌W͖`łB(data_multi[3n + 1] == data_multi[3n + 2] 藧̂̓f[^l -1 ̂ƂB)

      \~
      Example
      \code
      long *data_multi = malloc(3 * urg_max_data_size(&urg) * sizeof(long));

      ...

      urg_start_measurement(&urg, URG_MULTIECHO, 1, 0);
      int n = urg_get_distance_intensity(&urg, data_multi, NULLL); \endcode

      \~
      \see urg_start_measurement(), urg_max_data_size()
    */
    extern int urg_get_multiecho(urg_t *urg, long data_multi[], long *time_stamp, unsigned long long *system_time_stamp);


    /*!
      \~japanese
      \brief Ƌxf[^̎擾 (}`GR[)

      urg_get_multiecho() ɉAxf[^̎擾ł֐łBO urg_start_measurement()  #URG_MULTIECHO_INTENSITY wŌĂяoĂKv܂B

      \param[in,out] urg URG ZTǗ
      \param[out] data_multi f[^ [mm]
      \param[out] intensity_multi xf[^
      \param[out] time_stamp ^CX^v [msec]

      \retval >=0 Mf[^
      \retval <0 G[

      data_multi, time_stamp ɂĂ urg_get_multiecho() ƓłB

      intensity_multi ̃f[^̕т data_multi ƑΉ̂ɂȂ܂Bintensity_multi Ɋi[f[^ urg_max_data_size() Ŏ擾ł܂B

      \~
      Example
      \code
      int data_size = urg_max_data_size(&urg);
      long *data_multi = malloc(3 * data_size * sizeof(long));
      long *intensity_multi = malloc(3 * data_size * sizeof(unsigned short));

      ...

      urg_start_measurement(&urg, URG_DISTANCE_INTENSITY, 1, 0);
      int n = urg_get_multiecho_intensity(&urg, data_multi,
      intesnity_multi, NULLL); \endcode

      \~
      \see urg_start_measurement(), urg_max_data_size()
    */
    extern int urg_get_multiecho_intensity(urg_t *urg, long data_multi[],
                                           unsigned short intensity_multi[],
                                           long *time_stamp, unsigned long long *system_time_stamp);


    /*!
      \~japanese
      \brief v𒆒fA[U܂

      \ref urg_start_measurement() ̌v𒆒f܂B

      \param[in,out] urg URG ZTǗ

      \retval 0 
      \retval <0 G[

      \~
      Example
      \code
      urg_start_measurement(&urg, URG_DISTANCE, URG_SCAN_INFINITY, 0);
      for (int i = 0; i < 10; ++i) {
      urg_get_distance(&urg, data, NULL);
      }
      urg_stop_measurement(&urg); \endcode

      \~
      \see urg_start_measurement()
    */
    extern int urg_stop_measurement(urg_t *urg);


    /*!
      \~japanese
      \brief v͈͂ݒ肵܂

      ZTv͈͂ step lŎw肵܂Burg_get_distance() Ȃǂ̋f[^擾̊֐ŕԂf[^́AŎw肵͈͂Ő܂B

      \param[in,out] urg URG ZTǗ
      \param[in] first_step v̊Jn step
      \param[in] last_step v̏I step
      \param[in] skip_step vf[^O[sO

      \retval 0 
      \retval <0 G[

      ZT step ́AZTʂ 0 ƂAZT㕔猩Ĕv܂̌̒lƂȂ鏇ɊU܂B

      \image html sensor_angle_image.png ZT step ̊֌W

      step ̊ԊuƁAőlAŏl̓ZTˑłBstep l̍őlAŏl urg_step_min_max() Ŏ擾ł܂B\n

      first_step, last_step Ńf[^̌v͈͂w肵܂Bv͈͂ [first_step, last_step] ƂȂ܂B

      skip_step ́Avf[^O[sOw肵܂Bwłl [0, 99] łB\n
      skip_step ́Aw肳ꂽ̃f[^ 1 ɂ܂Ƃ߂邱ƂŁAZTMf[^ʂ炵A擾s֐̉߂ƂɎg܂BAf[^܂Ƃ߂邽߁Af[^̕\͌܂B

      ႦΈȉ̂悤ȋf[^ꍇ
      \verbatim
      100, 101, 102, 103, 104, 105, 106, 107, 108, 109
      \endverbatim

      skip_step  2 w肷ƁAf[^
      \verbatim
      \endverbatim

      f[^́A܂Ƃ߂f[^̂AԏȒl̃f[^p܂B

      \~
      Example
      \code
      urg_set_scanning_parameter(&urg, urg_deg2step(&urg, -45),
      urg_deg2step(&urg, +45), 1);
      urg_start_measurement(&urg, URG_DISTANCE, 0);
      int n = urg_get_distance(&urg, data, NULL);
      for (int i = 0; i < n; ++i) {
      printf("%d [mm], %d [deg]\n", data[i], urg_index2deg(&urg, i));
      } \endcode

      \~
      \see urg_step_min_max(), urg_rad2step(), urg_deg2step()
    */
    extern int urg_set_scanning_parameter(urg_t *urg, int first_step,
                                          int last_step, int skip_step);


    /*!
      \~japanese
      \brief ʐMf[^̃TCYύX

      f[^ZTM̍ۂ̃f[^TCYύX܂B

      \param[in,out] urg URG ZTǗ
      \param[in] data_byte l\f[^̃oCg

      \retval 0 
      \retval <0 G[

      data_byte ɂ

      - URG_COMMUNICATION_3_BYTE ...  3 byte ŕ\
      - URG_COMMUNICATION_2_BYTE ...  2 byte ŕ\

      wł܂B\n
      Ԃł͋ 3 byte ŕ\悤ɂȂĂ܂B̐ݒ 2 byte ɐݒ肷邱ƂŁAZTMf[^ 2/3 ɂȂ܂BA擾ł鋗̍ől 4095 ɂȂ邽߁AϑΏۂ 4 [m] ȓ͈̔͂ɑ݂ꍇ̂ݗpĉB
    */
    extern int urg_set_communication_data_size(urg_t *urg,
                                               urg_range_data_byte_t data_byte);


    /*! \~japanese [U𔭌 */
    extern int urg_laser_on(urg_t *urg);


    /*! \~japanese [U */
    extern int urg_laser_off(urg_t *urg);


    /*! \~japanese ZTċN */
    extern int urg_reboot(urg_t *urg);


    /*!
      \~japanese
      \brief ZTd͂̏ԂɑJڂ

      d͂̃[hł́AXLỉ]~vf܂B

      - d͂̃[h
        - [UČvfB
        - XLỉ]~B

      d͂̃[h甲邽߂ɂ \ref urg_wakeup() ֐ĂяoĉB

      \see urg_wakeup()
    */
    extern void urg_sleep(urg_t *urg);


    /*!
      \~japanese
      \brief ZTd͂̃[hʏ̏ԂɑJڂ

      \see urg_sleep()
    */
    extern void urg_wakeup(urg_t *urg);

    /*!
      \~japanese
      \brief ZTvłԂԂ

      \retval 1 ZTvłԂɂ
      \retval 0 ZTvłԂɂȂ

      NŃXLỉ]肵ĂȂꍇA炩̃G[ŌvłȂꍇÅ֐ 0 Ԃ܂B
    */
    extern int urg_is_stable(urg_t *urg);


    /*!
      \~japanese
      \brief ZT^𕶎ŕԂ

      ZŤ^𕶎ŕԂBԂ镶̓ZTˑƂȂB

      \param[in] urg URG ZTǗ

      \return ZT^̕
    */
    extern const char *urg_sensor_product_type(urg_t *urg);


    /*!
      \~japanese
      \brief ZT̃VA ID Ԃ

      ZT̃VA ID ԂBԂ镶̓ZTˑƂȂB

      \param[in] urg URG ZTǗ

      \return VA ID 
    */
    extern const char *urg_sensor_serial_id(urg_t *urg);

    /*!
      \brief returns the vendor name

      \param[in] URG

      \return The vendor name
    */
    extern const char *urg_sensor_vendor(urg_t *urg);


    /*!
      \~japanese
      \brief ZT̃o[WԂ

      ZT̃\tgEFAEo[WԂBԂ镶̓ZTˑƂȂB

      \param[in] urg URG ZTǗ

      \return o[W
    */
    extern const char *urg_sensor_firmware_version(urg_t *urg);

    extern const char *urg_sensor_firmware_date(urg_t *urg);

    /*!
      \brief returns the protocol version

      \param[in] URG

      \return The current protocol version
    */
    extern const char *urg_sensor_protocol_version(urg_t *urg);

    /*!
      \~japanese
      \brief ZT̃Xe[^XԂ

      ZT̃Xe[^XԂBԂ镶̓ZTˑƂȂB

      \param[in] urg URG ZTǗ
      \return Xe[^X
    */
    extern const char *urg_sensor_status(urg_t *urg);


    /*!
      \~japanese
      \brief ZT̏ԂԂ

      ZT̃Xe[^XԂBԂ镶̓ZTˑƂȂB

      \param[in] urg URG ZTǗ
      \return Ԃ

      \attention ԂɂĂ SCIP ̒ʐMdlQƂ̂ƁB
    */
    extern const char *urg_sensor_state(urg_t *urg);


    /*!
      \~japanese
      \brief vp̃G[nho^

      G[nh Gx, Mx ñR}h̉ "00"  "99" ȊÔƂɌĂяoB
    */
    extern void urg_set_error_handler(urg_t *urg, urg_error_handler handler);


    /*!
      \~japanese
      \brief SCIP ̃fR[hs

      \param[in] data SCIP 
      \param[in] data  byte TCY

      \retval fR[h̐l
    */
    extern long urg_scip_decode(const char data[], int size);


#ifdef __cplusplus
}
#endif

#endif /* !URG_SENSOR_H */
