#!/usr/bin/env python
# -*- coding: euc-jp -*-


##
# \file CorbaNaming.py
# \brief CORBA naming service helper class
# \author Noriaki Ando <n-ando@aist.go.jp> and Shinji Kurihara
#
# Copyright (C) 2006-2008
#     Noriaki Ando
#     Task-intelligence Research Group,
#     Intelligent Systems Research Institute,
#     National Institute of
#         Advanced Industrial Science and Technology (AIST), Japan
#     All rights reserved.

import omniORB.CORBA as CORBA
import CosNaming
import string
import sys
import traceback

##
# @if jp
# @class CorbaNaming
# @brief CORBA Naming Service إѡ饹
#
# Υ饹ϡCosNaming::NamingContext Фåѡ饹Ǥ롣
# CosNaming::NamingContext ĥڥ졼ȤۤƱǽ
# ڥ졼󶡤ȤȤˡ͡ॳݡͥ CosNaming::Name
# ʸˤ̾ɽդ륪ڥ졼󶡤롣
#
# ֥Ȥ뤤ľ CORBA ͡ॵФ³
# ʸ塢Υ͡ॵФΥ롼ȥƥȤФƼΥڥ졼
# 롣
# ؤΥ͡ߥ󥰥ƥȤκ䥪֥ȤΥХɤˤơ
# ΥƥȤ¸ߤʤǤ⡢Ū˥ƥȤХ
# ŪΥƥȤ䥪֥ȤΥХɤԤȤǤ롣
#
# @since 0.4.0
#
# @else
# @class CorbaNaming
# @brief CORBA Naming Service helper class
#
# This class is a wrapper class of CosNaming::NamingContext.
# Almost the same operations which CosNaming::NamingContext has are
# provided, and some operation allows string naming representation of
# context and object instead of CosNaming::Name.
#
# The object of the class would connect to a CORBA naming server at
# the instantiation or immediately after instantiation.
# After that the object invokes operations to the root context of it.
# This class realizes forced binding to deep NamingContext, without binding
# intermediate NamingContexts explicitly.
#
# @since 0.4.0
#
# @endif
class CorbaNaming:
  """
  """



  ##
  # @if jp
  #
  # @brief 󥹥ȥ饯
  #
  # @param self
  # @param orb ORB
  # @param name_server ͡ॵФ̾(ǥե:None)
  #
  # @else
  #
  # @brief Consructor
  #
  # @endif
  def __init__(self, orb, name_server=None):
    self._orb = orb
    self._nameServer = ""
    self._rootContext = CosNaming.NamingContext._nil
    self._blLength = 100

    if name_server:
      self._nameServer = "corbaloc::" + name_server + "/NameService"
      try:
        obj = orb.string_to_object(self._nameServer)
        self._rootContext = obj._narrow(CosNaming.NamingContext)
        if CORBA.is_nil(self._rootContext):
          print "CorbaNaming: Failed to narrow the root naming context."

      except CORBA.ORB.InvalidName:
        self.__print_exception()
        print "Service required is invalid [does not exist]."

    return
  

  ##
  # @if jp
  #
  # @brief ǥȥ饯
  # 
  # @param self
  # 
  # @else
  # 
  # @brief destructor
  # 
  # @endif
  def __del__(self):
    return


  ##
  # @if jp
  #
  # @brief ͡ߥ󥰥ӥν
  # 
  # ꤵ줿͡ॵоΥ͡ߥ󥰥ӥޤ
  # 
  # @param self
  # @param name_server ͡ॵФ̾
  # 
  # @else
  # 
  # @endif
  def init(self, name_server):
    self._nameServer = "corbaloc::" + name_server + "/NameService"
    obj = self._orb.string_to_object(self._nameServer)
    self._rootContext = obj._narrow(CosNaming.NamingContext)
    if CORBA.is_nil(self._rootContext):
      raise MemoryError

    return


  ##
  # @if jp
  #
  # @brief 롼ȥƥȤ¸Ƥ뤫֤
  # 
  # 롼ȥƥȤ¸Ƥ뤫ΥåԤ
  # 
  # @param self
  # @else
  # @brief Check on whether the root context is alive.
  # Check on whether the root context is alive.
  # @param self
  # @endif
  # bool CorbaNaming::isAlive()
  def isAlive(self):
    try:
      if self._rootContext._non_existent():
        return False
      return True
    except:
      self.__print_exception()
      return False

    return False


  ##
  # @if jp
  #
  # @brief Object  bind 
  #
  # CosNaming::bind() ȤۤƱƯ򤹤뤬Ϳ줿͡ॵФ
  # 롼ȥƥȤФbind()ƤӽФۤʤ롣
  #
  # Name <name>  Object <obj>  NamingContext ˥Хɤ롣
  # c_n  n ܤ NameComponent 򤢤魯Ȥȡ
  # name  n Ĥ NameComponent ȤʲΤ褦˰롣
  #
  # cxt->bind(<c_1, c_2, ... c_n>, obj) ϰʲƱǤ롣
  # cxt->resolve(<c_1, ... c_(n-1)>)->bind(<c_n>, obj)
  #
  # ʤ1ܤn-1ܤΥƥȤ褷n-1ܤΥƥ
  #  name <n> Ȥơobj  bind 롣
  # ̾˻ä <c_1, ... c_(n-1)>  NemingContext ϡ
  # bindContext()  rebindContext() Ǵ˥ХɺѤߤǤʤФʤʤ
  # ⤷ <c_1, ... c_(n-1)>  NamingContext ¸ߤʤˤϡ
  # NotFound 㳰ȯ롣
  #
  # Хɥե饰 force  true λϡ<c_1, ... c_(n-1)>
  # ¸ߤʤˤ⡢ƵŪ˥ƥȤХɤʤ顢
  # ǽŪ obj ̾ name <c_n> ˥Хɤ롣
  #
  # ξǤ⡢n-1ܤΥƥȾ name<n> Υ֥
  # (Object 뤤 ƥ) ХɤƤ
  # AlreadyBound 㳰ȯ롣
  #
  # @param self
  # @param name_list ֥Ȥդ̾ NameComponent
  # @param obj Ϣդ Object
  # @param force trueξ硢ΥƥȤŪ˥Хɤ
  #              (ǥե:None)
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name_list ̾
  # @exception AlreadyBound name <c_n>  Object Ǥ˥ХɤƤ롣
  #
  # @else
  #
  # @brief
  #
  # @endif
  def bind(self, name_list, obj, force=None):
    if force is None :
      force = True

    try:
      self._rootContext.bind(name_list, obj)
    except CosNaming.NamingContext.NotFound:
      if force:
        self.bindRecursive(self._rootContext, name_list, obj)
      else:
        raise
    except CosNaming.NamingContext.CannotProceed, err:
      if force:
        self.bindRecursive(err.cxt, err.rest_of_name, obj)
      else:
        raise
    except CosNaming.NamingContext.AlreadyBound:
      self._rootContext.rebind(name_list, obj)


  ##
  # @if jp
  #
  # @brief Object  bind 
  #
  # Object  bind ݤͿ̾ʸɽǤ뤳Ȱʳϡbind()
  # ƱǤ롣bind(toName(string_name), obj) 
  #
  # @param self
  # @param string_name ֥Ȥդ̾ʸɽ
  # @param obj Ϣդ륪֥
  # @param force trueξ硢ΥƥȤŪ˥Хɤ
  #              (ǥե:true)
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  string_name ̾
  # @exception AlreadyBound name <n>  Object Ǥ˥ХɤƤ롣
  #
  # @else
  #
  # @brief
  #
  # @endif
  def bindByString(self, string_name, obj, force=True):
    self.bind(self.toName(string_name), obj, force)


  ##
  # @if jp
  #
  # @brief ΥƥȤ bind ʤ Object  bind 
  #
  # context Ϳ줿 NamingContext Фơname ǻꤵ줿
  # ͡ॳݡͥ <c_1, ... c_(n-1)>  NamingContext Ȥ
  # 褷ʤ顢̾ <c_n> Ф obj  bind 롣
  # ⤷<c_1, ... c_(n-1)> б NamingContext ʤˤ
  #  NamingContext Хɤ롣
  #
  # ǽŪ <c_1, c_2, ..., c_(n-1)> б NamingContext 
  # ޤϲ褵줿ǡCosNaming::bind(<c_n>, object) ƤӽФ롣
  # ΤȤǤ˥Хǥ󥰤¸ߤ AlreadyBound㳰ȯ롣
  #
  # ΥƥȤ褹ǡ褷褦Ȥ륳ƥȤ
  # Ʊ̾ NamingContext ǤϤʤ Binding ¸ߤ硢
  # CannotProceed 㳰ȯߤ롣
  #
  # @param self
  # @param context bind 򳫻Ϥ롡NamingContext
  # @param name_list ֥Ȥդ̾Υ͡ॳݡͥ
  # @param obj Ϣդ륪֥
  #
  # @exception CannotProceed <c_1, ..., c_(n-1)> б NamingContext 
  #            ΤҤȤĤǤ NamingContext ʳ object ˥Х
  #            Ƥꡢ³Ǥʤ
  # @exception InvalidName ̾ name_list 
  # @exception AlreadyBound name <c_n> ˤǤ˲餫 object Х
  #            Ƥ롣
  # @else
  #
  # @brief
  #
  # @endif
  def bindRecursive(self, context, name_list, obj):
    length = len(name_list)
    cxt = context
    for i in range(length):
      if i == length -1:
        try:
          cxt.bind(self.subName(name_list, i, i), obj)
        except CosNaming.NamingContext.AlreadyBound:
          cxt.rebind(self.subName(name_list, i, i), obj)
        return
      else:
        if self.objIsNamingContext(cxt):
          cxt = self.bindOrResolveContext(cxt,self.subName(name_list, i, i))
        else:
          raise CosNaming.NamingContext.CannotProceed(cxt, self.subName(name_list, i))
    return


  ##
  # @if jp
  #
  # @brief Object  rebind 
  #
  # name_list ǻꤵ줿 Binding Ǥ¸ߤ bind() Ʊ
  # Ǥ롣Хǥ󥰤Ǥ¸ߤˤϡХǥ󥰤
  # ֤롣
  #
  # @param self
  # @param name_list ֥Ȥդ̾ NameComponent
  # @param obj Ϣդ륪֥
  # @param force trueξ硢ΥƥȤŪ˥Хɤ
  #              (ǥե:true)
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName ̾ name_list 
  #
  # @else
  #
  # @brief
  #
  # @endif
  def rebind(self, name_list, obj, force=True):
    if force is None:
      force = True
      
    try:
      self._rootContext.rebind(name_list, obj)

    except CosNaming.NamingContext.NotFound:
      if force:
        self.rebindRecursive(self._rootContext, name_list, obj)
      else:
        self.__print_exception()
        raise

    except CosNaming.NamingContext.CannotProceed, err:
      if force:
        self.rebindRecursive(err.cxt, err,rest_of_name, obj)
      else:
        self.__print_exception()
        raise
      
    return


  ##
  # @if jp
  #
  # @brief Object  rebind 
  #
  # Object  rebind ݤͿ̾ʸɽǤ뤳Ȱʳ rebind()
  # ƱǤ롣rebind(toName(string_name), obj) 
  #
  # @param self
  # @param string_name ֥Ȥդ̾ʸɽ
  # @param obj Ϣդ륪֥
  # @param force trueξ硢ΥƥȤŪ˥Хɤ
  #              (ǥե:true)
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  string_name ̾
  #
  # @else
  #
  # @brief
  #
  # @endif
  def rebindByString(self, string_name, obj, force=True):
    self.rebind(self.toName(string_name), obj, force)

    return


  ##
  # @if jp
  #
  # @brief ΥƥȤ bind ʤ Object  rebind 
  #
  # name_list <c_n> ǻꤵ줿 NamingContext ⤷ Object Ǥ¸ߤ
  #  bindRecursive() ƱǤ롣
  #
  # name_list <c_n> ǻꤵ줿Хǥ󥰤Ǥ¸ߤˤϡ
  # Хǥ󥰤֤롣
  #
  # @param self
  # @param context bind 򳫻Ϥ롡NamingContext
  # @param name_list ֥Ȥդ̾ NameComponent
  # @param obj Ϣդ륪֥
  #
  # @exception CannotProceed ΥƥȤǤʤ
  # @exception InvalidName Ϳ줿 name_list 
  #
  # @else
  #
  # @brief
  #
  # @endif
  def rebindRecursive(self, context, name_list, obj):
    length = len(name_list)
    for i in range(length):
      if i == length - 1:
        context.rebind(self.subName(name_list, i, i), obj)
        return
      else:
        if self.objIsNamingContext(context):
          try:
            context = context.bind_new_context(self.subName(name_list, i, i))
          except CosNaming.NamingContext.AlreadyBound:
            obj_ = context.resolve(self.subName(name_list, i, i))
            context = obj_._narrow(CosNaming.NamingContext)
        else:
          raise CosNaming.NamingContext.CannotProceed(context, self.subName(name_list, i))
    return


  ##
  # @if jp
  #
  # @brief NamingContext  bind 
  #
  # bind оݤȤƻꤵ줿 name ʸξ bindByString() ȡ
  # ʳξ bind() ƱǤ롣
  #
  # @param self
  # @param name ֥Ȥդ̾
  # @param name_cxt Ϣդ NamingContext
  # @param force trueξ硢ΥƥȤŪ˥Хɤ
  #              (ǥե:True)
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name ̾
  # @exception AlreadyBound name <c_n>  Object Ǥ˥ХɤƤ롣
  #
  # @else
  #
  # @brief
  #
  # @endif
  def bindContext(self, name, name_cxt, force=True):
    if isinstance(name, basestring):
      self.bind(self.toName(name), name_cxt, force)
    else:
      self.bind(name, name_cxt, force)
    return


  ##
  # @if jp
  #
  # @brief NamingContext  bind 
  #
  # bind 륪֥Ȥ NamingContext Ǥ뤳Ȥ
  # bindRecursive() ƱǤ롣
  #
  # @param self
  # @param context bind 򳫻Ϥ롡NamingContext
  # @param name_list ֥Ȥդ̾Υ͡ॳݡͥ
  # @param name_cxt Ϣդ NamingContext
  #
  # @else
  #
  # @brief
  #
  # @endif
  def bindContextRecursive(self, context, name_list, name_cxt):
    self.bindRecursive(context, name_list, name_cxt)
    return


  ##
  # @if jp
  #
  # @brief NamingContext  rebind 
  #
  # bind оݤȤƻꤵ줿 name ʸξ rebindByString() ȡ
  # ʳξ rebind() ƱǤ롣
  # ɤξХǥ󥰤Ǥ¸ߤˤϡ
  # Хǥ󥰤֤롣
  #
  # @param self
  # @param name ֥Ȥդ̾Υ͡ॳݡͥ
  # @param name_cxt Ϣդ NamingContext
  # @param force trueξ硢ΥƥȤŪ˥Хɤ
  #              (ǥե:true)
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name ̾
  #
  # @else
  #
  # @endif
  def rebindContext(self, name, name_cxt, force=True):
    if isinstance(name, basestring):
      self.rebind(self.toName(name), name_cxt, force)
    else:
      self.rebind(name, name_cxt, force)
    return


  ##
  # @if jp
  #
  # @brief ΥƥȤƵŪ rebind  NamingContext  rebind     #
  # bind 륪֥Ȥ NamingContext Ǥ뤳Ȥ
  # rebindRecursive() ƱǤ롣
  #
  # @param self
  # @param context bind 򳫻Ϥ롡NamingContext
  # @param name_list ֥Ȥդ̾ NameComponent
  # @param name_cxt Ϣդ NamingContext
  #
  # @else
  #
  # @brief
  #
  # @endif
  def rebindContextRecursive(self, context, name_list, name_cxt):
    self.rebindRecursive(context, name_list, name_cxt)
    return


  ##
  # @if jp
  #
  # @brief Object  name 褹
  #
  # name  bind Ƥ륪֥ȻȤ֤
  # ͡ॳݡͥ <c_1, c_2, ... c_n> ϺƵŪ˲褵롣
  # 
  #  name Ϳ줿ͤʸξˤϤޤǽ toName() ˤä
  # NameComponent Ѵ롣
  # 
  # CosNaming::resolve() ȤۤƱƯ򤹤뤬Ϳ줿
  # ͡ॵФΥ롼ȥƥȤФ resolve() ƤӽФ
  # ۤʤ롣
  #
  # @param self
  # @param name 褹٤֥Ȥ̾Υ͡ॳݡͥ
  #
  # @return 褵줿֥Ȼ
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name ̾
  #
  # @else
  #
  # @endif
  def resolve(self, name):
    if isinstance(name, basestring):
      name_ = self.toName(name)
    else:
      name_ = name
      
    try:
      obj = self._rootContext.resolve(name_)
      return obj
    except CosNaming.NamingContext.NotFound, ex:
      self.__print_exception()
      return None


  ##
  # @if jp
  #
  # @brief ꤵ줿̾Υ֥Ȥ bind 
  #
  # name  bind Ƥ륪֥ȻȤ롣
  # ͡ॳݡͥ <c_1, c_2, ... c_n> ϺƵŪ˲褵롣
  # 
  #  name Ϳ줿ͤʸξˤϤޤǽ toName() ˤä
  # NameComponent Ѵ롣
  # 
  # CosNaming::unbind() ȤۤƱƯ򤹤뤬Ϳ줿
  # ͡ॵФΥ롼ȥƥȤФ unbind() ƤӽФ
  # ۤʤ롣
  #
  # @param self
  # @param name 륪֥ȤΥ͡ॳݡͥ
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name ̾
  #
  # @else
  #
  # @endif
  # void unbind(const CosNaming::Name& name)
  #   throw(NotFound, CannotProceed, InvalidName);
  def unbind(self, name):
    if isinstance(name, basestring):
      name_ = self.toName(name)
    else:
      name_ = name

    try:
      self._rootContext.unbind(name_)
    except:
      self.__print_exception()

    return


  ##
  # @if jp
  #
  # @brief ƥȤ
  #
  # Ϳ줿͡ॵо줿 NamingContext ֤
  # ֤줿 NamingContext  bind Ƥʤ
  # 
  # @param self
  # 
  # @return 줿 NamingContext
  #
  # @else
  #
  # @endif
  def newContext(self):
    return self._rootContext.new_context()


  ##
  # @if jp
  #
  # @brief ƥȤ bind 
  #
  # Ϳ줿 name ФƿƥȤХɤ롣
  # 줿NamingContext ϥ͡ॵо줿ΤǤ롣
  # 
  #  name Ϳ줿ͤʸξˤϤޤǽ toName() ˤä
  # NameComponent Ѵ롣
  # 
  # @param self
  # @param name NamingContextդ̾Υ͡ॳݡͥ
  # @param force trueξ硢ΥƥȤŪ˥Хɤ
  #              (ǥե:true)
  #
  # @return 줿 NamingContext
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name ̾
  # @exception AlreadyBound name <n>  Object Ǥ˥ХɤƤ롣
  #
  # @else
  #
  # @endif
  def bindNewContext(self, name, force=True):
    if force is None:
      force = True
      
    if isinstance(name, basestring):
      name_ = self.toName(name)
    else:
      name_ = name

    try:
      return self._rootContext.bind_new_context(name_)
    except CosNaming.NamingContext.NotFound:
      if force:
        self.bindRecursive(self._rootContext, name_, self.newContext())
      else:
        self.__print_exception()
        raise
    except CosNaming.NamingContext.CannotProceed, err:
      if force:
        self.bindRecursive(err.cxt, err.rest_of_name, self.newContext())
      else:
        self.__print_exception()
        raise
    return None


  ##
  # @if jp
  #
  # @brief NamingContext 󥢥ƥֲ
  #
  # context ǻꤵ줿 NamingContext 󥢥ƥֲ롣
  # context ¾ΥƥȤХɤƤ NotEmpty 㳰
  # ȯ롣
  # 
  # @param self
  # @param context 󥢥ƥֲ NamingContext
  #
  # @exception NotEmpty оcontext ¾ΥƥȤХɤƤ롣
  #
  # @else
  #
  # @else
  #
  # @brief Destroy the naming context
  #
  # Delete the specified naming context.
  # any bindings should be <unbind> in which the given context is bound to
  # some names before invoking <destroy> operation on it. 
  #
  # @param context NamingContext which is destroied.
  #     
  # @exception NotEmpty 
  #
  # @else
  #
  # @endif
  def destroy(self, context):
    context.destroy()


  ##
  # @if jp
  # @brief NamingContext ƵŪ˲ä󥢥ƥֲ
  #
  # context Ϳ줿 NamingContext Фơname ǻꤵ줿
  # ͡ॳݡͥ <c_1, ... c_(n-1)>  NamingContext Ȥ
  # 褷ʤ顢̾ <c_n> Ф 󥢥ƥֲԤ
  #
  # @param self
  # @param context 󥢥ƥֲ NamingContext
  #
  # @exception NotEmpty оcontext ¾ΥƥȤХɤƤ롣
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name ̾
  #
  # @else
  # @brief Destroy the naming context recursively
  # @endif
  def destroyRecursive(self, context):
    cont = True
    bl = []
    bi = 0
    bl, bi = context.list(self._blLength)
    while cont:
      for i in range(len(bl)):
        if bl[i].binding_type == CosNaming.ncontext:
          obj = context.resolve(bl[i].binding_name)
          next_context = obj._narrow(CosNaming.NamingContext)

          self.destroyRecursive(next_context)
          context.unbind(bl[i].binding_name)
          next_context.destroy()
        elif bl[i].binding_type == CosNaming.nobject:
          context.unbind(bl[i].binding_name)
        else:
          assert(0)
      if CORBA.is_nil(bi):
        cont = False
      else:
        bi.next_n(self._blLength, bl)

    if not (CORBA.is_nil(bi)):
      bi.destroy()
    return


  ##
  # @if jp
  # @brief ٤Ƥ Binding 
  #
  # ϿƤƤBinding 롣
  #
  # @param self
  #
  # @else
  # @brief Destroy all binding
  # @endif
  def clearAll(self):
    self.destroyRecursive(self._rootContext)
    return


  ##
  # @if jp
  # @brief Ϳ줿 NamingContext  Binding 
  #
  # ꤵ줿 NamingContext  Binding 롣
  #
  # @param self
  # @param name_cxt Binding о NamingContext
  # @param how_many Binding 볬ؤο
  # @param rbl  Binding ݻۥ
  # @param rbi  Binding 򤿤ɤ뤿Υƥ졼
  #
  # @else
  # @endif
  def list(self, name_cxt, how_many, rbl, rbi):
    bl, bi = name_cxt.list(how_many)

    for i in bl:
      rbl.append(bl)

    rbi.append(bi)
  

  #============================================================
  # interface of NamingContext
  #============================================================

  ##
  # @if jp
  # @brief Ϳ줿 NameComponent ʸɽ֤
  #
  # ꤵ줿 NameComponent ʸѴ롣
  #
  # @param self
  # @param name_list Ѵо NameComponent
  #
  # @return ʸѴ
  #
  # @exception InvalidName  name_list ̾
  #
  # @else
  # @brief Get string representation of given NameComponent
  # @endif
  def toString(self, name_list):
    if len(name_list) == 0:
      raise CosNaming.NamingContext.InvalidName

    slen = self.getNameLength(name_list)
    string_name = [""]
    self.nameToString(name_list, string_name, slen)

    return string_name[0]


  ##
  # @if jp
  # @brief Ϳ줿ʸɽ NameComponent ʬ򤹤
  #
  # ꤵ줿ʸ NameComponent Ѵ롣
  #
  # @param self
  # @param sname Ѵоʸ
  #
  # @return NameComponent Ѵ
  #
  # @exception InvalidName  sname 
  #
  # @else
  # @brief Get NameComponent from gien string name representation
  # @endif
  def toName(self, sname):
    if not sname:
      raise CosNaming.NamingContext.InvalidName

    string_name = sname
    name_comps = []

    nc_length = 0
    nc_length = self.split(string_name, "/", name_comps)

    if not (nc_length > 0):
      raise CosNaming.NamingContext.InvalidName

    name_list = [CosNaming.NameComponent("","") for i in range(nc_length)]

    for i in range(nc_length):
      pos = string.rfind(name_comps[i][0:],".")
      if pos == -1:
        name_list[i].id   = name_comps[i]
        name_list[i].kind = ""
      else:
        name_list[i].id   = name_comps[i][0:pos]
        name_list[i].kind = name_comps[i][(pos+1):]

    return name_list


  ##
  # @if jp
  # @brief Ϳ줿 addr  string_name  URLɽ
  #
  # ꤵ줿ɥ쥹̾ΤURLѴ롣
  #
  # @param self
  # @param addr Ѵоݥɥ쥹
  # @param string_name Ѵо̾
  #
  # @return URL Ѵ
  #
  # @exception InvalidAddress  addr 
  # @exception InvalidName  string_name 
  #
  # @else
  # @brief Get URL representation from given addr and string_name
  # @endif
  def toUrl(self, addr, string_name):
    return self._rootContext.to_url(addr, string_name)


  ##
  # @if jp
  # @brief Ϳ줿ʸɽ resolve ֥Ȥ֤
  #
  # ꤵ줿ʸɽresolve֥Ȥ롣
  #
  # @param self
  # @param string_name оݥ֥ʸɽ
  #
  # @return 褵줿֥
  #
  # @exception NotFound  <c_1, c_2, ..., c_(n-1)> ¸ߤʤ
  # @exception CannotProceed 餫ͳǽ³Ǥʤ
  # @exception InvalidName  name ̾
  # @exception AlreadyBound name <n>  Object Ǥ˥ХɤƤ롣
  #
  # @else
  # @brief Resolve from name of string representation and get object 
  # @endif
  def resolveStr(self, string_name):
    return self.resolve(self.toName(string_name))


  #============================================================
  # Find functions
  #============================================================

  ##
  # @if jp
  #
  # @brief ֥Ȥ̾Хɤޤϲ褹
  #
  # ꤵ줿ƥȤФƥ֥Ȥ NameComponent ǻꤵ줿
  # ֤˥Хɤ롣
  # Ʊս˴¾ǤХɺѤߤξϡ¸ΥХɺѤǤ
  # 롣
  #
  # @param self
  # @param context bind ⤷ resole оݥƥ
  # @param name_list ֥Ȥդ̾ NameComponent
  # @param obj Ϣդ Object
  #
  # @return NameComponent ǻꤵ줿֤˥ХɤƤ륪֥
  #
  # @else
  # @brief Bind of resolve the given name component
  # @endif
  def bindOrResolve(self, context, name_list, obj):
    try:
      context.bind_context(name_list, obj)
      return obj
    except CosNaming.NamingContext.AlreadyBound:
      obj = context.resolve(name_list)
      return obj
    return CORBA.Object._nil


  ##
  # @if jp
  #
  # @brief ƥȤ̾Хɤޤϲ褹
  #
  # ꤵ줿ƥȤФ Context NameComponent ǻꤵ줿֤
  # Хɤ롣
  # Context ξϿƥȤƥХɤ롣
  # Ʊս˴¾ǤХɺѤߤξϡ¸ΥХɺѤǤ
  # 롣
  #
  # @param self
  # @param context bind ⤷ resole оݥƥ
  # @param name_list ƥȤդ̾ NameComponent
  # @param new_context Ϣդ Context(ǥե:None)
  #
  # @return NameComponent ǻꤵ줿֤˥ХɤƤContext
  #
  # @else
  # @brief Bind of resolve the given name component
  # @endif
  def bindOrResolveContext(self, context, name_list, new_context=None):
    if new_context is None:
      new_cxt = self.newContext()
    else:
      new_cxt = new_context

    obj = self.bindOrResolve(context, name_list, new_cxt)
    return obj._narrow(CosNaming.NamingContext)


  ##
  # @if jp
  # @brief ͡ॵФ̾
  #
  # ꤷ͡ॵФ̾롣
  #
  # @param self
  #
  # @return ͡ॵФ̾
  #
  # @else
  # @brief Get the name of naming server
  # @endif
  def getNameServer(self):
    return self._nameServer


  ##
  # @if jp
  # @brief 롼ȥƥȤ
  #
  # ꤷ͡ॵФΥ롼ȥƥȤ롣
  #
  # @param self
  #
  # @return ͡ॵФΥ롼ȥƥ
  #
  # @else
  # @brief Get the root context
  # @endif
  def getRootContext(self):
    return self._rootContext


  ##
  # @if jp
  # @brief ֥Ȥ͡ߥ󥰥ƥȤȽ̤
  #
  # ꤷǤ͡ߥ󥰥ƥȤȽ̤
  #
  # @param self
  # @param obj Ƚо
  #
  # @return Ƚ̷(͡ߥ󥰥ƥ:trueʳ:false)
  #
  # @else
  # @brief Whether the object is NamingContext
  # @endif
  def objIsNamingContext(self, obj):
    nc = obj._narrow(CosNaming.NamingContext)
    if CORBA.is_nil(nc):
      return False
    else:
      return True


  ##
  # @if jp
  # @brief Ϳ줿̾͡ߥ󥰥ƥȤɤȽ̤
  #
  # NameComponent ⤷ʸǻꤷǤ͡ߥ󥰥ƥȤ
  # Ƚ̤
  #
  # @param self
  # @param name_list Ƚо
  #
  # @return Ƚ̷(͡ߥ󥰥ƥ:trueʳ:false)
  #
  # @else
  # @brief Whether the given name component is NamingContext
  # @endif
  def nameIsNamingContext(self, name_list):
    return self.objIsNamingContext(self.resolve(name_list))


  ##
  # @if jp
  # @brief ͡ॳݡͥȤʬ֤
  #
  # ꤵ줿ϰϤΥ͡ॳݡͥȤ롣
  # λ֤ꤵƤʤϡǸǤ͡ॳݡͥ
  # ֤
  #
  # @param self
  # @param name_list оNameComponent
  # @param begin ϰϳϰ
  # @param end ϰϽλ(ǥե:None)
  #
  # @return NameComponent 
  #
  # @else
  # @brief Get subset of given name component
  # @endif
  def subName(self, name_list, begin, end = None):
    if end is None or end < 0:
      end = len(name_list) - 1

    sub_len = end - (begin -1)
    objId = ""
    kind  = ""
    
    sub_name = []
    for i in range(sub_len):
      sub_name.append(name_list[begin + i])

    return sub_name


  ##
  # @if jp
  # @brief ͡ॳݡͥȤʸɽ
  #
  # ꤷϰϤΥ͡ॳݡͥȤʸɽ롣
  # ʸɽϡNameComponentι{Nc[0],Nc[1],Nc[2]}ξ硢
  #   Nc[0]id.Nc[0].kind/Nc[1]id.Nc[1].kind/Nc[2].id/Nc[2].kind
  # ȤǼǤ롣
  # ʸĹꤷĹʾξϡ
  # ꤷĹڤΤƤ롣
  #
  # @param self
  # @param name_list оNameComponent
  # @param string_name ʸ
  # @param slen оʸ
  #
  # @else
  # @brief Get string representation of name component
  # @endif
  def nameToString(self, name_list, string_name, slen):
    for i in range(len(name_list)):
      for id_ in name_list[i].id:
        if id_ == "/" or id_ == "." or id_ == "\\":
          string_name[0] += "\\"
        string_name[0] += id_

      if name_list[i].id == "" or name_list[i].kind != "":
        string_name[0] += "."

      for kind_ in name_list[i].kind:
        if kind_ == "/" or kind_ == "." or kind_ == "\\":
          string_name[0] += "\\"
        string_name[0] += kind_

      string_name[0] += "/"


  ##
  # @if jp
  # @brief ͡ॳݡͥȤʸɽʸĹ
  #
  # ꤷ͡ॳݡͥȤʸɽĹ롣
  # ʸɽϡNameComponentι{Nc[0],Nc[1],Nc[2]}ξ硢
  #   Nc[0]id.Nc[0].kind/Nc[1]id.Nc[1].kind/Nc[2].id/Nc[2].kind
  # ȤǼǤ롣
  #
  # @param self
  # @param name_list оNameComponent
  #
  # @return ꤷ͡ॳݡͥȤʸĹ
  #
  # @else
  # @brief Get string length of the name component's string representation
  # @endif
  def getNameLength(self, name_list):
    slen = 0

    for i in range(len(name_list)):
      for id_ in name_list[i].id:
        if id_ == "/" or id_ == "." or id_ == "\\":
          slen += 1
        slen += 1
      if name_list[i].id == "" or name_list[i].kind == "":
        slen += 1

      for kind_ in name_list[i].kind:
        if kind_ == "/" or kind_ == "." or kind_ == "\\":
          slen += 1
        slen += 1

      slen += 1

    return slen


  ##
  # @if jp
  # @brief ʸʬ
  #
  # ʸꤷǥߥʬ䤹롣
  #
  # @param self
  # @param input ʬоʸ
  # @param delimiter ʬѥǥߥ
  # @param results ʬ
  #
  # @return ʬ䤷ʸǿ
  #
  # @else
  # @brief Split of string
  # @endif
  def split(self, input, delimiter, results):
    delim_size = len(delimiter)
    found_pos = 0
    begin_pos = 0
    pre_pos = 0
    substr_size = 0

    if input[0:delim_size] == delimiter:
      begin_pos = delim_size
      pre_pos = delim_size

    while 1:
      found_pos = string.find(input[begin_pos:],delimiter)
      if found_pos == -1:
        results.append(input[pre_pos:])
        break

      if found_pos > 0 and input[found_pos + begin_pos - 1] == "\\":
        begin_pos += found_pos + delim_size
      else:
        substr_size = found_pos + (begin_pos - pre_pos)
        if substr_size > 0:
          results.append(input[pre_pos:(pre_pos+substr_size)])
        begin_pos += found_pos + delim_size
        pre_pos   = begin_pos

    return len(results)


  ##
  # @if jp
  #
  # @brief 㳰
  #  㳰Ϥ롣
  #
  # @else
  #
  # @brief Print exception information 
  #  Print exception information 
  # @endif
  def __print_exception(self):
    if sys.version_info[0:3] >= (2, 4, 0):
      print traceback.format_exc()
    else:
      _exc_list = traceback.format_exception(*sys.exc_info())
      _exc_str = "".join(_exc_list)
      print _exc_str

    return
