#!/usr/bin/env python
# -*- coding: euc-jp -*-
  

##
# @file Properties.py
# @brief Property list class (derived from Java Properties)
# @date $Date: $
# @author Noriaki Ando <n-ando@aist.go.jp> and Shinji Kurihara
#
# Copyright (C) 2006-2008
#     Task-intelligence Research Group,
#     Intelligent Systems Research Institute,
#     National Institute of
#         Advanced Industrial Science and Technology (AIST), Japan
#     All rights reserved.


import sys
import string

import OpenRTM_aist


##
# @if jp
#
# @class Properties
# @brief ץѥƥåȤɽ륯饹
#
# Properties 饹ϡѤΥץѥƥåȤɽ Properties 򥹥ȥ꡼
# ݴɤꡢȥ꡼फɤꤹ뤳ȤǤ롣
# ץѥƥꥹȤγƥӤбͤʸȤʤäƤ롣
#
# ץѥƥꥹȤˤϡΡ֥ǥե͡פȤ̤ΥץѥƥꥹȤ
# ȤǤ롣ΥץѥƥꥹȤǥץѥƥĤʤȡ
# 2ܤΥץѥƥꥹȤ롣 
#
# ץѥƥμˤ getProperty() ץѥƥΥåȤˤ setProperty() 
# ä᥽åɤѤ뤳Ȥ侩롣
#
# ץѥƥ򥹥ȥ꡼¸Ȥޤϥȥ꡼फɤȤ
# ˡISO 8859-1 ʸ󥳡ǥ󥰤Ѥ롣Υ󥳡ǥ󥰤
# ľɽǤʤʸϡȤǤʤ
#
# Υ饹ϡJava  Properties 饹 (java.util.Properties) ȤۤƱͤ
# ᥽åɤġޤϤե Java  Properties 饹
# ϤΤȸߴ뤬Unicode ޤΤϰȤǤʤ
#
# @since 0.4.0
#
# @else
#
# @class Properties
#
# The Properties class represents a persistent set of properties. The
# Properties can be saved to a stream or loaded from a stream. Each key and
# its corresponding value in the property list is a string. 
#
# A property list can contain another property list as its "defaults"; this
# second property list is searched if the property key is not found in the
# original property list. 
#
# Because Properties inherits from Hashtable, the put and putAll methods can
# be applied to a Properties object. Their use is strongly discouraged as 
# they allow the caller to insert entries whose keys or values are not 
# Strings. The setProperty method should be used instead. If the store or 
# save method is called on a "compromised" Properties object that contains a 
# non-String key or value, the call will fail. 
#
# The load and store methods load and store properties in a simple
# line-oriented format specified below. This format uses the ISO 8859-1
# character encoding. Characters that cannot be directly represented in this
# encoding can be written using Unicode escapes ; only a single 'u' character
# is allowed in an escape sequence. The native2ascii tool can be used to
# convert property files to and from other character encodings. 
#
# This class has almost same methods of Java's Properties class. Input and 
# Output stream of this properties are compatible each other except Unicode
# encoded property file.
#
# @endif
class Properties:
  """
  """

  ##
  # @if jp
  #
  # @brief 󥹥ȥ饯
  #
  # ʲν˰å󥹥󥹤Ԥ
  #
  #  prop ͤꤵƤ硢
  # Ϳ줿 Properties Υͤӥǥեͤ
  # ƤΤޤޥԡ롣
  #
  #  key ͤꤵƤ硢
  # key  value ΤߤͿ Property Υ롼ȥΡɤ롣
  # ͤƥǥեͤȤꤵ롣
  #
  #  defaults_map ͤꤵƤ硢
  # defaults_map ꤵ줿Ƥǥեͤˤ Properties 롣
  # ͤƥǥեͤȤꤵ롣
  # 
  #  defaults_str ͤꤵƤ硢
  # ꤵ줿ǥեͤĶΥץѥƥꥹȤ롣
  # ͤƥǥեͤȤꤵ롣
  # ǥեͤ char* ˤͿ졢key  value Фˤʤä
  # ꡢꥹȤνüοɽ num ʸ key Ϳ
  # ʤФʤʤ
  # ʲ򼨤
  #
  # <pre>
  # const char* defaults = {
  #     "key1", "value1",
  #     "key2", "value2",
  #     "key3", "value3",
  #     "key4", "value4",
  #     "key5", "value5",
  #     "" };
  # Properties p(defaults);
  # // ⤷
  # Properties p(defaults, 10);
  # </pre>
  # 
  # @param self
  # @param key ץѥƥΥ(ǥե:None)
  # @param value ץѥƥ(ǥե:None)
  # @param defaults_map ǥեͤȤƻꤵmap(ǥե:None)
  # @param defaults_str ǥեͤꤹ(ǥե:None)
  # @param num ǥեͤꤹǿ(ǥե:None)
  # @param prop ǥեͤȤƻꤵproperty(ǥե:None)
  # 
  # @else
  #
  # @brief Constructor
  #
  # All of given Properties's keys, values and default values are copied to
  # new Properties.
  #
  # Creates a root node of Property with root's key and value.
  #
  # Creates an Properties with default value of std::string map.
  #
  # Creates an empty property list with the specified defaults.
  # The default values are given by array of char*, which should be pairs
  # of "key" and "value". The end of list is specified by argument "num",
  # which specifies number of array or null character of key.
  # The following is an example.
  #
  # const char* defaults = {
  #     "key1", "value1",
  #     "key2", "value2",
  #     "key3", "value3",
  #     "key4", "value4",
  #     "key5", "value5",
  #     "" };
  # Properties p(defaults);
  # // or
  # Properties p(defaults, 10);
  #
  # @endif
  def __init__(self, key=None, value=None, defaults_map=None, defaults_str=None, num=None, prop=None):
    self.default_value = ""
    self.root = None
    self.empty = ""
    self.leaf = []

    # Properties::Properties(const Properties& prop)
    if prop:
      self.name          = prop.name
      self.value         = prop.value
      self.default_value = prop.default_value

      keys = prop.propertyNames()
      for _key in keys:
        node = None
        node = prop.getNode(_key)
        if node:
          self.setDefault(_key, node.default_value)
          self.setProperty(_key, node.value)
          
      return

    # Properties::Properties(const char* key, const char* value)
    if key:
      self.name = key
      if value is None:
        self.value = ""
      else:
        self.value = value
      return

    self.name  = ""
    self.value = ""

    # Properties::Properties(std::map<std::string, std::string>& defaults)
    if defaults_map:
      #for i in range(len(defaults_map.items())):
      #  self.setDefault(defaults_map.keys()[i], defaults_map.values()[i])
      for key, value in defaults_map.items():
        self.setDefault(key, value)
      return

    if defaults_str:
      if num is None:
        _num = sys.maxint
      else:
        _num = num
      self.setDefaults(defaults_str, _num)
      return


  ##
  # @if jp
  # @brief 黻
  # 
  # ͤ Properties Υͤӥǥեͤƺ졢
  # ͤ Properties ΥͤӥǥեͤƤΤޤ
  # ԡ롣
  # 
  # @param self
  # @param prop OpenRTM_aist.Properties
  # 
  # @else
  # @brief Assignment operator
  # @param self
  # @param prop OpenRTM_aist.Properties
  # @endif
  def assigmentOperator(self, prop):
    self.clear()
    self.name = prop.name
    self.value = prop.value
    self.default_value = prop.default_value

    keys = prop.propertyNames()

    for key in keys:
      node = None
      node = prop.getNode(key)
      if node:
        self.setDefault(key, node.default_value)
        self.setProperty(key, node.value)

    return self


  ##
  # @if jp
  #
  # @brief ǥȥ饯
  #
  # @param self
  #
  # @else
  #
  # @brief Destructor
  #
  # @endif
  def __del__(self):
    self.clear()
    if self.root:
      self.root.removeNode(self.name)
    return

  #============================================================
  # public functions
  #============================================================


  ##
  # @if jp
  # @brief Name μ
  #
  # ץѥƥ̾Τ롣
  #
  # @param self
  #
  # @return ץѥƥ̾
  #
  # @else
  #
  # @endif
  def getName(self):
    return self.name


  ##
  # @if jp
  # @brief ͤμ
  #
  # ץѥƥͤ롣
  #
  # @param self
  #
  # @return ץѥƥ
  #
  # @else
  #
  # @endif
  def getValue(self):
    return self.value


  ##
  # @if jp
  # @brief ǥեͤμ
  #
  # ץѥƥΥǥեͤ롣
  #
  # @param self
  #
  # @return ץѥƥǥե
  #
  # @else
  #
  # @endif
  def getDefaultValue(self):
    return self.default_value


  ##
  # @if jp
  # @brief Ǥμ
  #
  # ץѥƥλǤ롣
  #
  # @param self
  #
  # @return 
  #
  # @else
  #
  # @endif
  def getLeaf(self):
    return self.leaf


  ##
  # @if jp
  # @brief 롼Ǥμ
  #
  # ץѥƥΥ롼Ǥ롣
  #
  # @param self
  #
  # @return 롼
  #
  # @else
  #
  # @endif
  def getRoot(self):
    return self.root


  ##
  # @if jp
  #
  # @brief ꤵ줿ĥץѥƥ򡢥ץѥƥꥹȤõ
  #
  # ꤵ줿ĥץѥƥ򡢥ץѥƥꥹȤõ
  # ΥץѥƥꥹȤˤʤϡǥեͤΰ֤롣 
  #
  # @param self
  # @param key ץѥƥ
  # @param default ǥե(ǥե:None)
  #
  # @return ꤵ줿ͤĤΥץѥƥꥹȤ
  #
  # @else
  #
  # @brief Searches for the property with the specified key in this property
  #
  # Searches for the property with the specified key in this property list.
  # The method returns the default value argument if the property is not 
  # found.
  #
  # @param key the property key
  # @param defaultValue a default value. 
  #
  # @return the value in this property list with the specified key value.
  #
  # @endif
  def getProperty(self, key, default=None):
    if default is None:
      keys = []
      #keys = string.split(key, ".")
      self.split(key, ".", keys)

      node = None
      node = self._getNode(keys, 0, self)
      if node:
        if node.value:
          return node.value
        else:
          return node.default_value
      return self.empty

    else:
      value = self.getProperty(key)
      if value:
        return value
      else:
        return default


  ##
  # @if jp
  # @brief ꤵ줿Фƥǥեͤ
  #
  # ꤵ줿ĥץѥƥΥǥե֤ͤ
  # ꤵ줿ĥץѥƥ¸ߤʤˤ϶ʸ֤
  #
  # @param self
  # @param key ץѥƥ
  #
  # @return ꤵ줿ͤĥץѥƥΥǥե
  #
  # @else
  # @brief Set value as the default value to specified key's property
  # @endif
  def getDefault(self, key):
    keys = []
    #keys = string.split(key, ".")
    self.split(key, ".", keys)
    node = None
    node = self._getNode(keys, 0, self)
    if node:
      return node.default_value

    return self.empty


  ##
  # @if jp
  #
  # @brief Properties  value  key ˤĤϿ
  #
  # Properties  value  key ˤĤϿ롣
  # Ǥ key ФͤäƤ硢ͤ˸Ť֤ͤ
  #
  # @param self
  # @param key ץѥƥꥹȤ֤륭
  # @param value key б(ǥե:None)
  #
  # @return ץѥƥꥹȤλꤵ줿͡줬ʤ null
  #
  # @else
  #
  # @brief Sets a value associated with key in the property list
  #
  # This method sets the "value" associated with "key" in the property list.
  # If the property list has a value of "key", old value is returned.
  #
  # @param key the key to be placed into this property list.
  # @param value the value corresponding to key. 
  #
  # @return the previous value of the specified key in this property list,
  #         or null if it did not have one.
  #
  #@endif
  def setProperty(self, key, value=None):
    if value is not None:
      keys = []
      #keys = string.split(key, ".")
      self.split(key, ".", keys)
      curr = self
      for _key in keys:
        next = curr.hasKey(_key)
        if next is None:
          next = OpenRTM_aist.Properties(key=_key)
          next.root = curr
          curr.leaf.append(next)
        curr = next
      retval = curr.value
      curr.value = value
      return retval

    else:
      self.setProperty(key, self.getProperty(key))
      prop = self.getNode(key)
      return prop.value


  ##
  # @if jp
  # @brief ǥեͤϿ
  #
  # key ǻꤵǤ˥ǥեͤϿ롣
  #
  # @param self
  # @param key ǥեͤϿץѥƥΥ
  # @param value Ͽǥե
  #
  # @return ꤵ줿ǥե
  #
  # @else
  # @brief Sets a default value associated with key in the property list
  # @endif
  def setDefault(self, key, value):
    keys = []
    self.split(key, ".", keys)
    #keys = string.split(key, ".")

    curr = self
    for _key in keys:
      next = curr.hasKey(_key)
      if next is None:
        next = OpenRTM_aist.Properties(key=_key)
        next.root = curr
        curr.leaf.append(next)
      curr = next
    if value != "" and value[-1] == "\n":
      value = value[0:len(value)-1]
    curr.default_value = value
    return value


  ##
  # @if jp
  # @brief Properties ˥ǥեͤޤȤϿ
  #
  # ǻꤵ줿Ǥ˥ǥեͤޤȤϿ롣
  # ǥեͤ char* ˤͿ졢key  value Фˤʤä
  # ꡢꥹȤνüοɽ num ʸ key Ϳ
  # ʤФʤʤ
  # 
  # @param self
  # @param defaults ǥեͤꤹ
  # @param num ǥեͤꤹǿ(ǥե:None)
  # 
  # @else
  # @brief Sets a default value associated with key in the property list
  # @endif
  def setDefaults(self, defaults, num = None):
    if num is None:
      num = sys.maxint

    i = 0
    len_ = len(defaults)
    while 1:
      if i > num or i > (len_ - 1) or defaults[i] == "":
        break

      key = [defaults[i]]
      value = [defaults[i+1]]

      OpenRTM_aist.eraseHeadBlank(key)
      OpenRTM_aist.eraseTailBlank(key)

      OpenRTM_aist.eraseHeadBlank(value)
      OpenRTM_aist.eraseTailBlank(value)

      self.setDefault(key[0], value[0])

      i +=2



  #============================================================
  # load and save functions
  #============================================================

  ##
  # @if jp
  #
  # @brief ꤵ줿ϥȥ꡼ˡץѥƥꥹȤϤ
  #
  # ꤵ줿ϥȥ꡼ˡץѥƥꥹȤϤ롣
  # Υ᥽åɤϼ˥ǥХåѤ롣
  #
  # @param self
  # @param out ϥȥ꡼
  #
  # @else
  #
  # @brief Prints this property list out to the specified output stream
  #
  # Prints this property list out to the specified output stream.
  # This method is useful for debugging.
  #
  # @param out an output stream.
  #
  # @endif
  def list(self, out):
    self._store(out, "", self)
    return


  ##
  # @if jp
  #
  # @brief ϥȥ꡼फ饭ǤФˤʤäץѥƥꥹȤɤ߹
  #
  # ϥȥ꡼फ饭ǤФˤʤäץѥƥꥹȤɤ߹ࡣ
  # ȥ꡼ϡISO 8859-1 ʸ󥳡ǥ󥰤ѤƤȤߤʤ롣
  # ƥץѥƥϡϥȥ꡼˹ñ̤ϿƤΤȤߤʤ졢
  # ƹԤϹԶڤʸ (\\n\\rޤ \\r\\n) ǽ롣
  # ϥȥ꡼फɤ߹Ԥϡϥȥ꡼ǥեν
  # ãޤǽ롣
  #
  # ʸιԡޤϺǽʸ ASCII ʸ # ޤ ! Ǥ
  # Ԥ̵뤵롣Ĥޤꡢ# ޤ ! ϥȹԤ򼨤
  #
  # Ԥޤϥȹ԰ʳΤ٤ƤιԤϡơ֥ɲäץѥƥ
  # 򵭽Ҥ롣Ԥν꤬ \ ξϡιԤз³ԤȤ
  #  (򻲾) ϡǽʸ顢ǽ ASCII ʸ
  # =:ޤ϶ʸľޤǤΡΤ٤Ƥʸ鹽롣
  #
  # ν򼨤ʸϡ \ դ뤳Ȥˤꥭ˴ޤ뤳Ȥ
  # Ǥ롣θζϤ٤ƥåפ롣
  # θκǽʸ = ޤ : ǤϡΥ
  # ̵뤵졢ΤȤζʸ⤹٤ƥåפ롣
  # ΤʳʸϤ٤ơϢʸΰȤʤ롣
  # ʸǤϡASCII ץ \\t\\n\\r\\\\\\"
  # \\'\\ (ߵȥڡ) \\uxxxx ǧ졢ñȤʸѴ
  # 롣
  # ޤԤκǸʸ \ ǤϡιԤϸߤιԤη³Ȥ
  # 롣ξ硢\ ȹԶڤʸ˴졢³ԤƬ˶
  # Ф⤹٤˴졢ʸΰˤϤʤʤ 
  #
  # ȤС 4 ԤϤ줾쥭 Truth ȴϢ Beauty ɽ
  # 
  # Truth = Beauty <BR>
  # Truth:Beauty <BR>
  # Truth\\t\\t\\t:Beauty <BR>
  #
  # ޤ 3 Ԥ 1 ĤΥץѥƥɽ 
  #
  # fruits\\t\\t\\t\\tapple, banana, pear, \ <BR>
  #                                  cantaloupe, watermelon, \ <BR>
  #                                  kiwi, mango <BR>
  #  fruits ǡǤ˴Ϣդ롣 
  # "apple, banana, pear, cantaloupe, watermelon, kiwi, mango"
  # ǽŪʷ̤ǥޤΤȤɬڡɽ褦ˡ
  #  \ ˥ڡ롣Ԥν򼨤 \ ȡ³ԤƬˤ
  # ˴졢¾ʸִʤ 
  # ޤ 3 ܤǤϡ cheeses ǡϢǤʸ
  # Ǥ뤳Ȥɽ 
  #
  # cheeses <BR>
  # ϡcheeses ǡϢǤ϶ʸǤ뤳ȤꤷƤ롣 
  #
  # @param self
  # @param inStream ϥȥ꡼ 
  #
  # @else
  #
  # @brief Loads property list consists of key:value from input stream
  #
  # Reads a property list (key and element pairs) from the input stream.
  # The stream is assumed to be using the ISO 8859-1 character encoding; that
  # is each byte is one Latin1 character. Characters not in Latin1, and
  # certain special characters, can be represented in keys and elements using
  # escape sequences similar to those used for character and string literals
  # The differences from the character escape sequences used for characters
  # and strings are: 
  # - Octal escapes are not recognized. 
  # - The character sequence \b does not represent a backspace character. 
  # - The method does not treat a backslash character, \, before a non-valid
  #   escape character as an error; the backslash is silently dropped. For
  #   example, in a Java string the sequence "\z" would cause a compile time
  #   error. In contrast, this method silently drops the backslash. 
  #   Therefore, this method treats the two character sequence "\b" as 
  #   equivalent to the single character 'b'. 
  # - Escapes are not necessary for single and double quotes; however, by the
  #   rule above, single and double quote characters preceded by a backslash
  #   still yield single and double quote characters, respectively. 
  # An IllegalArgumentException is thrown if a malformed Unicode escape
  # appears in the input. 
  #
  # This method processes input in terms of lines. A natural line of input is
  # terminated either by a set of line terminator characters
  # (\n or \r or \r\n) or by the end of the file. A natural line may be 
  # either a blank line, a comment line, or hold some part of a key-element 
  # pair. The logical line holding all the data for a key-element pair may 
  # be spread out across several adjacent natural lines by escaping the line 
  # terminator sequence with a backslash character, \. Note that a comment 
  # line cannot be extended in this manner; every natural line that is a 
  # comment must have its own comment indicator, as described below. If a 
  # logical line is continued over several natural lines, the continuation 
  # lines receive further processing, also described below. Lines are read 
  # from the input stream until end of file is reached. 
  #
  # A natural line that contains only white space characters is considered
  # blank and is ignored. A comment line has an ASCII '#' or '!' as its first
  # non-white space character; comment lines are also ignored and do not
  # encode key-element information. In addition to line terminators, this
  # method considers the characters space (' ', '\u0020'), tab 
  # ('\t', '\u0009'), and form feed ('\f', '\u000C') to be white space. 
  #
  # If a logical line is spread across several natural lines, the backslash
  # escaping the line terminator sequence, the line terminator sequence, and
  # any white space at the start the following line have no affect on the key
  # or element values. The remainder of the discussion of key and element
  # parsing will assume all the characters constituting the key and element
  # appear on a single natural line after line continuation characters have
  # been removed. Note that it is not sufficient to only examine the 
  # character preceding a line terminator sequence to see if the line 
  # terminator is escaped; there must be an odd number of contiguous 
  # backslashes for the line terminator to be escaped. Since the input is 
  # processed from left to right, a non-zero even number of 2n contiguous 
  # backslashes before a line terminator (or elsewhere) encodes n 
  # backslashes after escape processing. 
  #
  # The key contains all of the characters in the line starting with the 
  # first non-white space character and up to, but not including, the first
  # unescaped '=', ':', or white space character other than a line 
  # terminator. All of these key termination characters may be included in 
  # the key by escaping them with a preceding backslash character; 
  # for example,
  #
  # \:\=
  #
  # would be the two-character key ":=". Line terminator characters can be
  # included using \r and \n escape sequences. Any white space after the key
  # is skipped; if the first non-white space character after the key is '=' 
  # or ':', then it is ignored and any white space characters after it are 
  # also skipped. All remaining characters on the line become part of the
  # associated element string; if there are no remaining characters, the
  # element is the empty string "". Once the raw character sequences
  # constituting the key and element are identified, escape processing is
  # performed as described above. 
  #
  # As an example, each of the following three lines specifies the key 
  # "Truth" and the associated element value "Beauty": 
  #
  # Truth = Beauty <BR>
  #        Truth:Beauty <BR>
  # Truth                  :Beauty <BR>
  #  As another example, the following three lines specify a single 
  # property: 
  #
  # fruits                           apple, banana, pear, \ <BR>
  #                                  cantaloupe, watermelon, \ <BR>
  #                                  kiwi, mango <BR>
  # The key is "fruits" and the associated element is: 
  # "apple, banana, pear, cantaloupe, watermelon, kiwi, mango"Note that a
  # space appears before each \ so that a space will appear after each comma
  # in the final result; the \, line terminator, and leading white space on
  # the continuation line are merely discarded and are not replaced by one or
  # more other characters. 
  # As a third example, the line: 
  #
  # cheeses <BR>
  # specifies that the key is "cheeses" and the associated element is the
  # empty string "".
  #
  # @param inStream the input stream.
  #
  # @endif
  def load(self, inStream):
    pline = ""
    for readStr in inStream:
      if not readStr:
        continue
      
      tmp = [readStr]
      OpenRTM_aist.eraseHeadBlank(tmp)
      _str = tmp[0]
      
      if _str[0] == "#" or _str[0] == "!" or _str[0] == "\n":
        continue

      _str = _str.rstrip('\r\n')

      if _str[len(_str)-1] == "\\" and not OpenRTM_aist.isEscaped(_str, len(_str)-1):
        #_str = _str[0:len(_str)-1]
        tmp = [_str[0:len(_str)-1]]
        OpenRTM_aist.eraseTailBlank(tmp)
        #pline += _str
        pline += tmp[0]
        continue
      pline += _str
      if pline == "":
        continue

      key = []
      value = []
      self.splitKeyValue(pline, key, value)
      key[0] = OpenRTM_aist.unescape(key)
      OpenRTM_aist.eraseHeadBlank(key)
      OpenRTM_aist.eraseTailBlank(key)

      value[0] = OpenRTM_aist.unescape(value)
      OpenRTM_aist.eraseHeadBlank(value)
      OpenRTM_aist.eraseTailBlank(value)

      self.setProperty(key[0], value[0])
      pline = ""


  ##
  # @if jp
  #
  # @brief ץѥƥꥹȤꤵ줿ȥ꡼¸
  #
  # ץѥƥꥹȤꤵ줿ȥ꡼¸롣
  # Υ᥽åɤ Java Properties ȤθߴΤƤ롣
  # (Ūˤ store ᥽åɤѤƤ롣)
  #
  # @param self
  # @param out ϥȥ꡼
  # @param header ץѥƥꥹȤε 
  #
  # @else
  #
  # @brief Save the properties list to the stream
  #
  # Deprecated. 
  #
  # @param out The output stream
  # @param header A description of the property list
  #
  # @endif
  def save(self, out, header):
    self.store(out, header)
    return


  ##
  # @if jp
  #
  # @brief ץѥƥꥹȤϥȥ꡼¸
  #
  # Properties ơ֥Υץѥƥꥹ (ǤΥڥ) load
  # ᥽åɤȤä Properties ơ֥˥ɤΤŬڤʥեޥåȤ
  # ϥȥ꡼˽񤭹ࡣ 
  #
  # Properties ơ֥Υץѥƥꥹ (ǤΥڥ) load
  # ᥽åɤȤä Properties ơ֥˥ɤΤŬڤʥեޥåȤ
  # ϥȥ꡼˽񤭹ࡣȥ꡼ϡISO 8859-1 ʸ
  # 󥳡ǥ󥰤Ѥƽ񤭹ޤ롣 
  # Properties ơ֥ (¸ߤ) Υǥեȥơ֥뤫
  # ץѥƥϡΥ᥽åɤˤäƤϽ񤭹ޤʤ 
  #
  # header  null ǤʤϡASCII ʸ #header ʸ
  # ӹԶڤʸǽ˽ϥȥ꡼˽񤭹ޤޤΤᡢ
  # header ϼ̥ȤȤƻȤȤǤ롣 
  #
  # ˡASCII ʸ #ߤ (Date  toString ᥽åɤˤä
  # ߻郎ΤƱ) Writer ˤäԶڤ
  # ʤ륳ȹԤ񤭹ޤ롣 
  #
  # ³ơ Properties ơ֥Τ٤ƤΥȥ꤬ 1 ԤĽ񤭽Ф롣
  # ƥȥΥʸASCII ʸ=Ϣʸ󤬽񤭹ޤ롣
  # ʸγʸϡץ󥹤Ȥ褹ɬפ뤫
  # ɤǧ롣ASCII ʸ \֡ԡϤ줾 \\\\
  # \\t\\n \\r Ȥƽ񤭹ޤ롣\\u0020 꾮ʸ
  # \\u007E 礭ʸϡб 16  xxxx Ȥä \\uxxxx Ȥ
  # 񤭹ޤ롣߶ʸǤ񤭶ʸǤʤԶʸϡ
  #  \ դƽ񤭹ޤ롣ͤʸ #!= : ϡ
  # ɬɤ褦ˡ˥åդƽ񤭹ޤ롣 
  #
  # ȥ꤬񤭹ޤ줿Ȥǡϥȥ꡼बեå夵롣
  # ϥȥ꡼ϤΥ᥽åɤȤⳫޤޤȤʤ롣 
  #
  # @param self
  # @param out ϥȥ꡼
  # @param header ץѥƥꥹȤε 
  #
  # @else
  #
  # @brief Stores property list to the output stream
  #
  # Writes this property list (key and element pairs) in this Properties 
  # table to the output stream in a format suitable for loading into a 
  # Properties table using the load method. The stream is written using the 
  # ISO 8859-1 character encoding. 
  #
  # Properties from the defaults table of this Properties table (if any) are
  # not written out by this method. 
  #
  # If the comments argument is not null, then an ASCII # character, the
  # comments string, and a line separator are first written to the output
  # stream. Thus, the comments can serve as an identifying comment. 
  #
  # Next, a comment line is always written, consisting of an ASCII #
  # character, the current date and time (as if produced by the toString
  # method of Date for the current time), and a line separator as generated
  # by the Writer. 
  #
  # Then every entry in this Properties table is written out, one per line.
  # For each entry the key string is written, then an ASCII =, then the
  # associated element string. Each character of the key and element strings
  # is examined to see whether it should be rendered as an escape sequence.
  # The ASCII characters \, tab, form feed, newline, and carriage return are
  # written as \\, \t, \f \n, and \r, respectively. Characters less than
  # \u0020 and characters greater than \u007E are written as \uxxxx for the
  # appropriate hexadecimal value xxxx. For the key, all space characters are
  # written with a preceding \ character. For the element, leading space
  # characters, but not embedded or trailing space characters, are written
  # with a preceding \ character. The key and element characters #, !, =, and
  # : are written with a preceding backslash to ensure that they are properly
  # loaded. 
  #
  # After the entries have been written, the output stream is flushed. The
  # output stream remains open after this method returns. 
  #
  # @param out an output stream.
  # @param header a description of the property list.
  #
  # @endif
  def store(self, out, header):
    out.write("#"+header+"\n")
    self._store(out, "", self)


  #============================================================
  # other util functions
  #============================================================

  ##
  # @if jp
  #
  # @brief ץѥƥΥΥꥹȤ vector ֤
  #
  # ᥤץѥƥꥹȤƱ̾ΥĤʤϡǥեȤ
  # ץѥƥꥹȤˤ̤ΥޤࡢΥץѥƥꥹȤˤ뤹٤
  # ΥΥꥹȤ֤ 
  #
  # @param self
  #
  # @return ץѥƥꥹȤˤ뤹٤ƤΥΥꥹȡ
  #         ǥեȤΥץѥƥꥹȤˤ륭ޤ
  #
  # @else
  #
  # @brief Returns an vector of all the keys in this property
  #
  # Returns an enumeration of all the keys in this property list, including
  # distinct keys in the default property list if a key of the same name has
  # not already been found from the main properties list.
  #
  # @return an vector of all the keys in this property list, including the
  #         keys in the default property list.
  #
  # @endif
  def propertyNames(self):
    names = []
    for leaf in self.leaf:
      self._propertyNames(names, leaf.name, leaf)
    return names


  ##
  # @if jp
  # @brief ץѥƥο
  #
  # ѤߤΥץѥƥ롣
  #
  # @param self
  #
  # @return ץѥƥ
  #
  # @else
  # @brief Get number of Properties
  # @endif
  def size(self):
    return len(self.propertyNames())


  ##
  # @if jp
  # @brief Ρɤ򸡺
  # @else
  # @brief Find node of properties
  # @endif
  # Properties* const Properties::findNode(const std::string& key) const
  def findNode(self, key):
    if not key:
      return None

    keys = []
    self.split(key, '.', keys)
    return self._getNode(keys, 0, self)


  ##
  # @if jp
  # @brief Ρɤ
  #
  # ꤷĥΡɤ롣
  #
  # @param self
  # @param key оݥΡɤΥ
  #
  # @return оݥΡ
  #
  # @else
  # @brief Get node of Properties
  # @endif
  def getNode(self, key):
    if not key:
      return self

    leaf = self.findNode(key)
    if leaf:
      return leaf

    self.createNode(key)
    return self.findNode(key)


  ##
  # @if jp
  # @brief Ρɤ
  #
  # ꤷĿΡɤ롣
  # Ʊ쥭ĥΡɤϿѤߤξˤϥ顼֤
  #
  # @param self
  # @param key ΡɤΥ
  #
  # @return Ρ
  #         ꤷĥΡɤ¸ߤˤfalse
  #
  # @else
  #
  # @endif
  def createNode(self, key):
    if not key:
      return False

    if self.findNode(key):
      return False
    
    self.setProperty(key,"")
    return True


  ##
  # @if jp
  # @brief Ρɤ
  #
  # ꤷ̾Τĥץѥƥ롣
  # ץѥƥ֤
  #
  # @param self
  # @param leaf_name оݥץѥƥ̾
  #
  # @return ץѥƥ
  #
  # @else
  # @brief Get node of Properties
  # @endif
  def removeNode(self, leaf_name):
    len_ = len(self.leaf)
    for i in range(len_):
      idx = (len_ - 1) - i
      if self.leaf[idx].name == leaf_name:
        prop = self.leaf[idx]
        del self.leaf[idx]
        return prop
    return None


  ##
  # @if jp
  # @brief ҥΡɤkey뤫ɤ
  #
  # ꤷĻҥΡɤ¸ߤ뤫ɤǧ롣
  # ¸ߤ硢ҥΡɤ֤
  #
  # @param self
  # @param key ǧоݤΥ
  #
  # @return ҥΡ
  #
  # @else
  # @brief If key exists in the children
  # @endif
  def hasKey(self, key):
    for leaf in self.leaf:
      if leaf.name == key:
        return leaf

    return None


  ##
  # @if jp
  # @brief ҥΡɤƺ
  #
  # @param self
  #
  # @else
  # @brief If key exists in the children
  # @endif
  def clear(self):
    len_ = len(self.leaf)
    for i in range(len_):
      if self.leaf[-1]:
        del self.leaf[-1]

    return


  ##
  # @if jp
  # @brief Propertyޡ
  #
  # ߤΥץѥƥꤷץѥƥޡ롣
  #
  # @param self
  # @param prop ޡץѥƥ
  #
  # @return ץѥƥޡ
  #
  # @else
  # @brief Merge properties
  # @endif
  def mergeProperties(self, prop):
    keys = prop.propertyNames()

    for i in range(prop.size()):
      self.setProperty(keys[i], prop.getProperty(keys[i]))

    return self


  ##
  # @if jp
  # @brief ʸ򥭡ͤΥڥʬ䤹
  #
  # Ϳ줿ʸꤵ줿ǥߥǥͤΥڥʬ䤹롣
  # ޤǽͿ줿ʸ':'⤷'='ޤޤ뤫򸡺
  # ɤ餫ʸޤޤƤˤϤǥߥȤƻѤ롣
  # ξȤޤޤƤʤˤϡ' '(ڡ)Ѥʬߤ롣
  # ƤΥǥߥ䤬ޤޤƤʤˤϡͿ줿ʸ򥭡Ȥ
  # ꤷͤ˶ʸꤹ롣
  # ɤΥǥߥˤĤƤ⥨פƤ(ľ'\'ꤵƤ)
  # ˤϡǥߥȤƻѤʤ
  #
  # @param self
  # @param _str ʬоʸ
  # @param key ʬ̥
  # @param value ʬ
  #
  # @else
  #
  # @endif
  def splitKeyValue(self, _str, key, value):
    i = 0
    length = len(_str)

    while i < length:
      if (_str[i] == ":" or _str[i] == "=") and not OpenRTM_aist.isEscaped(_str, i):
        key.append(_str[0:i])
        value.append(_str[i+1:])
        return
      i += 1

    # If no ':' or '=' exist, ' ' would be delimiter.
    i = 0
    while i < length:
      if (_str[i] == " ") and not OpenRTM_aist.isEscaped(_str, i):
        key.append(_str[0:i])
        value.append(_str[i+1:])
        return
      i += 1

    key.append(_str)
    value.append("")
    return


  ##
  # @if jp
  # @brief ʸʬ䤹
  #
  # Ϳ줿ʸͿ줿ǥߥʬ䤹롣
  # Ϳ줿ʸ󤬶ξϡ顼֤
  # Ϳ줿ǥߥפƤ(ľ'\'ꤵƤ)
  # ˤϡǥߥȤƻѤʤ
  #
  # @param self
  # @param _str ʬоʸ
  # @param delim ǥߥ
  # @param value ʬͥꥹ
  #
  # @return ʬ
  #
  # @else
  #
  # @endif
  def split(self, _str, delim, value):
    if _str == "":
      return False

    begin_it = end_it = 0

    length = len(_str)

    while end_it < length:
      if _str[end_it] == delim and not OpenRTM_aist.isEscaped(_str, end_it):
        value.append(_str[begin_it:end_it])
        begin_it = end_it + 1
      end_it += 1

    value.append(_str[begin_it:end_it])
    return True


  ##
  # @if jp
  # @brief ץѥƥ
  #
  # ꥹȤǻꤵ줿ץѥƥ롣
  # ꥹȤǤϡꤹ륭ΥץѥƥǤγشطꥹȷɽ
  # 롣
  # ꤷꥹȤ˳ץѥƥ¸ߤʤNone֤
  #
  # @param self
  # @param keys оݥץѥƥΥΥꥹɽ
  # @param index ꥹȤγؿ
  # @param curr оݥץѥƥ
  #
  # @return оݥץѥƥ
  #
  # @else
  #
  # @endif
  def _getNode(self, keys, index, curr):
    next = curr.hasKey(keys[index])
    if next is None:
      return None

    if index < (len(keys) - 1):
      index+=1
      return next._getNode(keys, index, next)
    else:
      return next

    return None


  ##
  # @if jp
  # @brief ץѥƥ̾ΥꥹȤ
  #
  # ץѥƥ̾Τ'.'ڤɽꥹȤ롣
  #
  # @param self
  # @param names ץѥƥ̾Υꥹ
  # @param curr_name ߤΥץѥƥ̾
  # @param curr оݥץѥƥ
  #
  # @else
  #
  # @endif
  def _propertyNames(self, names, curr_name, curr):
    if len(curr.leaf) > 0:
      for i in range(len(curr.leaf)):
        next_name = curr_name+"."+curr.leaf[i].name
        self._propertyNames(names, next_name, curr.leaf[i])
    else:
      names.append(curr_name)

    return


  ##
  # @if jp
  # @brief ץѥƥ̾ΥꥹȤ¸
  #
  # ץѥƥ̾Τ'.'ڤɽꥹȤ¸롣
  #
  # @param self
  # @param out ץѥƥ̾Υꥹ¸νϥȥ꡼
  # @param curr_name ߤΥץѥƥ̾
  # @param curr оݥץѥƥ
  #
  # @else
  #
  # @endif
  def _store(self, out, curr_name, curr):
    if len(curr.leaf) > 0:
      for i in range(len(curr.leaf)):
        if curr_name == "":
          next_name = curr.leaf[i].name
        else:
          next_name = curr_name+"."+curr.leaf[i].name
        self._store(out, next_name, curr.leaf[i])
        
    else:
      val = curr.value
      if val == "":
        val = curr.default_value
      out.write(curr_name+": "+val+"\n")

    return


  ##
  # @if jp
  # @brief ǥȤ
  #
  # ꤵ줿˽äǥȤ֤
  # ֤륤ǥȤϡ2Ĥζ
  #
  # @param self
  # @param index ǥȿλ
  #
  # @return 줿ǥ
  #
  # @else
  #
  # @endif
  def indent(self, index):
    space = ""

    for i in range(index-1):
      space += "  "

    return space


  ##
  # @if jp
  # @brief ץѥƥƤ¸
  #
  # ץѥƥꤵ줿Ƥ¸롣
  # ¸ˤϥץѥƥؤοɽղä롣
  # ͤꤵƤʤץѥƥˤĤƤϡǥեͤϤ롣
  #
  # @param self
  # @param out ץѥƥ¸νϥȥ꡼
  # @param curr оݥץѥƥ
  # @param index ߤΥץѥƥ
  #
  # @else
  #
  # @endif
  def _dump(self, out, curr, index):
    if index != 0:
      #ut.write(self.indent(index)+"- "+curr.name)
      out[0]+=self.indent(index)+"- "+curr.name

    if curr.leaf == []:
      if curr.value == "":
        #out.write(": "+curr.default_value+"\n")
        out[0]+=": "+curr.default_value+"\n"
      else:
        #out.write(": "+curr.value+"\n")
        out[0]+=": "+str(curr.value)+"\n"
      return out[0]

    if index != 0:
      #out.write("\n")
      out[0]+="\n"

    for i in range(len(curr.leaf)):
      self._dump(out, curr.leaf[i], index + 1)

    return out[0]


  ##
  # @if jp
  # @brief ץѥƥƤϤ
  #
  # ץѥƥꤵ줿ƤϤ롣<br>
  # friend std::ostream& operator<<(std::ostream& lhs, const Properties& rhs);
  # ˡprint objˤƸƤӽФǽȤ뤿Υ᥽åɡ
  #
  # @param self
  #
  # @return ץѥƥʸɽ
  #
  # @else
  #
  # @endif
  def __str__(self): 
    string=[""]
    return self._dump(string, self, 0)

  
