// -*- C++ -*-
/*!
 * @file PeriodicTask.h
 * @brief PeriodicTask class
 * @date $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2009
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef COIL_PERIODICTASK_H
#define COIL_PERIODICTASK_H

#include <map>
#include <string>
#include <vector>
#include <algorithm>

#include <coil/Mutex.h>
#include <coil/Guard.h>
#include <coil/Condition.h>
#include <coil/TimeValue.h>
#include <coil/TimeMeasure.h>
#include <coil/PeriodicTaskBase.h>

namespace coil
{
  /*!
   * @if jp
   * @class PeriodicTask
   * @brief åɼ¹ԥ饹
   *
   * δؿ¹Ԥ뤿Υåɥ֥Ȥ¸롣
   * Ѽϰʲ̤ꡣ
   *
   * task; // 󥹥
   * task.setTask(TaskFuncBase(obj, mem_func)); // ¹ԤؿͿ
   * task.activate(); // åɤ򥹥Ȥ
   *
   * task.suspend(); // ¹Ԥߤ
   * task.signal(); // 1¹
   * task.resume(); // ¹ԤƳ
   *
   * task.finalize(); // λ
   * 
   * @else
   * @class PeriodicTask
   * @brief PeriodicTask class
   *
   * @endif
   */
  class PeriodicTask
    : public coil::PeriodicTaskBase
  {
  public:
    typedef coil::Guard<coil::Mutex> Guard;

    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *
     * @else
     *
     * @brief Constructor
     *
     * Constructor
     *
     * @endif
     */
    PeriodicTask();
    
    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * Destructor
     *
     * @endif
     */
    virtual ~PeriodicTask();
    
    /*!
     * @if jp
     * @brief ¹Ԥ򳫻Ϥ
     *
     * μ¹Ԥ򳫻Ϥ뤿˥åɤ򥹥Ȥ롣  
     * ˳Ϥ줿 true ֤ꡢǤ˥ϺѤߡޤ
     * ϼ¹Ԥ륿ꤵƤʤ false ֤
     *
     * @return true: ﳫϡfalse: åɺѤߤ̤Ǥ롣
     *
     * @else
     * @brief Starting the task
     *
     * Starting a thread to execute a task.  If the task/thread is
     * started properly, it will return 'TRUE'.  if the task/thread
     * are already started or task function object is not set, 'FALSE'
     * will be returned.
     *
     * @return true: normal start, false: already started  or task is not set
     *
     * @endif
     */
    virtual void activate();

    /*!
     * @if jp
     * @brief ¹Ԥλ
     *
     * ¹Υλ롣
     *
     * @else
     * @brief Finalizing the task
     *
     * Finalizing the task running.
     *
     * @endif
     */
    virtual void finalize();

    /*!
     * @if jp
     * @brief ¹ԤǤ
     *
     * ¹ΥǤ롣
     *
     * @else
     * @brief Suspending the task
     *
     * Suspending the task running.
     *
     * @endif
     */
    virtual int suspend(void);

    /*!
     * @if jp
     * @brief ǤƤ륿Ƴ
     *
     * ǤƤ륿Ƴ
     *
     * @else
     * @brief Resuming the suspended task
     *
     * Resuming the suspended task
     *
     * @endif
     */
    virtual int resume(void);

    /*!
     * @if jp
     * @brief ǤƤ륿1¹Ԥ
     *
     * ǤƤ륿1¹Ԥ
     *
     * @else
     * @brief Executing the suspended task one tick
     *
     * Executing the suspended task one tick
     *
     * @endif
     */
    virtual void signal();

    /*!
     * @if jp
     * @brief ¹Դؿ򥻥åȤ
     *
     * @param func int (*)() δؿݥ
     *
     * @else
     * @brief Setting task execution function
     *
     * @param func Set int (*)() type function pointer
     *
     * @endif
     */
    virtual bool setTask(TaskFuncBase* func, bool delete_in_dtor = true);

    /*!
     * @if jp
     * @brief ¹Դؿ򥻥åȤ
     *
     * @param func int (*)() δؿݥ
     *
     * @return true: , false: 
     *
     * @else
     * @brief Setting task execution function
     *
     * @param func Set int (*)() type function pointer
     *
     * @return true: successful, false: failed
     *
     * @endif
     */
    template <class O, class F>
    bool setTask(O* obj, F fun)
    {
      return this->setTask(new TaskFunc<O, F>(obj, fun));
    }

    /*!
     * @if jp
     * @brief ¹Լ򥻥åȤ
     *
     * @param period ¹Լ [sec]
     *
     * @else
     * @brief Setting task execution period
     *
     * @param period Execution period [sec]
     *
     * @endif
     */
    virtual void setPeriod(double period);

    /*!
     * @if jp
     * @brief ¹Լ򥻥åȤ
     *
     * @param period ¹Լ
     *
     * @else
     * @brief Setting task execution period
     *
     * @param period Execution period
     *
     * @endif
     */
    virtual void setPeriod(TimeValue& period);

    /*!
     * @if jp
     * @brief ؿ¹Իַ¬ͭˤ뤫
     * @else
     * @brief Validate a Task execute time measurement
     * @endif
     */
    virtual void executionMeasure(bool value);
    
    /*!
     * @if jp
     * @brief ؿ¹Իַ¬
     * @else
     * @brief Task execute time measurement period
     * @endif
     */
    virtual void executionMeasureCount(int n);
    
    /*!
     * @if jp
     * @brief ַ¬ͭˤ뤫
     * @else
     * @brief Validate a Task period time measurement
     * @endif
     */
    virtual void periodicMeasure(bool value);
    
    /*!
     * @if jp
     * @brief ַ¬
     * @else
     * @brief Task period time measurement count
     * @endif
     */
    virtual void periodicMeasureCount(int n);
    
    /*!
     * @if jp
     * @brief ؿ¹Իַ¬̤
     * @else
     * @brief Get a result in task execute time measurement
     * @endif
     */
    virtual TimeMeasure::Statistics getExecStat();
    
    /*!
     * @if jp
     * @brief ַ¬̤
     * @else
     * @brief Get a result in task period time measurement
     * @endif
     */
    virtual TimeMeasure::Statistics getPeriodStat();
    
  protected:
    /*!
     * @if jp
     * @brief PeriodicTask ѤΥåɼ¹
     * @else
     * @brief Thread execution for PeriodicTask
     * @endif
     */
    virtual int svc();

    /*!
     * @if jp
     * @brief åɵٻ
     * @else
     * @brief Thread sleep
     * @endif
     */
    virtual void sleep();

    /*!
     * @if jp
     * @brief ¹Ծֹ
     * @else
     * @brief Update for execute state
     * @endif
     */
    virtual void updateExecStat();

    /*!
     * @if jp
     * @brief ֹ
     * @else
     * @brief Update for period state
     * @endif
     */
    virtual void updatePeriodStat();

  protected:
    /*!
     * @if jp
     * @brief ¹Լ
     * @else
     * @brief Task execution period
     * @endif
     */
    coil::TimeValue m_period;

    /*!
     * @if jp
     * @brief åɵٻߥե饰
     * @else
     * @brief Thread sleep flag
     * @endif
     */
    bool m_nowait;

    /*!
     * @if jp
     * @brief ¹Դؿ
     * @else
     * @brief Task execution function
     * @endif
     */
    TaskFuncBase* m_func;

    /*!
     * @if jp
     * @brief ¹Դؿե饰
     * @else
     * @brief Task execution function delete flag
     * @endif
     */
    bool m_deleteInDtor;

    /*!
     * @if jp
     * @class alive_t
     * @brief alive_t 饹
     * @else
     * @class alive_t
     * @brief alive_t class
     * @endif
     */
    class alive_t
    {
    public:
      alive_t(bool val) : value(val) {}
      bool value;
      coil::Mutex mutex;
    };

    /*!
     * @if jp
     * @brief ¸ե饰
     * @else
     * @brief Task alive flag
     * @endif
     */
    alive_t m_alive;

    /*!
     * @if jp
     * @brief Ǵѹ¤
     * @else
     * @brief Structure for task suspend management
     * @endif
     */
    struct suspend_t
    {
      suspend_t(bool sus) : suspend(sus), mutex(), cond(mutex) {}
      bool suspend;
      coil::Mutex mutex;
      coil::Condition<coil::Mutex> cond;
    };

    /*!
     * @if jp
     * @brief Ǿ
     * @else
     * @brief Task suspend infomation
     * @endif
     */
    suspend_t m_suspend;
      
    /*!
     * @if jp
     * @brief ¹Իַ¬ѹ¤
     * @else
     * @brief Structure for task execution time measurement management
     * @endif
     */
    struct statistics_t
    {
      coil::TimeMeasure::Statistics stat;
      coil::Mutex mutex;
    };

    /*!
     * @if jp
     * @brief ¹Իַ¬ե饰
     * @else
     * @brief Task execution time measurement flag
     * @endif
     */
    bool              m_execMeasure;

    /*!
     * @if jp
     * @brief ¹Իַ¬
     * @else
     * @brief Task execution time measurement count
     * @endif
     */
    unsigned int      m_execCount;

    /*!
     * @if jp
     * @brief ¹Իַ¬
     * @else
     * @brief Task execution time measurement max count
     * @endif
     */
    unsigned int      m_execCountMax;

    /*!
     * @if jp
     * @brief ¹Իַ¬
     * @else
     * @brief Task execution time measurement statistics
     * @endif
     */
    statistics_t      m_execStat;

    /*!
     * @if jp
     * @brief ¹Իַ¬
     * @else
     * @brief Task execution time  measurement infomation
     * @endif
     */
    coil::TimeMeasure m_execTime;

    /*!
     * @if jp
     * @brief ַ¬ե饰
     * @else
     * @brief Task periodic time measurement flag
     * @endif
     */
    bool              m_periodMeasure;

    /*!
     * @if jp
     * @brief ַ¬
     * @else
     * @brief Task periodic time measurement count
     * @endif
     */
    unsigned int      m_periodCount;

    /*!
     * @if jp
     * @brief ַ¬
     * @else
     * @brief Task periodic time measurement max count
     * @endif
     */
    unsigned int      m_periodCountMax;

    /*!
     * @if jp
     * @brief ַ¬
     * @else
     * @brief Task periodic time measurement statistics
     * @endif
     */
    statistics_t      m_periodStat;

    /*!
     * @if jp
     * @brief ַ¬
     * @else
     * @brief Task periodic time  measurement infomation
     * @endif
     */
    coil::TimeMeasure m_periodTime;

  };

}; // namespace coil

#endif // COIL_PERIODICTASK_H
