// -*- C++ -*-
/*!
 * @file  Condition_posix.h
 * @brief Condition variable for POSIX
 * @date  $Date$
 * @author Noriaki Ando <n-ando@aist.go.jp>
 *
 * Copyright (C) 2008
 *     Noriaki Ando
 *     Task-intelligence Research Group,
 *     Intelligent Systems Research Institute,
 *     National Institute of
 *         Advanced Industrial Science and Technology (AIST), Japan
 *     All rights reserved.
 *
 * $Id$
 *
 */

#ifndef COIL_CONDITION_H
#define COIL_CONDITION_H

#include <pthread.h>
#include <algorithm>
#include <ctime>
#include <sys/time.h>

namespace coil
{
  /*!
   * @if jp
   *
   * @class Condition
   * @brief Condition ƥץ졼ȥ饹
   *
   * @else
   *
   * @class Condition
   * @brief Condition template class
   *
   * @endif
   */
  template <class M>
  class Condition
  {
  public:

    /*!
     * @if jp
     *
     * @brief 󥹥ȥ饯
     *
     * 󥹥ȥ饯
     *
     * @else
     *
     * @brief Constructor
     *
     * Constructor
     *
     * @endif
     */
    Condition(M& mutex)
      : m_mutex(mutex)
    {
      ::pthread_cond_init(&m_cond, 0);
    }

    /*!
     * @if jp
     *
     * @brief ǥȥ饯
     *
     * ǥȥ饯
     *
     * @else
     *
     * @brief Destructor
     *
     * Destructor
     *
     * @endif
     */
    ~Condition()
    {
      ::pthread_cond_destroy(&m_cond);
    }

    /*!
     * @if jp
     *
     * @brief åɼ¹ԤκƳ
     *
     * ԵƤ륹åɼ¹ԤƳ롣
     *
     * @else
     *
     * @brief Resume of the thread practice
     *
     * Let the practice of a waiting thread resume.
     *
     * @endif
     */
    inline void signal()
    {
      ::pthread_cond_signal(&m_cond);
    }

    /*!
     * @if jp
     *
     * @brief åɼ¹ԤκƳ
     *
     * ԵƤƤΥåɼ¹ԤƳ롣
     *
     * @else
     *
     * @brief Resume of all the thread practice
     *
     * Let all waiting thread practice resume.
     *
     * @endif
     */
    inline void broadcast()
    {
      ::pthread_cond_broadcast(&m_cond);
    }

    /*!
     * @if jp
     *
     * @brief åɼ¹ԤԵ
     *
     * ѿޤǥåɤμ¹Ԥߤ롣
     *
     * @return true: , false: 
     *
     * @else
     *
     * @brief Wait of the thread practice
     *
     * Stop the practice of the thread till a condition variable is transmitted.
     *
     * @return true: successful, false: failed
     *
     * @endif
     */
    bool wait()
    {
      return 0 == ::pthread_cond_wait(&m_cond, &m_mutex.mutex_);
    }

    /*!
     * @if jp
     *
     * @brief ֤Υåɼ¹Ե
     *
     * ꤵ줿֡åɤμ¹Ԥߤ롣
     *
     * @param second ñ̤λ
     * @param nano_second ʥñ̤λ
     *
     * @return true: , false: 
     *
     * @else
     *
     * @brief Thread practice wait of set time
     *
     * In set time, stop the practice of the thread.
     *
     * @param second Time of the seconds.
     * @param nano_second time of the nanoseconds.
     *
     * @return true: successful, false: failed
     *
     * @endif
     */
    bool wait(long second, long nano_second = 0)
    {
      struct timeval tv;
      timespec abstime;

      ::gettimeofday(&tv, NULL);
      abstime.tv_sec  = tv.tv_sec + second;
      abstime.tv_nsec = tv.tv_usec * 1000 + nano_second;
      if (abstime.tv_nsec >= 1000000000) {
        abstime.tv_nsec -= 1000000000;
        abstime.tv_sec ++;
      }
      return 0 == ::pthread_cond_timedwait(&m_cond, &m_mutex.mutex_, &abstime);
    }

  private:
    Condition(const M&);
    Condition& operator=(const M &);
    pthread_cond_t m_cond;
    M& m_mutex;
  };
};
#endif // COIL_CONDITION_H
