/**
 *
 *  \file
 *  \brief      Adapter which allows a single-ended UDP connection to
 *              present the stream interface.
 *  \author     Mike Purvis <mpurvis@clearpathrobotics.com>
 *  \copyright  Copyright (c) 2016, Clearpath Robotics, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *     * Redistributions of source code must retain the above copyright
 *       notice, this list of conditions and the following disclaimer.
 *     * Redistributions in binary form must reproduce the above copyright
 *       notice, this list of conditions and the following disclaimer in the
 *       documentation and/or other materials provided with the distribution.
 *     * Neither the name of Clearpath Robotics, Inc. nor the
 *       names of its contributors may be used to endorse or promote products
 *       derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL CLEARPATH ROBOTICS, INC. BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Please send comments, questions, or patches to code@clearpathrobotics.com
 *
 */

#ifndef ROSSERIAL_SERVER_UDP_STREAM_H
#define ROSSERIAL_SERVER_UDP_STREAM_H

#include <iostream>
#include <boost/bind.hpp>
#include <boost/asio.hpp>

#include <ros/ros.h>

#include "rosserial_server/session.h"


namespace rosserial_server
{

using boost::asio::ip::udp;
using boost::asio::handler_type;


class UdpStream : public udp::socket
{
public:
  explicit UdpStream(boost::asio::io_service& io_service) : udp::socket(io_service)
  {
  }

  void open(udp::endpoint server_endpoint, udp::endpoint client_endpoint)
  {
    boost::system::error_code ec;
    const protocol_type protocol = server_endpoint.protocol();
    this->get_service().open(this->get_implementation(), protocol, ec);
    boost::asio::detail::throw_error(ec, "open");
    this->get_service().bind(this->get_implementation(), server_endpoint, ec);
    boost::asio::detail::throw_error(ec, "bind");

    client_endpoint_ = client_endpoint;
  }

  template <typename ConstBufferSequence, typename WriteHandler>
  BOOST_ASIO_INITFN_RESULT_TYPE(WriteHandler,
      void (boost::system::error_code, std::size_t))
  async_write_some(const ConstBufferSequence& buffers,
      BOOST_ASIO_MOVE_ARG(WriteHandler) handler)
  {
    // If you get an error on the following line it means that your handler does
    // not meet the documented type requirements for a WriteHandler.
    BOOST_ASIO_WRITE_HANDLER_CHECK(WriteHandler, handler) type_check;
#if (BOOST_VERSION >= 106600)
    // See: http://www.boost.org/doc/libs/1_66_0/doc/html/boost_asio/net_ts.html
    boost::asio::async_completion<WriteHandler,
      void (boost::system::error_code, std::size_t)> init(handler);

    this->get_service().async_send_to(
        this->get_implementation(), buffers, client_endpoint_, 0,
        init.completion_handler);

    return init.result.get();
#else
    return this->get_service().async_send_to(
        this->get_implementation(), buffers, client_endpoint_, 0,
        BOOST_ASIO_MOVE_CAST(WriteHandler)(handler));
#endif
  }

  template <typename MutableBufferSequence, typename ReadHandler>
  BOOST_ASIO_INITFN_RESULT_TYPE(ReadHandler,
      void (boost::system::error_code, std::size_t))
  async_read_some(const MutableBufferSequence& buffers,
      BOOST_ASIO_MOVE_ARG(ReadHandler) handler)
  {
    // If you get an error on the following line it means that your handler does
    // not meet the documented type requirements for a ReadHandler.
    BOOST_ASIO_READ_HANDLER_CHECK(ReadHandler, handler) type_check;
#if (BOOST_VERSION >= 106600)
    // See: http://www.boost.org/doc/libs/1_66_0/doc/html/boost_asio/net_ts.html
    boost::asio::async_completion<ReadHandler,
      void (boost::system::error_code, std::size_t)> init(handler);

    this->get_service().async_receive(this->get_implementation(),
        buffers, 0, init.completion_handler);

    return init.result.get();
#else
    return this->get_service().async_receive_from(
        this->get_implementation(), buffers, client_endpoint_, 0,
        BOOST_ASIO_MOVE_CAST(ReadHandler)(handler));
#endif
  }

private:
  udp::endpoint client_endpoint_;
};

}  // namespace

#endif  // ROSSERIAL_SERVER_UDP_STREAM_H
