/** **************************************************************************************
 * @file       example_common.c
 * @ingroup    Toposens Sensor Library
 * @copyright  Copyright (c) Toposens GmbH 2021. All rights reserved.
 ******************************************************************************************/

#include "example_common.h"
#include <stdio.h>

void GetSettings()
{
  // ADC settings
  printf("\n\nADC settings:\n");
  printf("GetParameterADCUseFixedFrameRate_b: %d\n", GetParameterADCUseFixedFrameRate_b());
  printf("GetParameterADCFixedFrameRate_u8: %d\n", GetParameterADCFixedFrameRate_u8());

  // Transducer settings
  printf("\n\nTransducer settings:\n");
  printf("GetParameterTransducerVolume_u8: %d\n", GetParameterTransducerVolume_u8());
  printf("GetParameterTransducerNumOfPulses_u8: %d\n", GetParameterTransducerNumOfPulses_u8());

  // Signal processing settings
  printf("\n\nSignal processing settings:\n");
  printf("GetParameterSignalProcessingTemperature_f: %.6f\n",
         GetParameterSignalProcessingTemperature_f());
  printf("GetParameterSignalProcessingHumidity_u8: %d\n",
         GetParameterSignalProcessingHumidity_u8());
  printf("GetParameterSignalProcessingNoiseLevelThresholdFactor_f: %f\n",
         GetParameterSignalProcessingNoiseLevelThresholdFactor_f());
  printf("GetParameterSignalProcessingNoiseRatioThreshold_u8: %d\n",
         GetParameterSignalProcessingNoiseRatioThreshold_u8());
  printf("GetParameterSignalProcessingEnableNearFieldDetection_b: %d\n",
         GetParameterSignalProcessingEnableNearFieldDetection_b());
  printf("GetParameterSignalProcessingEnableMultipathFilter_b: %d\n",
         GetParameterSignalProcessingEnableMultipathFilter_b());
  printf("GetParameterSignalProcessingEnableAutoGain_b: %d\n",
         GetParameterSignalProcessingEnableAutoGain_b());

  // System settings
  printf("\n\nSystem settings:\n");
  printf("GetParameterSystemNodeID_u16: 0x%02X\n", GetParameterSystemNodeID_u16());
  printf("GetParameterSystemMCUTemperature_f:  %.6f\n", GetParameterSystemMCUTemperature_f());
  printf("GetParameterSystemLogLevel_t: %d\n", GetParameterSystemLogLevel_t());
  printf("GetParameterSystemResetReason_t: %d\n", GetParameterSystemResetReason_t());
  printf("GetParameterSystemSensorState_t: %d\n", GetParameterSystemSensorState_t());
  printf("GetParameterSystemUniqueID_t:");

  for (int Index = 0; Index < 3; Index++)
  {
    UID_t UniqueID_t = GetParameterSystemUniqueID_t(Index);
    for (int i = 0; i < 4; i++)
    {
      printf(" %02X", UniqueID_t.values[i]);
    }
  }
  printf("\nGetParameterSystemSensorMode_t: %d\n", GetParameterSystemSensorMode_t());
  printf("GetParameterSystemCalibrationState_b: %d\n", GetParameterSystemCalibrationState_b());
}

void PrintVersion(Version_t Version_t)
{
  printf("%d.%d.%d\n", Version_t.major, Version_t.minor, Version_t.hotfix);
}

void GetVersions()
{
  Version_t Version;
  printf("Bootloader version: ");
  Version = RequestVersion_t(VERSION_BYTE_BOOTLOADER);
  PrintVersion(Version);
  printf("Application version: ");
  Version = RequestVersion_t(VERSION_BYTE_APP);
  PrintVersion(Version);
  printf("Hardware version: ");
  Version = RequestVersion_t(VERSION_BYTE_HW);
  PrintVersion(Version);
  printf("Signal Processing Library version: ");
  Version = RequestVersion_t(VERSION_BYTE_SIG_PRO_LIB);
  PrintVersion(Version);
  printf("Communication Library version: ");
  Version = RequestVersion_t(VERSION_BYTE_COMMS_LIB);
  PrintVersion(Version);
}

void PrintSessionData(Sensor_Session_t* Session)
{
  if (Session != NULL)
  {
    printf("---------------------------\n");
    printf("SessionState:%d\n", Session->SessionState);
    printf("SenderId_u16:%d\n", Session->SenderId_u16);
    printf("NumberOfPoints_u8:%d\n", Session->NumberOfPoints_u8);
    printf("NoiseLevel_u16:%d\n", Session->NoiseLevel_u16);
    printf("NearFieldPoint_b:%d\n", Session->NearFieldPoint_b);
    printf("NumberOf1DPoints:%d\n", Session->NumberOf1DPoints);
    printf("NumberOf3DPoints:%d\n", Session->NumberOf3DPoints);
    printf("1-D Points: \n");
    for (int i = 0; i < Session->NumberOf1DPoints; i++)
    {
      printf(" 1D-Point-%d VectorLength: %d\n", i, Session->Point1D_tp[i].VectorLength_u16);
      printf(" 1D-Point-%d Intensity_u8: %d\n", i, Session->Point1D_tp[i].Intensity_u8);
    }
    printf("3-D Points: \n");
    for (int i = 0; i < Session->NumberOf3DPoints; i++)
    {
      printf(" 3D-Point-%d X_i16: %d\n", i, Session->Point3D_tp[i].X_i16);
      printf(" 3D-Point-%d Y_i16: %d\n", i, Session->Point3D_tp[i].Y_i16);
      printf(" 3D-Point-%d Z_i16: %d\n", i, Session->Point3D_tp[i].Z_i16);
      printf(" 3D-Point-%d Intensity_u8: %d\n", i, Session->Point3D_tp[i].Intensity_u8);
    }
    printf("---------------------------\n");
  }
}

void PrintADCBlobData(ADCDump_t* ADCDump)
{
  uint32_t DumpSize_u32 = ADCDump->ReceivedDumpSize_u32;
  printf("ADC-Session closed\n");
  printf("ReceivedDumpSize_u32:%d\n", DumpSize_u32);
  for (uint32_t Idx_u32 = 0; Idx_u32 < DumpSize_u32; Idx_u32++)
  {
    printf("%02X ", ADCDump->DumpBlob_pu8[Idx_u32]);
    if (Idx_u32 % 48 == 0)
    {
      printf("\n");
    }
  }
  printf("\n");
}

void SaveADCBlobData(ADCDump_t* ADCDump, const char* filepath)
{
  uint32_t DumpSize_u32 = ADCDump->ReceivedDumpSize_u32;

  FILE* fptr;
  fptr = fopen(filepath, "wb");

  if (fptr == NULL)
  {
    printf("\033[0;31mADC Dump file open error!\n\033[0m");
    return;
  }

  size_t number_of_bytes_written = fwrite(ADCDump->DumpBlob_pu8, 1, DumpSize_u32, fptr);

  if (number_of_bytes_written == DumpSize_u32)
    printf("Save sucessful\n");
  else
    printf("\033[0;31mADC Dump file write error!\n\033[0m");

  fclose(fptr);
}

void PrintLogMessage(uint16_t SenderId_u16, const uint8_t* ReceivedPayload_pu8)
{
  char message_buffer[MAX_PARSED_LOG_MESSAGE_LEN];
  ParseLogMessageToText(message_buffer, SenderId_u16, ReceivedPayload_pu8);
  switch (ReceivedPayload_pu8[LOGGING_SEVERITY_BYTE])
  {
    case CONTROL_BYTE_LOG_DEBUG:
      printf("\033[0;34m");
      break;
    case CONTROL_BYTE_LOG_INFO:
      printf("\033[0;32m");
      break;
    case CONTROL_BYTE_LOG_WARN:
      printf("\033[0;33m");
      break;
    case CONTROL_BYTE_LOG_ERROR:
      printf("\033[0;31m");
      break;
    default:
      break;
  }
  printf("%s\n", message_buffer);
  printf("\033[0m");
}

void PrintSensorsOnBus()
{
  Sensor_t* KnownSensors_pu16 = GetKnownSensors();
  uint8_t NumberOfSensors_u8 = GetNumberOfKnownSensors();
  for (int i = 0; i < NumberOfSensors_u8; i++)
  {
    printf("Sensor %d has ID: %02X\n", i, KnownSensors_pu16[i].InterfaceSensorId_u16);
  }
}
