/** **************************************************************************************
 * @file       message_flags.h
 * @ingroup    Toposens Sensor Library
 * @copyright  Copyright (c) Toposens GmbH 2021. All rights reserved.
 * @brief      Contains various enums and structs used to facilitate communication between Toposens
 *sensors and other devices
 ******************************************************************************************/

#ifndef TOPOSENS_MESSAGE_FLAGS_H
#define TOPOSENS_MESSAGE_FLAGS_H

#include <stdint.h>

// CAN frame byte offsets
#define CAN_MAX_FRAME_LEN 8
#define UART_MAX_FRAME_LEN 255
#define CONTROL_BYTE_IDX 0
#define SUBCONTROL_BYTE_IDX 1
#define PARAM_BYTE_1_IDX 2
#define PARAM_BYTE_2_IDX 3
#define PARAM_BYTE_3_IDX 4
#define PARAM_BYTE_4_IDX 5
#define PARAM_BYTE_5_IDX 6
#define PARAM_BYTE_6_IDX 7

#define VERSION_REPLY_LEN 5
#define GENERAL_NACK_LEN 4
#define GENERAL_ACK_LEN 3
#define GET_NACK_LEN 4
#define SET_ACK_LEN 4
#define TRIGGER_ACK_LEN 3
#define READY_NOTIFICATION_LEN 3
#define TRIGGER_LEN 2

#define START_POINT_SESSION_LEN 3
#define START_ADC_DUMP_SESSION_LEN 6
#define START_POINT_SESSION_ACK_LEN 4
#define START_ADC_DUMP_SESSION_ACK_LEN 7
#define EOS_ACK_LEN 2
#define NOISE_LEVEL_FRAME_LEN 3
#define X_OFFSET 1
#define Y_OFFSET 3
#define Z_OFFSET 5
#define INTENSITY_3D_OFFSET 7
#define INTENSITY_1D_OFFSET 3
#define DIST_1D_OFFSET 1
#define POINT_FRAME_3D_LEN 8
#define POINT_FRAME_1D_LEN 4
#define ADC_FRAME_COUNT_OFFSET 1
#define ADC_FRAME_CH_1_OFFSET 2
#define ADC_FRAME_CH_2_OFFSET 4
#define ADC_FRAME_CH_3_OFFSET 6
#define ADC_FRAME_LEN 8
#define EOS_FRAME_LEN 1
#define STANDARD_GET_MESSAGE_LEN 3
#define INDEXED_GET_MESSAGE_LEN 4

#define INIT_APP_UPDATE_LEN 46
#define APP_PACKET_LEN 142
#define COMPLETE_ACK_UPDATE_LEN 2

#define UART_APP_PACKET_DATA_LEN 128
#define USB_APP_PACKET_DATA_LEN 47
#define LOG_STRING_OVERHEAD_BYTES 3
#define MAX_UART_MESSAGE_LEN 255
#define MAX_LOG_STRING_LEN (255 - LOG_STRING_OVERHEAD_BYTES)
#define FW_ACTION_NACK_LEN 2
#define FW_ACTION_ACK_LEN 2
#define MAX_APP_SIZE 262144

#define ENCRYPTED_SECRET_KEY_LEN 48

#define LOGGING_SEVERITY_BYTE 0
#define LOGGING_CATEGORY_BYTE 1
#define LOGGING_ISSUE_BYTE_1 2
#define LOGGING_ISSUE_BYTE_2 3
#define LOGGING_ISSUE_BYTE_3 4

typedef enum ControlByte_t
{
  // General
  CONTROL_BYTE_EOS = 0x00,
  CONTROL_BYTE_ACK = 0x01,
  CONTROL_BYTE_NACK = 0x02,
  CONTROL_BYTE_READY = 0x03,

  // Specific sessions
  CONTROL_BYTE_BEGIN_POINT_OUTPUT_SESSION = 0x10,
  CONTROL_BYTE_BEGIN_ADC_DUMP_SESSION = 0x20,

  // Action triggers
  CONTROL_BYTE_TRIGGER_ACTION = 0x30,
  CONTROL_BYTE_TRIGGER_ACTION_ACK = 0x31,

  // FW update
  CONTROL_BYTE_FW_UPDATE_ACTION = 0x40,
  CONTROL_BYTE_FW_UPDATE_ACTION_NACK = 0x41,
  CONTROL_BYTE_FW_UPDATE_ACTION_ACK = 0x42,

  // Set params
  CONTROL_BYTE_SET = 0x60,
  CONTROL_BYTE_SET_ACK = 0x61,

  // Get params
  CONTROL_BYTE_GET = 0x90,
  CONTROL_BYTE_GET_REPLY = 0x91,
  CONTROL_BYTE_GET_NACK = 0x92,

  // Get versions
  CONTROL_BYTE_VERSION = 0xB0,
  CONTROL_BYTE_GET_VERSION_REPLY = 0xB1,

  // Logging
  CONTROL_BYTE_LOG_DEBUG = 0XA0,
  CONTROL_BYTE_LOG_INFO = 0XA1,
  CONTROL_BYTE_LOG_WARN = 0XA2,
  CONTROL_BYTE_LOG_ERROR = 0XA3,

} ControlByte_t;

typedef enum ParamGroupByte_t
{
  PARAM_GROUP_BYTE_ADC = 0x00,
  PARAM_GROUP_BYTE_TRANSDUCER = 0x01,
  PARAM_GROUP_BYTE_SIGNAL_PROCESSING = 0x02,
  PARAM_GROUP_BYTE_SYSTEM = 0x03
} ParamGroupByte_t;

typedef enum ActionByte_t
{
  ACTION_BYTE_REBOOT = 0x00,
  ACTION_BYTE_MEASUREMENT = 0x01,
  ACTION_BYTE_FACTORY_DEFAULTS = 0x02,
  ACTION_BYTE_STORE_SETTINGS = 0x03,
  ACTION_BYTE_LOAD_SETTINGS = 0x04,
  ACTION_BYTE_ADC_DUMP = 0x05,
} ActionByte_t;

static const char* kActionCommands[] = {"Perform Reboot",        "Perform Measurement",
                                        "Perform Factory Reset", "Store Settings",
                                        "Load Settings",         "Perform ADC Dump"};

typedef enum VersionByte_t
{
  VERSION_BYTE_BOOTLOADER = 0x00,
  VERSION_BYTE_APP = 0x01,
  VERSION_BYTE_HW = 0x02,
  VERSION_BYTE_SIG_PRO_LIB = 0x03,
  VERSION_BYTE_COMMS_LIB = 0x04,
} VersionByte_t;

static const char* kVersionStrings[] = {"Bootloader Version", "Application Version",
                                        "Hardware Version", "Signal Processing Library Version",
                                        "Communication Library Version"};

typedef enum PointTypeByte_t
{
  POINT_TYPE_BYTE_3D = 0x11,
  POINT_TYPE_BYTE_NEAR_FIELD = 0x12,
  POINT_TYPE_BYTE_NOISE_LEVEL = 0x13,
  POINT_TYPE_BYTE_1D = 0x14,
} PointTypeByte_t;

typedef enum ADCFrameTypeByte_t
{
  ADC_FRAME_TYPE_DATA = 0x21,
} ADCFrameTypeByte_t;

typedef enum FirmwareUploadActionByte_t
{
  FW_UPLOAD_ACTION_BYTE_INIT = 0x00,
  FW_UPLOAD_ACTION_BYTE_BLOB_IN = 0x01,
  FW_UPLOAD_ACTION_BYTE_COMPLETE = 0x02
} FirmwareUploadActionByte_t;

static const char* kFirmwareUploadActionStrings[] = {"Init FW Upload", "App Packet",
                                                     "Complete Update"};

typedef enum ACKStausByte_t
{
  ACK_STATUS_BYTE_SUCCESS = 0x00,
  ACK_STATUS_BYTE_OUT_OF_RANGE = 0x01,
  ACK_STATUS_BYTE_OPERATION_INVALID = 0x02,
  ACK_STATUS_BYTE_FALLBACK = 0x03
} ACKStausByte_t;

static const char* kACKStatusStrings[] = {"successfully acknowledged",
                                          "negatively acknowledged (Parameter out of range)",
                                          "negatively acknowledged (Invalid operation)",
                                          "negatively acknowledged (Falling back to default)"};

typedef enum ADCParamByte_t
{
  ADC_PARAM_BYTE_USE_FIXED_FRAME_RATE = 0x00,
  ADC_PARAM_BYTE_FRAME_RATE = 0x01
} ADCParamByte_t;

typedef enum TransducerParamByte_t
{
  TRANSDUCER_PARAM_BYTE_VOLUME = 0x00,
  TRANSDUCER_PARAM_BYTE_NUMBER_PULSES = 0x01,
} TransducerParamByte_t;

typedef enum SignalProcessingParamByte_t
{
  SIGPRO_PARAM_BYTE_TEMPERATURE = 0x01,
  SIGPRO_PARAM_BYTE_RELATIVE_HUMIDITY = 0x02,
  SIGPRO_PARAM_BYTE_NOISE_LEVEL_THRESHOLD_FACTOR = 0x03,
  SIGPRO_PARAM_BYTE_NOISE_RATIO_THRESHOLD = 0x04,
  SIGPRO_PARAM_BYTE_ENABLE_NEAR_FIELD_DETECTION = 0x05,
  SIGPRO_PARAM_BYTE_ENABLE_DIRECT_MULTIPATH_FILTER = 0x06,
  SIGPRO_PARAM_BYTE_ENABLE_AUTO_GAIN = 0x07,
} SignalProcessingParamByte_t;

typedef enum SystemParamByte_t
{
  SYS_PARAM_BYTE_NODE_ID = 0x00,
  SYS_PARAM_BYTE_MCU_TEMPERATURE = 0x01,
  SYS_PARAM_BYTE_LOG_LEVEL = 0x02,
  SYS_PARAM_BYTE_RESET_REASON = 0x03,
  SYS_PARAM_BYTE_SENSOR_STATE = 0x04,
  SYS_PARAM_BYTE_SENSOR_MODE = 0x05,
  SYS_PARAM_BYTE_UNIQUE_ID = 0x06,
  SYS_PARAM_BYTE_CALIBRATION_STATE = 0x07,
} SystemParamByte_t;

#endif  // TOPOSENS_MESSAGE_FLAGS_H
