/** **************************************************************************************
 * @file       sensor_lib.h
 * @ingroup    Toposens Sensor Library
 * @copyright  Copyright (c) Toposens GmbH 2021. All rights reserved.
 * @brief      This file contains the highlevel interface prototypes to the low-level Protocol.
 ******************************************************************************************/

#ifndef TOPOSENS_SENSOR_LIB_H /* include control, avoid multiple includes of header file */
#define TOPOSENS_SENSOR_LIB_H

#ifdef __cplusplus
extern "C"
{
#endif

/*---- "library includes" -------------------------------------------------------------*/

/*---- "project includes" -------------------------------------------------------------*/
#include "config/sensor_lib_config.h"
#include "custom_structs.h"
#include "custom_types.h"
#include "message_flags.h"

/*---- <system includes> --------------------------------------------------------------*/
#include <stdbool.h>
#include <stdint.h>

/*---- public macros and constants ----------------------------------------------------*/
#define UID_SIZE 4
#define MAX_PARSED_LOG_MESSAGE_LEN 200
#define TIMEOUT_M_SECONDS 200000

  /*---- public types -------------------------------------------------------------------*/
  typedef struct UID_t
  {
    uint8_t values[UID_SIZE];
  } UID_t;

  typedef struct Point3D_t
  {
    int16_t X_i16;
    int16_t Y_i16;
    int16_t Z_i16;
    uint8_t Intensity_u8;
  } Point3D_t;

  typedef struct Point1D_t
  {
    uint16_t VectorLength_u16;
    uint8_t Intensity_u8;
  } Point1D_t;

  typedef enum SessionState_t
  {
    INIT,
    EXPECTED,
    RUNNING,
    COMPLETE,
  } SessionState_t;

  typedef struct Sensor_Session_t
  {
    SessionState_t SessionState;
    uint16_t SenderId_u16;
    uint8_t NumberOfPoints_u8;
    uint16_t NoiseLevel_u16;
    bool NearFieldPoint_b;
    uint8_t NumberOf1DPoints;
    uint8_t NumberOf3DPoints;
    Point1D_t Point1D_tp[MAX_NUMBER_OF_POINTS_PER_SESSION];
    Point3D_t Point3D_tp[MAX_NUMBER_OF_POINTS_PER_SESSION];
  } Sensor_Session_t;

  typedef struct ADCDump_t
  {
    SessionState_t SessionState;
    uint16_t SenderId_u16;
    uint32_t ExpectedDumpSize_u32;
    uint32_t ReceivedDumpSize_u32;
    uint8_t* DumpBlob_pu8;
  } ADCDump_t;

  typedef struct Sensor_t
  {
    uint16_t InterfaceSensorId_u16;
    uint8_t InterfaceIndex_u8;
    enum SensorMode_t SensorMode;
  } Sensor_t;

  /**
   * A structure to store ACK-Status Data
   */
  typedef struct ACK_Status_t
  {
    uint16_t SenderId_u16; /**< Sender Id */
    bool WaitForACK_b;     /**< Wait for ACK-Status will turn false if responce was received */
    bool ACK_b;            /**< ACK-Status */
    uint8_t ResponsePayload_pu8[CAN_MAX_FRAME_LEN]; /**< Response Payload that was received from
                                                       this sender */
  } ACK_Status_t;

  /*---- public variables prototypes ----------------------------------------------------*/

  /*---- public functions prototypes ----------------------------------------------------*/

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Returns pointer to array containing all known sensors. With
  GetNumberOfKnownSensors() users can get length of array.
    @return       Sensor_t* Pointer to array containing all currently known Sensors on Bus
  *******************************************************************************/
  Sensor_t* GetKnownSensors();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Returns number of known sensors. With GetKnownSensors() users an pointer to an
  array containing the ID's of the sensors.
    @return       uint8_t Number of currently known sensors on Bus
  *******************************************************************************/
  uint8_t GetNumberOfKnownSensors();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        With this function, the user can set the current target-id, all comming messages
  will be send to
    @param[in]    TargetSensor_u16 Target Id
  *******************************************************************************/
  void SetTargetSensor(uint16_t TargetSensor_u16);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking - Request transmition of the current version of selected component from
  selected sensor
    @param[in]    TargetComponent_t Component whose version number is required
    @return       Version_t Received version
  *******************************************************************************/
  Version_t RequestVersion_t(VersionByte_t TargetComponent_t);

  // Actions

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Request selected sensor to reboot
    @return       bool true if ACK
                  bool false in case of NACK / unexpected
  CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
  *******************************************************************************/
  bool RequestReboot();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Send's a request to start a new measurement
    @return       bool true if ACK
                  bool false in case of NACK / unexpected
  CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
  *******************************************************************************/
  bool RequestMeasurement();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Request selected sensor to reset to factory defaults
    @return       bool true if ACK
                  bool false in case of NACK / unexpected
  CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
  *******************************************************************************/
  bool RequestFactoryReset();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Request selected sensor to store current settings
    @return       bool true if ACK
                  bool false in case of NACK / unexpected
  CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
  *******************************************************************************/
  bool RequestStoreSettings();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Request selected sensor to load saved settings
    @return       bool true if ACK
                  bool false in case of NACK / unexpected
  CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
  *******************************************************************************/
  bool RequestLoadSettings();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Request selected sensor to create an ADC-Dump
    @return       bool true if ACK
                  bool false in case of NACK / unexpected
  CurrentACKStatus[REQUEST_RESPONSE_INDEX].ResponsePayload_pu8
  *******************************************************************************/
  bool RequestADCDump();

  // Setter

  // ADC

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to use the fixed framerate for current target sensor
    @param[in]    UseFixedFrameRate_b
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterADCUseFixedFrameRate(bool UseFixedFrameRate_b);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the fixed framerate for current target sensor
    @param[in]    FrameRate_u8
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterADCFixedFrameRate(uint8_t FrameRate_u8);

  // Transducer

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Transducer Volume for current target sensor
    @param[in]    Volume_u8
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterTransducerVolume(uint8_t Volume_u8);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Number of Transducer Pulses for current target sensor
    @param[in]    NumOfPulses_u8
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterTransducerNumOfPulses(uint8_t NumOfPulses_u8);

  // SignalProcessing

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Temperature for current target sensor
    @param[in]    Temperature_f
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSignalProcessingTemperature(float Temperature_f);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Humidity for current target sensor
    @param[in]    Humidity_u8
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSignalProcessingHumidity(uint8_t Humidity_u8);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Noise Level Threshold Factor for current target sensor
    @param[in]    Factor_f
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSignalProcessingNoiseLevelThresholdFactor(float Factor_f);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Noise Ratio Threshold for current target sensor
    @param[in]    Threshold_u8
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSignalProcessingNoiseRatioThreshold(uint8_t Threshold_u8);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to enable near field detection for current target sensor
    @param[in]    Enable_b
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSignalProcessingEnableNearFieldDetection(bool Enable_b);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to enable near the multipath filter for current target sensor
    @param[in]    Enable_b
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSignalProcessingEnableMultipathFilter(bool Enable_b);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to enable auto gain for current target sensor
    @param[in]    Enable_b
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSignalProcessingEnableAutoGain(bool Enable_b);

  // System

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set a new NodeID for current target sensor
    @param[in]    NodeID_u16
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSystemNodeID(uint16_t NodeID_u16);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Loglevel for current target sensor
    @param[in]    LogLevel_t LogLevel_t
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSystemLogLevel(LogLevel_t LogLevel_t);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to set the Sensor Mode for current target sensor
    @param[in]    Mode_t
    @return       bool true if setting was successful
                  bool false in case of an error
  *******************************************************************************/
  bool SetParameterSystemSensorMode(SensorMode_t Mode_t);

  // Getter

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the "Use fixed framerate" value from current sensor
    @return       bool true if fixed framerate is used
  *******************************************************************************/
  bool GetParameterADCUseFixedFrameRate_b();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current set ADC-Framerate from current sensor
    @return       uint8_t framerate
  *******************************************************************************/
  uint8_t GetParameterADCFixedFrameRate_u8();

  // Transducer

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current Transducer Volume from current sensor
    @return       uint8_t current Transducer Volume
  *******************************************************************************/
  uint8_t GetParameterTransducerVolume_u8();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current Transducer Number of Pulses from current
  sensor
    @return       uint8_t Transducer Number of Pulses
  *******************************************************************************/
  uint8_t GetParameterTransducerNumOfPulses_u8();

  // SignalProcessing

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current Transducer Temperature setting from current
  sensor
    @return       float Set Temperature
  *******************************************************************************/
  float GetParameterSignalProcessingTemperature_f();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current Transducer Humidity setting from current
  sensor
    @return       uint8_t Set Humidity
  *******************************************************************************/
  uint8_t GetParameterSignalProcessingHumidity_u8();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current Noise Level Threshold Factor from current
  sensor
    @return       float Noise Level Threshold Factor
  *******************************************************************************/
  float GetParameterSignalProcessingNoiseLevelThresholdFactor_f();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current Noise Ratio Threshold from current sensor
    @return       uint8_t Noise Ratio Threshold
  *******************************************************************************/
  uint8_t GetParameterSignalProcessingNoiseRatioThreshold_u8();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read whether near field detection is enabled in the current
  sensor
    @return       bool Enabled status
  *******************************************************************************/
  bool GetParameterSignalProcessingEnableNearFieldDetection_b();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read whether the multipath filter is enabled in the current
  sensor
    @return       bool Enabled status
  *******************************************************************************/
  bool GetParameterSignalProcessingEnableMultipathFilter_b();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read whether auto gain is enabled in the current sensor
    @return       bool Enabled status
  *******************************************************************************/
  bool GetParameterSignalProcessingEnableAutoGain_b();

  // System

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current NodeID setting from current sensor
    @return       uint16_t NodeID
  *******************************************************************************/
  uint16_t GetParameterSystemNodeID_u16();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current MCU Temperature from current sensor
    @return       float Temperature
  *******************************************************************************/
  float GetParameterSystemMCUTemperature_f();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the current System Log Level from current sensor
    @return       LogLevel_t SystemLogLevel
  *******************************************************************************/
  LogLevel_t GetParameterSystemLogLevel_t();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the last Reset Reason from current sensor
    @return       ResetReason_t ResetReason
  *******************************************************************************/
  ResetReason_t GetParameterSystemResetReason_t();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the Sensor State from current sensor
    @return       SensorState_t Sensor State
  *******************************************************************************/
  SensorState_t GetParameterSystemSensorState_t();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the Unique ID from current sensor
    @return       UID_t Unique ID
  *******************************************************************************/
  UID_t GetParameterSystemUniqueID_t(uint8_t Index_u8);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read the Sensor Mode from current sensor
    @return       SensorMode_t Sensor Mode
  *******************************************************************************/
  SensorMode_t GetParameterSystemSensorMode_t();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Request to read whether the current sensor has been calibrated
    @return       bool Calibration status
  *******************************************************************************/
  bool GetParameterSystemCalibrationState_b();

  // Sessions

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Function call to wait for SessionStart Payload
    @return       uint16_t SenderId
  *******************************************************************************/
  uint16_t WaitForSessionStart();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Function call to wait for SessionEnd Payload
    @return       uint16_t SenderId
  *******************************************************************************/
  uint16_t WaitForSessionEnd();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Function call to wait for SessionEnd Payload
    @return       uint16_t SenderId
  *******************************************************************************/
  uint16_t WaitForADCSessionEnd();

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to query current session-data (ongoing and closed session)
    @param[in]    Sender_u16 Sender-Id the suer wants to get current session-data from
    @return       Sensor_Session_t*
  *******************************************************************************/
  Sensor_Session_t* GetSessionData(uint16_t Sender_u16);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to get current ADC-Dump from SenderId
    @param[in]    Sender_u16
    @return       ADCDump_t* ADC-Dump that belongs to SenderId. NULL if not found.
  *******************************************************************************/
  ADCDump_t* GetADCDumpData(uint16_t Sender_u16);

  // Ready

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Blocking Function call to wait for Ready Payload
    @return       uint16_t SenderId
  *******************************************************************************/
  uint16_t WaitForReady();

  // Connections

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Instruction to initialize the specified interface with the specified bit rate.
  Will also initialize the semaphores if necessary.
    @param[in]    *InterfaceName name of the linux /dev/  device that should be used for
  communication
    @param[in]    DataRate_u32 Datarate the device should use. 1000000 for example on CAN Bus
    @param[in]    Interface_t The kind of interface to be used. Attention! A wrong interfacename
  might lead to unexpected system behavior!
  *******************************************************************************/
  void InitInterface(char* InterfaceName, uint32_t DataRate_u32, CommsInterfaceType_t Interface_t);

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Instruction to deinit the specified interface type.
    @param[in]    Interface_t The kind of interface to be deinit.
  *******************************************************************************/
  void DeinitInterface(CommsInterfaceType_t Interface_t);

  // Callbacks

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to register a callback function that is called whenever an
  SessionStart Payload is received.
    @param[in]    Callback function pointer
  *******************************************************************************/
  void RegisterSessionStartCallback(void (*Callback)(uint16_t Sender_u16));

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to register a callback function that is called whenever an
  SessionEnd Payload is received.
    @param[in]    Callback function pointer
  *******************************************************************************/
  void RegisterPointSessionEndCallback(void (*Callback)(uint16_t Sender_u16));

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to register a callback function that is called whenever an
  SessionEnd Payload for an active ADCDump is received.
    @param[in]    Callback function pointer
  *******************************************************************************/
  void RegisterADCDumpSessionEndCallback(void (*Callback)(uint16_t Sender_u16));

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to register a callback function that is called whenever an
  Ready Payload is received.
    @param[in]    Callback function pointer
  *******************************************************************************/
  void RegisterRdyCallback(void (*Callback)(uint16_t Sender_u16));

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to register a callback function that is called whenever an
  ADC-Dump Start Payload is received.
    @param[in]    Callback function pointer
  *******************************************************************************/
  void RegisterADCDumpStartRequestCallback(void (*Callback)(uint16_t Sender_u16,
                                                            uint32_t ADCDumpSize_32));

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Interface function to register a callback function that is called whenever a
  Logmessage Payload is received.
    @param[in]    Callback function pointer
  *******************************************************************************/
  void RegisterLogMsgCallback(void (*Callback)(uint16_t Sender_u16, uint8_t* ReceivedPayload_pu8));

  // Logging

  /** ******************************************************************************
    @ingroup      Toposens Sensor Library
    @brief        Parses log messages into human readable text.
    @param[out]   *Output_p8
    @param[in]    SenderId_u16
    @param[in]    *ReceivedPayload_pu8
  *******************************************************************************/
  void ParseLogMessageToText(char* Output_p8, uint16_t SenderId_u16,
                             const uint8_t* ReceivedPayload_pu8);

  // Helpers

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        This function is called from blocking Set* Functions and blocks until the
wait_ack_sem is unlocked by the EvaluateACKStatus_b function. The blocking behavior is achieved by
trying to fetch the same semaphore twice. The second sem_wait blocks until sem_post is executed
elsewhere. That happens in EvaluateACKStatus_b.
  @param[in]    *payload The payload is copied to wait_ack_payload_pu8. EvaluateACKStatus_b compares
incoming ACKs whether they match the desired ACK.
  @param[in]    length_u16
  *******************************************************************************/
  void WaitForACK(uint8_t* payload, uint16_t length_u16);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called from Getter functions to extract values from payload. In this case a
bool-value.
  @param[in]    *Payload_pu8
  @return       bool true value was set
                bool false value was not set
  *******************************************************************************/
  bool GetBoolFromPayload_b(const uint8_t* Payload_pu8);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called from Getter functions to extract values from payload. In this case a
uint8-value.
  @param[in]    *Payload_pu8
  @return       uint8_t value extracted from payload
  *******************************************************************************/
  uint8_t GetU8FromPayload_u8(const uint8_t* Payload_pu8);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called from Getter functions to extract values from payload. In this case a
uint8-value.
  @param[in]    *Payload_pu8
  @return       uint8_t value extracted from payload
  *******************************************************************************/
  uint8_t GetU8FromTwoArgumentPayload_u8(const uint8_t* Payload_pu8);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called from Getter functions to extract values from payload. In this case a
uint16-value.
  @param[in]    *Payload_pu8
  @return       uint16_t value extracted from payload
  *******************************************************************************/
  uint16_t GetU16FromPayload_u16(const uint8_t* Payload_pu8);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called from Getter functions to extract values from payload. In this case a
uint16-value.
  @param[in]    *Payload_pu8
  @return       uint16_t value extracted from payload
  *******************************************************************************/
  uint16_t GetU16FromTwoArgumentPayload_u16(const uint8_t* Payload_pu8);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @return       bool
  *******************************************************************************/
  bool RequestWasSuccessful_b();

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called internally to extract the payload of a response.
  @param[in]    *ackstatus
  @return       uint8_t* pointer to the payload
  *******************************************************************************/
  uint8_t* GetResponsePayloadFromACKStatus(ACK_Status_t* ackstatus);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called internally to get the current ACKStatus.
  @param[in]    index
  @return       ACK_Status_t* pointer to the struct
  *******************************************************************************/
  ACK_Status_t* GetCurrentACKStatus(uint8_t index);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called from Getter functions to extract values from payload. In this case a
i32-value.
  @param[in]    *Payload_pu8
  @return       int32_t value extracted from payload
  *******************************************************************************/
  int32_t GetI32FromPayload_i32(const uint8_t* Payload_pu8);

  /** ******************************************************************************
  @ingroup      Toposens Sensor Library
  @brief        Is called from Getter functions to extract values from two argument
payload. In this case a i32-value.
  @param[in]    *Payload_pu8
  @return       int32_t value extracted from payload
  *******************************************************************************/
  int32_t GetI32FromTwoArgumentPayload_i32(const uint8_t* Payload_pu8);

  /*---- public inline functions --------------------------------------------------------*/

#ifdef __cplusplus
}
#endif

#endif  // TOPOSENS_SENSOR_LIB_H
