#include "toposens_pointcloud/logging.h"

namespace toposens_pointcloud
{
  /** Maintains a persistent PointCloud structure holding the cumulative
   *  of all scans received during a run. This cumulative data is saved
   *  periodically as a PCD file in the package folder for
   *  subsequent use with other PCL tools (like pcl_octree_viewer).
   */
  Logging::Logging(ros::NodeHandle nh, ros::NodeHandle private_nh)
  {
    int pcd_save_interval=0;
    private_nh.param<int>("pcd_save_interval", pcd_save_interval, 0);
    private_nh.param<std::string>("target_frame", target_frame, "toposens");
    private_nh.param<std::string>("pcd_path", pcd_path, "");

    /** If no file is specified as a launch argument, the default path
     *  "/home/$USER/.ros/toposens.pcd" is used to store the PCD.
     */
    if(pcd_path == "") pcd_path = std::string(get_current_dir_name()) + "/" + "toposens.pcd";

  	_cloud_sub = nh.subscribe(kPointCloudTopic, 100, &Logging::_accumulate, this);

    timer = nh.createTimer(ros::Duration(pcd_save_interval), &Logging::save, this); // continuous timer

    store = boost::make_shared<XYZINCloud>();
    pcl_conversions::toPCL(ros::Time::now(), store->header.stamp);
    store->header.frame_id = target_frame;
    store->height = 1;
  }

  Logging::~Logging(){
    timer.stop();
  };

  /** Filename is provided as a launch argument. File is saved as a .pcd. */
  void Logging::save(const ros::TimerEvent& event)
  {
    boost::mutex::scoped_lock lock(_store_mutex);

    if (!store->width) {
      ROS_WARN("No pointcloud data to save.");
      return;
    }
    pcl_conversions::toPCL(ros::Time::now(), store->header.stamp);
    try
    {
      if (pcl::io::savePCDFile(pcd_path, *store) == 0) {
        ROS_INFO("Saved latest point cloud data (%s)", pcd_path.c_str());
      }
    }
    catch (pcl::io::IOException &e)
    {
      ROS_ERROR("%s", e.what());
    }
  }

  /** Waits for a pointcloud updates to become available and accumulates them
   *  in a persistent pointcloud structure.
   */
  void Logging::_accumulate(const XYZINCloud::ConstPtr& msg)
  {
    boost::mutex::scoped_lock lock(_store_mutex);

    for (auto it = msg->points.begin(); it != msg->points.end(); ++it)
    {
      pcl::PointXYZINormal pt = *it;
      store->points.push_back(pt);
    }

    store->width = store->points.size();
  }

} // namespace toposens_pointcloud
