"""
Toposens sensor library Python bindings util class.
"""
import os
import datetime
import termios
from dataclasses import dataclass
from typing import Union
import toposens_cffi.lib as lib
from cffi import FFI


# This creates a CFFI object containing the C callback function that needs to
# be alive for as long as the library can call the callback. Thus it needs to
# be declared as a global variable.

log_callback_function_python = None

ffi = FFI()
@ffi.callback("void(unsigned short, unsigned char*)")
def log_callback_function(sender, payload):
    message = ffi.new("char[200]")

    lib.ParseLogMessageToText(message, sender, payload)

    message = ffi.string(message).decode("ascii")

    if log_callback_function_python is None:
        print(message)
    else:
        log_callback_function_python(message)

point_session_end_callback_function_python = None

@ffi.callback("void(unsigned short)")
def point_session_end_callback_function(sender):
    result = lib.GetSessionData(sender)

    if point_session_end_callback_function_python is not None:
        point_session_end_callback_function_python(result)

@dataclass
class Point3D:
    """
    Class representing a 3D point.
    """
    x: int = 0
    y: int = 0
    z: int = 0
    intensity: int = 0
    confidence: int = 0

@dataclass
class Point1D:
    """
    Class representing a 1D point.
    """
    vector_length: int = 0
    intensity: int = 0

@dataclass
class Result:
    """
    Class representing session result object in python.
    """
    points_3d_list = ()
    points_1d_list = ()
    points_3d_number = 0
    points_1d_number = 0

    session_state = 0
    sender_id = 0
    number_points = 0
    noise_level = 0
    nearfield_point = 0
    silentFrame = False
    alienSensor = False
    blind = False

def init_can(interface: str = "can0", baud: int = 1000000) -> Union[int, str]:
    """
    Function initializing the CAN interface passed as an argument.
    Defaults to can0.
    """
    ffi = FFI()
    err = lib.InitInterface(interface.encode("ascii"), baud, lib.IF_CAN)
    err_str = ffi.string(lib.getErrorString(err)).decode("ascii")
    return err, err_str

def deinit_can(interface: str = "can0") -> Union[int, str]:
    """
    Function de-initializing the CAN interface passed as an argument.
    Defaults to can0.
    """
    ffi = FFI()
    err = lib.DeinitInterfaceByName(interface.encode("ascii"), lib.IF_CAN)
    err_str = ffi.string(lib.getErrorString(err)).decode("ascii")
    return err, err_str

def init_uart(interface: str = "/dev/ttyUSB0", baud: int = termios.B115200) -> Union[int, str]:
    """
    Function initializing the UART interface passed as an argument.
    """
    ffi = FFI()
    err = lib.InitInterface(interface.encode("ascii"), baud, lib.IF_UART)
    err_str = ffi.string(lib.getErrorString(err)).decode("ascii")
    return err, err_str

def deinit_uart(interface: str = "/dev/ttyUSB0") -> Union[int, str]:
    """
    Function de-initializing the UART interface passed as an argument.
    """
    ffi = FFI()
    err = lib.DeinitInterfaceByName(interface.encode("ascii"), lib.IF_UART)
    err_str = ffi.string(lib.getErrorString(err)).decode("ascii")
    return err, err_str

def prepare_single_shot(session_id: int = None) -> bool:
    """
    Function setting up the sensor for single shot measurement.
    """
    if session_id is not None:
        lib.SetTargetSensor(session_id)

    lib.RequestReboot()
    lib.WaitForReady()

    lib.SetParameterSystemSensorMode(lib.SENSOR_MODE_SINGLE_SHOT_TRANSMIT_LISTEN)
    # first measurement is empty so we trigger it here to get "real" results on
    # subsequent tries
    return lib.RequestMeasurement()

def measure(session_id: int = None) -> Result:
    """
    Function doing a measurement and returning a Result() with the measured points.
    """
    if session_id is not None:
        lib.SetTargetSensor(session_id)

    lib.RequestMeasurement()

    session_id = lib.WaitForSessionStart()
    session_id = lib.WaitForSessionEnd()

    session_data = lib.GetSessionData(session_id)

    points_3d = []
    for i in range(session_data.NumberOf3DPoints):
        points_3d.append((session_data.Point3D_tp[i].X_i16,
                          session_data.Point3D_tp[i].Y_i16,
                          session_data.Point3D_tp[i].Z_i16,
                          session_data.Point3D_tp[i].Intensity_u8,
                          session_data.Point3D_tp[i].Confidence_u8))

    points_1d = []
    for i in range(session_data.NumberOf1DPoints):
        points_1d.append((session_data.Point1D_tp[i].VectorLength_u16,
                          session_data.Point3D_tp[i].Intensity_u8))

    result = Result()

    result.points_3d_list = tuple(points_3d)
    result.points_3d_number = session_data.NumberOf3DPoints

    result.points_1d_list = tuple(points_1d)
    result.points_1d_number = session_data.NumberOf1DPoints

    result.session_state = session_data.SessionState
    result.sender_id = session_data.SenderId_u16
    result.number_points = session_data.NumberOfPoints_u8
    result.noise_level = session_data.NoiseLevel_u16
    result.nearfield_point = session_data.NearfieldPoint_b

    result.silentFrame = session_data.silentFrame
    result.alienSensor = session_data.alienSensor
    result.blind = session_data.blind

    return result

def setup_logging(loglevel=lib.LOG_LEVEL_DEBUG):
    """
    Function to set the log level and configure a callback function.

    Default log level is LOG_LEVEL_DEBUG. Set it by passing
    toposens.lib.LOG_LEVEL_* as a parameter.
    """

    lib.SetParameterSystemLogLevel(loglevel)

    lib.RegisterLogMsgCallback(log_callback_function)

def setup_point_cloud_session_end_callback(cb=None):
    """
    Function to configure a callback function then the point cloud
    session ended. cb is a python function which gets called with
    the Result() object from the measurement.
    """

    point_session_end_callback_function_python = cb

    lib.RegisterPointSessionEndCallback(point_session_end_callback_function)

def print_sensor_config():
    """
    Function to print sensor config.
    """
    sensor_node_id: int = lib.GetParameterSystemNodeID_u16()
    print(f'Sensor Node ID {sensor_node_id:04d} (0x{sensor_node_id:03x})')

    sensor_mode: int = lib.GetParameterSystemSensorMode_t()
    print(f'Sensor is in mode {sensor_mode}')

    print("Software Version")
    version = lib.RequestVersion_t(lib.VERSION_BYTE_BOOTLOADER)
    print(f" Bootloader Version {version.major}.{version.minor}.{version.hotfix}")

    version = lib.RequestVersion_t(lib.VERSION_BYTE_APP)
    print(f" Application Version {version.major}.{version.minor}.{version.hotfix}")

    version = lib.RequestVersion_t(lib.VERSION_BYTE_HW)
    print(f" Hardware Version {version.major}.{version.minor}.{version.hotfix}")

    version = lib.RequestVersion_t(lib.VERSION_BYTE_COMMS_LIB)
    print(f" Comms Lib Version {version.major}.{version.minor}.{version.hotfix}")

    version = lib.RequestVersion_t(lib.VERSION_BYTE_ESP_LIB)
    print(f" ESP Version {version.major}.{version.minor}.{version.hotfix}")

    print("")
    print(f"Sensor configuration:")
    print(f" Calibration transducer state: {lib.GetParameterSystemCalibTransducerState_b()}")
    print(f"             nearfield state:  {lib.GetParameterSystemCalibNearfieldState_b()}")
    print(f" Transducer number of pulses: {lib.GetParameterTransducerNumOfPulses_u8()}")
    print(f"            volume:           {lib.GetParameterTransducerVolume_u8()}")
    print(f"            frequency:        {lib.GetParameterTransducerFrequency_u16()}")
    print(f" Nearfield enabled: {lib.GetParameterSignalProcessingEnableNearfieldDetection_b()}")
    print(f" AutoGain enabled:  {lib.GetParameterSignalProcessingEnableAutoGain_b()}")



def dump_ADC(filename: str = None, filedir: str = None, createdir: bool = False, silent_frame: bool = False) -> int:
    """
    Function to trigger and save an ADC dump to file.

    Args:
        filename (str): A name will be generated, if none is given.
        filedir (str): The current directory will be used, if none is given.
        createdir (bool): If the given directory should be created.
        
    Return:
        number of bytes witten to the given file

    Notes:
        - In silent frame mode, the gain of region 0 is set/used for the whole dump.
          Use lib.SetParameterADCStepGainGainFactor(0, gain_index) to set the gain.
        - Dump file extension should be '.bin' from ESP v1.5.6 upwards.

    TODO(kruber) Check success and return status.
    TODO(kruber) Add verbose level (on/off)
    TODO(kruber) Use python os path constructs
    """

    if filedir == None:
        filedir = './'

    # Create folder or get out
    if not os.path.exists(filedir):
        if createdir:
            os.mkdir(filedir)
        else:
            print(f'Given folder {filedir} does not exist')
            return 0

    # Create filename
    if filename is None:
        node_id = lib.GetParameterSystemNodeID_u16()
        print(f'Sensor Node ID {node_id:04d}')
        version = lib.RequestVersion_t(lib.VERSION_BYTE_ESP_LIB)
        print(f'ESP Version {version.major}.{version.minor}.{version.hotfix}')
        is_calibrated = lib.GetParameterSystemCalibTransducerState_b()
        print(f'Transducer calibration state is {is_calibrated!s:^5}')

        now = datetime.datetime.now()
        filename = f'node_{node_id:04d}_date_{now.year}.{now.month:02d}.{now.day:02d}' \
                   f'_esp_{version.major:02d}.{version.minor:02d}.{version.hotfix:02d}' \
                   f'_calib_{is_calibrated!s}.bin'

    # Create filepath
    filepath: str = filedir + '/' + filename

    # Request and store ADC dump
    if silent_frame:
        print(f"Request silent ADC dump")
    else:
        print(f"Request ADC dump")
    lib.RequestADCDump()
    if silent_frame:
        lib.SetParameterSystemSensorMode(lib.SENSOR_MODE_SINGLE_SHOT_LISTEN)
    else:
        lib.SetParameterSystemSensorMode(lib.SENSOR_MODE_SINGLE_SHOT_TRANSMIT_LISTEN)
    lib.RequestMeasurement()
    sender = lib.WaitForADCSessionEnd()
    blob = lib.GetADCDumpData(sender)
    lib.GetParameterSystemSensorMode_t()

    print(f"Received {blob.ReceivedDumpSize_u32.__str__()} " \
          f"out of {blob.ExpectedDumpSize_u32.__str__()} bytes.")

    if not blob.ReceivedDumpSize_u32:
        raise EOFError

    ffi = FFI()
    data = ffi.buffer(blob.DumpBlob_pu8, blob.ReceivedDumpSize_u32)

    with open(filepath, "wb") as binary_file:
        num_bytes_written = binary_file.write(data)
        print(f"Written {num_bytes_written} bytes.")

    return num_bytes_written


def set_single_gain(gain_index:int = 1) -> bool:
    """
    Set sensor to have one single gain region right after the primary pulse.

    Args:
        gain_index (int): Index pointing to an element in the gain factor array [0..9].
    Return:
        bool for success

    The available gain factors are: {0.25F, 1.0F, 10.0F, 20.0F, 30.0F, 40.0F, 60.0F, 80.0F, 119.0F, 157.0F}

    The end of the first gain region [0] is set right after the primary pulse at 25*50us. The gain value
    of the primary pulse region remains untouched.
    """
    if gain_index < 0 or gain_index > 9:
        print(f"given gain index {gain_index} is not within [0..9].")
        return False

    if lib.SetParameterADCStepGainNumberOfSteps(2) == False:
        return False

    if lib.SetParameterADCStepGainTimeSlot(1, 25) == False:
        return False

    if lib.SetParameterADCStepGainGainFactor(1, gain_index) == False:
        return False

    return True


def canhigh(interface="can0"):
    """
    Function to set the given can interface to 1000000 baud. Default interface is "can0".
    """
    can(interface, 1000000)

def canlow(interface="can0"):
    """
    Function to set the given can interface to 125000 baud. Default interface is "can0".
    """
    can(interface, 125000)

def can(interface="can0", baud=1000000):
    """
    Function to set the given can interface to the given baud rate. Default values are "can0"
    and 1000000.
    """
    os.system("sudo ip link set " + interface + " down")
    os.system("sudo ip link set " + interface + " up type can bitrate " + str(baud))

def filter_3d_points_x_closer(point_list, distance=5) -> tuple:
    """
    Function returning the points as a list that are closer on the X axis than a threshold.
    """
    return tuple(filter(lambda x: x[0] < distance , point_list))

def filter_3d_points_intensity_bigger(point_list, intensity=5) -> tuple:
    """
    Function returning the points with intensity bigger than a theshhold.
    """
    return tuple(filter(lambda x: x[3] > intensity , point_list))

def filter_3d_points_get_coordinates(point_list) -> tuple:
    """
    Function returning the coordinates as a list for the 3D point list argument.
    """
    return tuple([(e[0],e[1], e[2]) for e in point_list])

def filter_3d_points_get_intensity(point_list) -> tuple:
    """
    Function returning the intensities as a list for the 3D point list argument.
    """
    return tuple([e[3] for e in point_list])

def filter_1d_points_get_vector_length(point_list) -> tuple:
    """
    Function returning the vector lengths as a list for the 1D point list argument.
    """
    return tuple([e[0]for e in point_list])

def filter_1d_points_get_intensity(point_list) -> tuple:
    """
    Function returning the intensities as a list for the 1D point list argument.
    """
    return tuple([e[1] for e in point_list])
